'use strict';

include("scripts/CheckFTU.js");
include("scripts/CheckUnderflow.js");
include("scripts/CheckStates.js");
include("scripts/Handler_Function_class");
// To de done:
//  - CheckLID status (should be open or closed)
//  - Is it necessary to switch the bias-voltage off?
//  - Get reasonable timeouts for all steps (wait, get, run)
//  - Improve order to accelerate execution
//
// =================================================================

/**
 *var table =
 *[
 * [ "AGILENT_CONTROL" ],
 * [ "BIAS_CONTROL"    ],
 * [ "CHAT"            ],
 * [ "DATA_LOGGER"     ],
 * [ "DRIVE_CONTROL"   ],
 * [ "FEEDBACK"        ],
 * [ "FAD_CONTROL"     ],
 * [ "FSC_CONTROL"     ],
 * [ "FTM_CONTROL"     ],
 * [ "LID_CONTROL"     ],
 * [ "MAGIC_WEATHER"   ],
 * [ "MCP"             ],
 * [ "PWR_CONTROL"     ],
 * [ "RATE_CONTROL"    ],
 * [ "RATE_SCAN"       ],
 * [ "SMART_FACT"      ],
 * [ "TIME_CHECK"      ],
 * [ "TNG_WEATHER"     ],
 *];
 *
 *if (dim.state("DRIVE_CONTROL").name=="Locked")
 *{
 *    throw new Error("Drivectrl still locked... needs UNLOCK first.");
 *    //while (!dim.send("DRIVE_CONTROL"))
 *    //    v8.sleep();
 *    //dim.send("DRIVE_CONTROL/UNLOCK");
 *    //dim.wait("DRIVE_CONTROL", "Armed", 1000);
 *}
 *
 */
var switch_camera_on = function(){
    console.out("");
    dim.alarm(); // I believe this *resets* any prior alarm.

    CheckStates.checkSend(["PWR_CONTROL"], undefined, true);

    var camera_on_handler = new Handler_Function_class.Handler("PowerOn");
    camera_on_handler.add(Handler_Function_class.power_camera_on);
    camera_on_handler.run();
    console.out("");
    // If power was switched on: wait for a few seconds
};

var set_up_systems = function(){
    // -----------------------------------------------------------------
    // Now take care that the bias control, the ftm and the fsc are
    // properly connected and are in a reasonable state (e.g. the
    // trigger is switched off)
    // -----------------------------------------------------------------


    CheckStates.checkSend(
        ["BIAS_CONTROL",
         "FAD_CONTROL",
         "FTM_CONTROL", 
         "FSC_CONTROL", 
         "FEEDBACK", 
         "RATE_CONTROL", 
         "MCP"],
        undefined,
        true);

    dim.send("MCP/RESET");

    var set_up_handler = new Handler_Function_class.Handler("SystemSetup");
    set_up_handler.add(Handler_Function_class.bias_voltage_off);
    set_up_handler.add(Handler_Function_class.make_ftm_idle);
    set_up_handler.add(Handler_Function_class.connect_fsc);
    set_up_handler.add(Handler_Function_class.connect_fad);
    set_up_handler.add(Handler_Function_class.connect_feedback); // Feedback needs FAD to be Connected
    set_up_handler.add(Handler_Function_class.connect_ratecontrol);
    set_up_handler.add(Handler_Function_class.close_lid);
    set_up_handler.run();

    console.out("biasctrl:    "+dim.state("BIAS_CONTROL").name);
    console.out("ftmctrl:     "+dim.state("FTM_CONTROL").name);
    console.out("fscctrl:     "+dim.state("FSC_CONTROL").name);
    console.out("feedback:    "+dim.state("FEEDBACK").name);
    console.out("ratecontrol: "+dim.state("RATE_CONTROL").name);
    console.out("fadctrl:     "+dim.state("FAD_CONTROL").name);
    console.out("mcp:         "+dim.state("MCP").name);
    console.out("");

    console.out("Enable all FTU");
    dim.send("FTM_CONTROL/ENABLE_FTU", -1, true);
};

var check_clock_conditioner = function(){
    var sub_counter = new Subscription("FTM_CONTROL/COUNTER");
    var counter = sub_counter.get(3000, false).counter;
    dim.send("FTM_CONTROL/REQUEST_STATIC_DATA");
    v8.timeout(3000, 
        function() { 
            if (sub_counter.get(0, false).counter>counter) 
                return true; 
        });
    if (sub_counter.get(0, false).qos & 0x100 == 0)
        throw new Error("Clock conditioner not locked.");
    sub_counter.close();
};

var power_on_drive = function(){
    if ((dim.state("PWR_CONTROL").index & 16) == 0)
    {
        console.out("Drive cabinet not powered... Switching on.");
        dim.send("PWR_CONTROL/TOGGLE_DRIVE");
        v8.timeout(5000, 
            function() { 
                if (dim.state("PWR_CONTROL").index&16) 
                    return true; 
            });
    }

    checkSend(["DRIVE_CONTROL"]);

    var arm_drive_handler = new Handler("ArmDrive");
    arm_drive_handler.add(Handler_Function_class.arm_drive);
    arm_drive_handler.run();
};

var makeCurrentCalibration = function(){
    // At this point we know that:
    //  1) The lid is closed
    //  2) The feedback is stopped
    //  3) The voltage is off


    dim.send("BIAS_CONTROL/SET_ZERO_VOLTAGE");
    dim.wait("BIAS_CONTROL", "VoltageOff", 30000); // waS: 15000

    var now = new Date();
    dim.send("FEEDBACK/CALIBRATE");

    console.out("Wait for calibration to start");
    dim.wait("FEEDBACK", "Calibrating", 5000);

    console.out("Wait for calibration to end");
    dim.wait("FEEDBACK", "Calibrated", 90000);

    console.out("Calibration finished ["+(new Date()-now)+"ms]");

    console.out("Wait for voltage to be off");
    dim.wait("BIAS_CONTROL", "VoltageOff", 30000); // was: 15000
};

var perform_current_calibration_if_needed = function(){
    // ================================================================
    // Bias crate calibration
    // ================================================================
    // Bias crate calibration if necessary (it is aftr 4pm (local tome)
    // and the last calibration was more than eight hours ago.
    // -----------------------------------------------------------------

    // Check age of calibration
    var service_calibration = new Subscription("FEEDBACK/CALIBRATION");

    var data_calibration = service_calibration.get(3000, false);

    var age = data_calibration.time;
    var now = new Date();

    var diff = (now-age)/3600000;

    var fb_state = dim.state("FEEDBACK").index;

    // !data_calibration.data: FEEDBACK might just be freshly
    // started and will not yet serve this service.
    if (fb_state<5 || (diff>8 && now.getHours()>16))
    {
        if (fb_state<5)
            console.out("No BIAS crate calibration available: New calibration needed.");
        else
            console.out("Last BIAS crate calibration taken at "+age.toUTCString()+": New calibration needed.");

        makeCurrentCalibration();
    }

    service_calibration.close();
};

var set_up_gps_control = function(){
    // ================================================================
    // Setup GPS control and wait for the satellites to be locked
    // ================================================================

    checkSend(["GPS_CONTROL"]);

    if (dim.state("GPS_CONTROL").name=="Disconnected")
        dim.send("GPS_CONTROL/RECONNECT");

    // Wait for being connectes
    v8.timeout(5000, function() { if (dim.state("GPS_CONTROL").name!="Disconnected") return true; });

    // Wait for status available
    v8.timeout(5000, function() { if (dim.state("GPS_CONTROL").name!="Connected") return true; });

    if (dim.state("GPS_CONTROL").name=="Disabled")
        dim.send("GPS_CONTROL/ENABLE");

    // Wait for gps to be enabled and locked
    dim.wait("GPS_CONTROL", "Locked", 15000);
};

var crosscheck_all_states = function(){
    // ================================================================
    // Crosscheck all states
    // ================================================================

    // FIXME: Check if there is a startup scheduled, if not do not force
    // drive to be switched on
    var table =
    [
     [ "TNG_WEATHER"   ],
     [ "MAGIC_WEATHER" ],
     [ "CHAT"          ],
     [ "SMART_FACT"    ],
     [ "TEMPERATURE"   ],
     [ "EVENT_SERVER",        [ "Running", "Standby" ] ],
     [ "DATA_LOGGER",         [ "NightlyFileOpen", "WaitForRun", "Logging" ] ],
     [ "FSC_CONTROL",         [ "Connected"                       ] ],
     [ "MCP",                 [ "Idle"                            ] ],
     [ "TIME_CHECK",          [ "Valid"                           ] ],
     [ "PWR_CONTROL",         [ "SystemOn"                        ] ],
     [ "AGILENT_CONTROL_24V", [ "VoltageOn"                       ] ],
     [ "AGILENT_CONTROL_50V", [ "VoltageOn"                       ] ],
     [ "AGILENT_CONTROL_80V", [ "VoltageOn"                       ] ],
     [ "BIAS_CONTROL",        [ "VoltageOff"                      ] ],
     [ "FEEDBACK",            [ "Calibrated"                      ] ],
     [ "RATE_SCAN",           [ "Connected"                       ] ],
     [ "RATE_CONTROL",        [ "Connected"                       ] ],
     [ "DRIVE_CONTROL",       [ "Armed", "Tracking", "OnTrack", "Locked" ] ],
     [ "LID_CONTROL",         [ "Open", "Closed"                  ] ],
     [ "FTM_CONTROL",         [ "Valid", "TriggerOn"              ] ],
     [ "FAD_CONTROL",         [ "Connected", "WritingData"        ] ],
     [ "GPS_CONTROL",         [ "Locked" ] ],
     [ "SQM_CONTROL",         [ "Valid" ] ],
     [ "PFMINI_CONTROL",      [ "Receiving" ] ],
    ];

    if (!checkStates(table))
    {
        throw new Error("Something unexpected has happened. Although the startup-"+
                        "procedure has finished, not all servers are in the state "+
                        "in which they ought to be. Please, try to find out what "+
                        "happened...");
    }
};

var perform_the_entire_startup = function(){
    switch_camera_on();
    set_up_systems();
    CheckFTU.checkFTU();
    check_clock_conditioner();
    CheckUnderflow.checkUnderflow();
    power_on_drive();
    perform_current_calibration_if_needed();
    set_up_gps_control();
    crosscheck_all_states();
};

perform_the_entire_startup();