'use strict';
if (!("Handler_Function_class" in this)){
    var Handler_Function_class = {
        Handler : function(name){
            this.name  = name;
            this.handler_function_list = [];
        },
        Handler_Function: function(server_name){
            this.name = server_name;
            this.definition = {
                final_states: new Set([]),
                just_wait_states: new Set([]),
                reset_wait_states: new Set([]),
                action_states: {}, 
            };
            this.wait_state = undefined;
            this.done = false;

            this.current_state = undefined;
            this.last_state = undefined;
        },
        send_command_expect_state: function(command, expectation){
            return function(){
                console.out(this.name+'in '+this.current_state.name
                    +"..."+command+".");    
                dim.send(command);
                this.wait_state = expectation;
            }
        },
        throw_string_exception: function(message){
            return function(){
                throw message;
            }
        },
        throw_error: function(){
            throw new Error(this.name+':'
                +state.name+"["+state.index+"] in error state.");
        },
    };
}
else{
    console.out("multiple include of 'Handler_Function_class.js'");
}


/* 
    add a handler_function 
    to the internal handler_function_list.

    A handler_function tries every time it is invoked,
    to get a certain subsystem into a certain *fixed* state.
    When it succeeded, it return an empty string "".
    if it not (yet) acomplished its goal it will return:
        * The name of the state it waits for.
        * undefined, when it does not know, what state to wait for.

    It is important to feed it back, the wait_state_name it returned
    last time, so it knows, it should not do any action, while waiting
    for a certain state.
*/
Handler_Function_class.Handler.prototype.add = function(func)
{
    this.handler_function_list.push(func);
}

/* run each handler_function in the handler_function_list
   until either all of them return the empty string ""
   or the timeout occured.

*/
Handler_Function_class.Handler.prototype.run = function(timeout)
{
    console.out(this.name+":start");

    var rc = [];

    var start_date = new Date();
    while (!timeout || (new Date()-start_date)<timeout)
    {
        var done = true;
        for (var i=0; i<this.handler_function_list.length; i++)
        {
            var current_handler_function = this.handler_function_list[i];
            if (!current_handler_function.call())
                done = false;
        }

        if (done)
        {
            console.out(this.name+":success [time="+(new Date()-start_date)+"ms]");
            return true;
        }

        v8.sleep();
    }

    console.out(this.name+":timeout ["+timeout+"ms]");
    return false;
}


// ------------------------------------------------------------------------------

Handler_Function_class.Handler_Function.prototype.call = function(){
    this.last_state = state;
    var state = dim.state(this.server_name);
    this.current_state = state;

    if (state === undefined){
        // if the current state is undefined, we cannot be done!
        this.wait_state=undefined;
        this.done = false;
        return false;
    }
    else if(this.done){
        return true;
    }
    else if (this.wait_state && state.name!=this.wait_state){
        // if we *have* a state to actually wait for, we just wait
        return false;
    }else if(state.name in this.definition.final_states){
        this.we_are_done();
    }else if(state.name in this.definition.just_wait_states){
        this.wait();
    }else if(state.name in this.definition.reset_wait_states){
        this.set_wait_state_undefined();
    }else if(state.name in this.definition.action_states){
        this.definition.action_states[state.name]();
    }
    else
        throw new Error(this.name+":"+state.name+"["+state.index+"] unknown or not handled.");
}

Handler_Function_class.Handler_Function.prototype.set_wait_state_undefined = function(){
    console.log("set_wait_state_undefined: this:");
    print_object(this);
    this.wait_state = undefined;
}

Handler_Function_class.Handler_Function.prototype.wait = function(){
    console.log("wait: this:"+this);
    // we do nothing here, 
    // esp. we do not alter the wait_state
}

Handler_Function_class.Handler_Function.prototype.we_are_done = function(){
    this.done = true;
}

//--------- bias_voltage_off

Handler_Function_class.bias_voltage_off = new Handler_Function_class.Handler_Function("BIAS_CONTROL");
Handler_Function_class.bias_voltage_off.definition = {
    final_states: new Set(["VoltageOff"]),
    just_wait_states: new Set(["Connecting", "Initializing", "Connected", "Ramping"]),
    reset_wait_states: new Set([]),
    action_states: {
        "Locked": function(){
            console.out("WARNING - Bias is LOCKED. Please report, "
                +"this is serious, unlock manually and go on.");
            this.done=true;
            },
        "Disconnected": Handler_Function_class.send_command_expect_state(
            "BIAS_CONTROL/RECONNECT", "VoltageOff"),
        "NotReferenced": Handler_Function_class.send_command_expect_state(
            "BIAS_CONTROL/SET_ZERO_VOLTAGE", "VoltageOff"),
        "VoltageOn": Handler_Function_class.send_command_expect_state(
            "BIAS_CONTROL/SET_ZERO_VOLTAGE", "VoltageOff"),
        "OverCurrent" : Handler_Function_class.throw_string_exception("BIAS_CONTROL in OverCurrent"),
        "ExpertMode": Handler_Function_class.throw_string_exception("BIAS_CONTROL in expert mode"),
    },
};
//--------- power_camera_on
Handler_Function_class.power_camera_on = new Handler_Function("PWR_CONTROL");
Handler_Function_class.set_camera_power_on = function (wait_for_state){
    return function(){
        this.wait_state=wait_for_state;
        console.out(this.name+" in "+this.current_state.name
            +"... sending CAMERA_POWER ON... waiting for "+this.wait_state+".");
    };
};
Handler_Function_class.power_camera_on.definition ={
    final_states: new Set(["DriveOff", "SystemOn"]),
    just_wait_states: new Set(["CameraOn", "BiasOn", "CameraOff", "BiasOff"]),
    reset_wait_states: new Set(["Disconnected", "Connected", "NoConnection"]),
    action_states: { 
        "PowerOff" : Handler_Function_class.set_camera_power_on("DriveOff"),
        "DriveOn": Handler_Function_class.set_camera_power_on("SystemOn"),
        "CoolingFailure": function (){
            throw new Error("Cooling unit reports failure... please check.");
        },
    },
};
//--------- make_ftm_idle
Handler_Function_class.make_ftm_idle = new Handler_Function_class.Handler_Function("FTM_CONTROL");
Handler_Function_class.make_ftm_idle.definitions = {
    final_states = new Set(["Valid"]),
    action_states = {
        "Disconnected": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/RECONNECT", "Valid"),
        "Idle": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/DISCONNECT", "Disconnected"),
        "TriggerOn": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/STOP_TRIGGER", "Valid"),
        "Configuring1": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/RESET_CONFIGURE", "Valid"),
        "Configuring2": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/RESET_CONFIGURE", "Valid"),
        "Configured1": Handler_Function_class.send_command_expect_state(
            "FTM_CONTROL/RESET_CONFIGURE", "Valid"),
        "Configured2": function(){
            this.wait_state = "TriggerOn";
        },
        "ConfigError1": Handler_Function_class.throw_error,
        "ConfigError2": Handler_Function_class.throw_error,
        "ConfigError3": Handler_Function_class.throw_error,
    }
};
//--------- connect_fsc
Handler_Function_class.connect_fsc = new Handler_Function_class.Handler_Function("FSC_CONTROL");
Handler_Function_class.connect_fsc.definition = {
    final_states : new Set(["Connected"]),
    action_states : {
        "Disconnected" : Handler_Function_class.send_command_expect_state(
            "FSC_CONTROL/RECONNECT", "Connected"),
    },
};
//--------- connect_feedback
var send_stop_expect_calibrated = Handler_Function_class.send_command_expect_state("FEEDBACK/STOP", "Calibrated");

Handler_Function_class.connect_feedback = new Handler_Function_class.Handler_Function("FEEDBACK");
Handler_Function_class.connect_feedback.definition = {
    final_states: new Set(["Connected", "Calibrated"]),
    reset_wait_states: new Set(["Disconnected", "Connecting"]),
    action_states: {
        "Calibrating" : Handler_Function_class.send_command_expect_state(
            "FEEDBACK/STOP", "Connected"),
        "WaitingForData": send_stop_expect_calibrated,
        "OnStandby": send_stop_expect_calibrated,
        "InProgress": send_stop_expect_calibrated,
        "Warning": send_stop_expect_calibrated,
        "Critical": send_stop_expect_calibrated,
    },
};
//--------- connect_ratecontrol
var send_stop_wait_for_connected = send_command_expect_state("RATE_CONTROL/STOP", "Connected");

Handler_Function_class.connect_ratecontrol = new Handler_Function_class.Handler_Function("RATE_CONTROL");
Handler_Function_class.connect_ratecontrol.definition = {
    final_states: new Set(["Connected"]),
    reset_wait_states: new Set(["DimNetworkNotAvailable", "Disconnected"]),
    action_states: {
        "Calibrating": send_stop_wait_for_connected,
        "GlobalThresholdSet": send_stop_wait_for_connected,
        "InProgress": send_stop_wait_for_connected,
    }
};
//--------- close_lid
var send_close_wait_for_closed = Handler_Function_class.send_command_expect_state(
    "LID_CONTROL/CLOSE", "Closed");

Handler_Function_class.close_lid = new Handler_Function_class.Handler_Function("LID_CONTROL");
Handler_Function_class.close_lid.definition = {
    final_states: new Set(["Closed"]),
    just_wait_states: new Set(["NotReady", "Ready", 
        "NoConnection", "Connected", "Moving"]),
    action_states: {
        "Unknown": send_close_wait_for_closed,
        "Inconsistent": send_close_wait_for_closed,
        "PowerProblem": send_close_wait_for_closed,
        "Open": send_close_wait_for_closed,
    }
};
//--------- connect_fad
var send_reset_configure = Handler_Function_class.send_command_expect_state(
    "FAD_CONTROL/RESET_CONFIGURE", "Connected");
var send_start = Handler_Function_class.send_command_expect_state(
    "FAD_CONTROL/START", "Connected");
var send_close_files = Handler_Function_class.send_command_expect_state(
    "FAD_CONTROL/CLOSE_OPEN_FILES", "Connected");

var check_final_state = function(){
    var sub_con = new Subscription("FAD_CONTROL/CONNECTIONS");
    var con = sub_con.get(5000);
    var all = true;
    for (var i=0; i<40; i++)
        if (con.obj['status'][i]&66!=66)
        {
            console.out("Board "+i+" not connected... sending CONNECT... waiting for 'Connected'.");
            dim.send("FAD_CONTROL/CONNECT", i);
            all = false;
        }
    sub_con.close();
    if (all)
        this.done = true;
    else
        this.wait_state = "Connected";
}

Handler_Function_class.connect_fad = new Handler_Function_class.Handler_Function("FAD_CONTROL");
Handler_Function_class.connect_fad.definitions = {
    reset_wait_states: new Set(["Offline"]),
    just_wait_states: new Set(["Connecting"]),
    action_states: {
        "Disconnected": Handler_Function_class.send_command_expect_state(
            "FAD_CONTROL/START", "Connected"),
        "Configuring1": send_reset_configure,
        "Configuring2": send_reset_configure,
        "Configuring3": send_reset_configure,
        "Configured": send_reset_configure,
        "Disengaged": send_start,
        "RunInProgress": send_close_files,
        "Connected": check_final_state,
    },
};
//--------- arm_drive
var send_stop = Handler_Function_class.send_command_expect_state("DRIVE_CONTROL", "Armed");

Handler_Function_class.arm_drive = new Handler_Function_class.Handler_Function("DRIVE_CONTROL")
Handler_Function_class.arm_drive.definitions = {
    final_states: new Set(["Armed"]),
    reset_wait_states: new Set(["Disconnected", "Connected", "Ready"]),
    action_states: {
        "Moving" : send_stop,
        "Tracking": send_stop,
        "OnTrack": send_stop,
        "NotReady": function(){
            send_stop(),
            dim.wait("DRIVE_CONTROL", "Armed", 60000);
            this.wait_state = undefined;
        },
        "ERROR": function(){
            send_stop(),
            dim.wait("DRIVE_CONTROL", "Armed", 60000);
            this.wait_state = undefined;
        },
        "Locked": function(){
            console.out("WARNING - Drive is LOCKED. Please unlock manually.");
            this.done = true;
        },
    },
};
