int callisto(const char *seqfile="seq/2012/01/23/20120123_023.seq", const char *outpath = "output", bool use_delays=true)
{
    // ======================================================

    // true:  Display correctly mapped pixels in the camera displays
    //        but the value-vs-index plot is in software/spiral indices
    // false: Display pixels in hardware/linear indices,
    //        but the order is the camera display is distorted
    bool usemap = true;

    // map file to use (get that from La Palma!)
    const char *map = usemap ? "/home/fact/FACT++/FACTmap111030.txt" : NULL;

    Bool_t maximum = kTRUE;

    const char *lp_template    = maximum ?
        "template-lp-extractor-maximum.root" :
        "template-lp-extractor-leading-edge.root";

    const char *pulse_template = "template-pulse.root";

    // ------------------------------------------------------

    // Calib: 51 / 90 / 197 (20% TH)
    // Data:  52 / 64 / 104 (20% TH)

    // Extraction range in slices. It will always(!) contain the full range
    // of integration
    const int first_slice =  20; //  10ns
    const int last_slice  = 250; // 125ns

    // Note that rise and fall time mean different things whether you use IntegralFixed or IntegraRel:
    //
    //  IntegralFixed:
    //    * fRiseTime: Number of slices left  from arrival time
    //    * fFallTime: Number of slices right from arrival time
    //  IntegralRel:
    //    * fRiseTime: Number of slices left  from maximum time
    //    * fFallTime: Number of slices right from maximum time
    //
    const int rise_time_cal = maximum ?  40 :  10; // was 13;   5ns
    const int fall_time_cal = maximum ? 120 : 160; // was 23;  80ns

    const int rise_time_dat = maximum ?  10 :   2; // was 13; was 10;   1ns
    const int fall_time_dat = maximum ?  40 :  48; // was 23; was 40;  24ns

    // Extraction type: Extract integral and half leading edge

    const int type = maximum ? (MExtralgoSpline::kIntegralRel) : (MExtralgoSpline::kIntegralFixed);
    //const int type = MExtralgoSpline::kIntegralFixed;


    const double heighttm   = 0.5; // IntegralAbs { 1.5pe * 9.6mV/pe } / IntegralRel { 0.5 }

    Long_t max  =    0;  // All
    Long_t max0 =  max;  // Time marker
    Long_t max1 =  max;  // Light pulser
    //Long_t max2 = 3000;  // Calibration ratio
    Long_t max3 =  max;  // Pedestal Rndm
    Long_t max4 =  max;  // Pedestal Ext
    Long_t max5 =  max;  // Data

    // ======================================================

    if (map && gSystem->AccessPathName(map, kFileExists))
    {
        gLog << "ERROR - Cannot access mapping file '" << map << "'" << endl;
        return 1;
    }

    // The sequence file which defines the files for the analysis
    MSequence seq(seqfile);
    if (!seq.IsValid())
    {
        gLog << "ERROR - Sequence '" << seqfile << "' invalid!" << endl;
        return 2;
    }

    // --------------------------------------------------------------------------------

    gLog.Separator("Callisto");
    gLog << "Calibrate data of sequence '" << seq.GetFileName() << "'" << endl;
    gLog << endl;

    // ------------------------------------------------------

    ostringstream drsname;
    drsname << gSystem->DirName(seqfile) << "/";
    drsname << seq.GetNight().GetNightAsInt() << "_";
    drsname << Form("%03d", seq.GetDrsSequence()) << ".drs.seq";

    MSequence drs(drsname.str().c_str());
    if (!drs.IsValid())
    {
        gLog << "ERROR - DRS sequence invalid!" << endl;
        return 3;
    }

    gLog << "DRS sequence file: " << drsname.str() << '\n' << endl;

    TString drsfile = seq.GetFileName(0, MSequence::kRawDrs);
    if (drsfile.IsNull())
    {
        cout << "No DRS file available in sequence." << endl;
        return 4;
    }

    TString timfile = drs.GetFileName(0, MSequence::kFitsDat);
    TString drs1024 = drs.GetFileName(0, MSequence::kFitsDrs);
    TString pedfile = seq.GetFileName(0, MSequence::kFitsPed);
    TString calfile = seq.GetFileName(0, MSequence::kFitsCal);

    gLog << "DRS calib     300: " << drsfile << '\n';
    gLog << "DRS calib    1024: " << drs1024 << "\n\n";

    MDrsCalibration drscalib300;
    if (!drscalib300.ReadFits(drsfile.Data()))
        return 5;

    MDrsCalibration drscalib1024;
    if (!drscalib1024.ReadFits(drs1024.Data()))
        return 6;

    gLog << "Time calibration : " << timfile << '\n';
    gLog << "Pedestal     file: " << pedfile << '\n';
    gLog << "Light Pulser file: " << calfile << '\n' << endl;

    // ------------------------------------------------------

    MDirIter iter;
    if (seq.GetRuns(iter, MSequence::kFitsDat)<=0)
    {
        gLog << "ERROR - Sequence valid but without files." << endl;
        return 7;
    }
    iter.Print();

    // ======================================================

    MStatusArray arrt, arrp;

    TFile ft(lp_template);
    if (arrt.Read()<=0)
    {
        cout << "ERROR - Reading LP template from " << lp_template << endl;
        return 100;
    }

    MHCamera *lpref = (MHCamera*)arrt.FindObjectInCanvas("ExtCalSig;avg", "MHCamera", "Cam");
    if (!lpref)
    {
        cout << "ERROR - LP Template not found in " << lp_template << endl;
        return 101;
    }
    lpref->SetDirectory(0);

    MHCamera *gain = (MHCamera*)arrt.FindObjectInCanvas("gain", "MHCamera", "Gain");
    if (!gain)
    {
        cout << "ERROR - Gain not found in " << lp_template << endl;
        return 101;
    }
    gain->SetDirectory(0);

    TFile fp(pulse_template);
    if (arrp.Read()<=0)
    {
        cout << "ERROR - Reading Pulse template from " << pulse_template << endl;
        return 102;
    }

    TH1F *hpulse = (TH1F*)arrp.FindObjectInCanvas("hPixelEdgeMean0_0", "TH1F", "cgpPixelPulses0");
    if (!hpulse)
    {
        cout << "ERROR - Pulse Template not found in " << pulse_template << endl;
        return 103;
    }
    hpulse->SetDirectory(0);

    // ======================================================

    MStatusDisplay *d = new MStatusDisplay;

    MBadPixelsCam badpixels;
    badpixels.InitSize(1440);
    badpixels[ 424].SetUnsuitable(MBadPixelsPix::kUnsuitable);
    badpixels[ 583].SetUnsuitable(MBadPixelsPix::kUnsuitable);
    badpixels[ 830].SetUnsuitable(MBadPixelsPix::kUnsuitable);
    badpixels[ 923].SetUnsuitable(MBadPixelsPix::kUnsuitable);
    badpixels[1208].SetUnsuitable(MBadPixelsPix::kUnsuitable);
    badpixels[1399].SetUnsuitable(MBadPixelsPix::kUnsuitable);

    //  Twin pixel
    //     113
    //     115
    //     354
    //     423
    //    1195
    //    1393

    MDrsCalibrationTime timecam;

    // Plot the trigger pattern rates vs. run-number
    MH3 hrate("MRawRunHeader.GetFileID", "MRawEvtHeader.GetTriggerID&0xff00");
    hrate.SetWeight("1./TMath::Max(MRawRunHeader.GetRunLength,1)");
    hrate.SetName("Rate");
    hrate.SetTitle("Event rate [Hz];File Id;Trigger Type;");
    hrate.InitLabels(MH3::kLabelsXY);
    hrate.DefineLabelY(    0, "Data"); // What if TriggerID==0 already???
    hrate.DefineLabelY(0x100, "Cal");
    hrate.DefineLabelY(0x400, "Ped");
    // hrate.DefaultLabelY("ERROR");

    Bool_t isinteg =
        (type&MExtralgoSpline::kIntegral)    ||
        (type&MExtralgoSpline::kFixedWidth)  ||
        (type&MExtralgoSpline::kDynWidth)
        ? kTRUE : kFALSE;

    gStyle->SetOptFit(kTRUE);

    // ======================================================

    gLog << endl;
    gLog.Separator("Processing DRS timing calibration run");

    MTaskList tlist0;

    MParList plist0;
    plist0.AddToList(&tlist0);
    plist0.AddToList(&drscalib1024);
    plist0.AddToList(&badpixels);
    plist0.AddToList(&timecam);

    MEvtLoop loop0("DetermineTimeCal");
    loop0.SetDisplay(d);
    loop0.SetParList(&plist0);

    // ------------------ Setup the tasks ---------------

    MRawFitsRead read0(timfile);

    MContinue cont0("MRawEvtHeader.GetTriggerID!=33792", "SelectTim");

    MGeomApply apply0;

    MDrsCalibApply drsapply0;

    MFillH fill0("MHDrsCalibrationTime");
    fill0.SetNameTab("DeltaT");

    tlist0.AddToList(&read0);
    tlist0.AddToList(&apply0);
    tlist0.AddToList(&drsapply0);
    tlist0.AddToList(&cont0);
    tlist0.AddToList(&fill0);

    if (!loop0.Eventloop(max0))
        return 8;

    if (!loop0.GetDisplay())
        return 9;

    /*
     MHDrsCalibrationT *t = (MHDrsCalibrationT*)plist4.FindObject("MHDrsCalibrationT");
     t->SetDisplay(d);
     t->PlotAll();
     */

    // ======================================================

    gLog << endl;
    gLog.Separator("Processing external light pulser run");

    MTaskList tlist1;

    MParList plist1;
    plist1.AddToList(&tlist1);
    plist1.AddToList(&drscalib300);
    plist1.AddToList(&badpixels);
    plist1.AddToList(&timecam);

    MEvtLoop loop1("DetermineCalConst");
    loop1.SetDisplay(d);
    loop1.SetParList(&plist1);

    // ------------------ Setup the tasks ---------------

    MRawFitsRead read1;
    read1.LoadMap(map);
    read1.AddFile(calfile);

    MContinue cont1("(MRawEvtHeader.GetTriggerID&0xff00)!=0x100", "SelectCal");

    MGeomApply apply1;

    MDrsCalibApply drsapply1;

    /*
    MPedestalCam  fPedestalCamOut1a;
    MPedestalCam  fPedestalCamOut1b;
    MPedestalCam  fPedestalCamOut1c;

    MPedCalcPedRun pedcalc1a;
    MPedCalcPedRun pedcalc1b;
    MPedCalcPedRun pedcalc1c;
    pedcalc1a.SetPedestalsOut(&fPedestalCamOut1a);
    pedcalc1b.SetPedestalsOut(&fPedestalCamOut1b);
    pedcalc1c.SetPedestalsOut(&fPedestalCamOut1c);

    MExtractTimeAndChargeSpline extractor1ab;
    extractor1a.SetRange(first_slice, last_slice);
    extractor1a.SetRiseTimeHiGain(rise_time);
    extractor1a.SetFallTimeHiGain(fall_time);
    extractor1a.SetChargeType(type);
    extractor1a.SetSaturationLimit(600000);
    extractor1a.SetNoiseCalculation(kFALSE);

    pedcalc1a.SetRandomCalculation(kTRUE);
    pedcalc1b.SetRandomCalculation(kFALSE);
    pedcalc1a.SetExtractor(&extractor1a);
    pedcalc1b.SetExtractor($extractor1a);
    pedcalc1c.SetRangeFromExtractor(&extractor1a);
    */

    // ---

    MExtractTimeAndChargeSpline extractor1b("ExtractPulse");
    extractor1b.SetRange(first_slice, last_slice);
    extractor1b.SetRiseTimeHiGain(rise_time_cal);
    extractor1b.SetFallTimeHiGain(fall_time_cal);
    extractor1b.SetHeightTm(heighttm);
    extractor1b.SetChargeType(type);
    extractor1b.SetSaturationLimit(600000);
    extractor1b.SetNoiseCalculation(kFALSE);

    MExtractTimeAndChargeSpline extractor1c("ExtractAmplitude");
    extractor1c.SetRange(first_slice, last_slice);
    extractor1c.SetChargeType(MExtralgoSpline::kAmplitude);
    extractor1c.SetSaturationLimit(600000);
    extractor1c.SetNoiseCalculation(kFALSE);
    extractor1c.SetNameSignalCam("Amplitude");
    extractor1c.SetNameTimeCam("AmplitudePos");

    // ---

    MHCamEvent evt1a(5, "CalRatio", "Ratio per slice between integrated signal and amplitude;; r [1/n]");
    evt1a.SetNameSub("Amplitude", kTRUE);
    MFillH fill1a(&evt1a, "MExtractedSignalCam", "FillRatio");
    fill1a.SetDrawOption("gaus");

    MParameterD ratio1a;
    ratio1a.SetVal(1./(fall_time_cal+rise_time_cal));
    fill1a.SetWeight(&ratio1a);

    // ---

    MHCamEvent evt1f(0, "ExtCalSig", "Extracted calibration signal;;S [mVsl]");
    MHCamEvent evt1g(4, "ExtCalTm",  "Extracted arrival times;;T [sl]");
    MHCamEvent evt1h(6, "CalCalTm",  "Calibrated arrival times;;T [sl]");

    MHSectorVsTime hist1rmsb("ExtSigVsTm");
    MHSectorVsTime hist1tmb("CalTmVsTm");
    hist1rmsb.SetTitle("Extracted calibration vs event number;;S [mVsl]");
    hist1rmsb.SetType(0);
    hist1tmb.SetTitle("Extracted arrival time vs event number;;T [sl]");
    //hist1tmb.SetType(4);
    hist1tmb.SetType(6);

    MFillH fill1f(&evt1f, "MExtractedSignalCam", "FillExtSig");
    MFillH fill1g(&evt1g, "MArrivalTimeCam",     "FillExtTm");
    MFillH fill1h(&evt1h, "MSignalCam",          "FillCalTm");
    MFillH fill1r(&hist1rmsb, "MExtractedSignalCam", "FillExtSigVsTm");
    //MFillH fill1j(&hist1tmb,  "MArrivalTimeCam",     "FillExtTmVsTm");
    MFillH fill1j(&hist1tmb,  "MSignalCam",     "FillCalTmVsTm");

    fill1f.SetDrawOption("gaus");
    fill1h.SetDrawOption("gaus");

    // ---

    MCalibrateDrsTimes calctm1a("CalibrateCalEvents");
    calctm1a.SetNameUncalibrated("UncalibratedTimes");

    MBadPixelsTreat treat1;
    treat1.SetProcessPedestalRun(kFALSE);
    treat1.SetProcessPedestalEvt(kFALSE);

    // ---

    MHCamEvent evt1c(6, "ExtCalTmShift", "Relative extracted arrival time of calibration pulse (w.r.t. event-median);;\\Delta T [ns]");
    MHCamEvent evt1d(6, "CalCalTmShift", "Relative calibrated arrival time of calibration pulse (w.r.t. event-median);;\\Delta T [ns]");

    evt1c.SetMedianShift();
    evt1d.SetMedianShift();

    MFillH fill1c(&evt1c, "UncalibratedTimes", "FillExtCalTm");
    MFillH fill1d(&evt1d, "MSignalCam",        "FillCalCalTm");
    fill1d.SetDrawOption("gaus");

    // ------------------ Setup eventloop and run analysis ---------------

    tlist1.AddToList(&read1);
    tlist1.AddToList(&apply1);
    tlist1.AddToList(&drsapply1);
    tlist1.AddToList(&cont1);
    tlist1.AddToList(&extractor1b);
    if (isinteg)
    {
        tlist1.AddToList(&extractor1c);
        tlist1.AddToList(&fill1a);
    }
    tlist1.AddToList(&calctm1a);
    tlist1.AddToList(&treat1);
    tlist1.AddToList(&fill1f);
    tlist1.AddToList(&fill1g);
    tlist1.AddToList(&fill1h);
    tlist1.AddToList(&fill1r);
    tlist1.AddToList(&fill1j);
    tlist1.AddToList(&fill1c);
    tlist1.AddToList(&fill1d);

    if (!loop1.Eventloop(max1))
        return 10;

    if (!loop1.GetDisplay())
        return 11;

    if (use_delays)
        timecam.SetDelays(*evt1h.GetHist());

    // ========================= Result ==================================

    Double_t avgS = evt1f.GetHist()->GetMean();
    Double_t medS = evt1f.GetHist()->GetMedian();
    Double_t rmsS = evt1f.GetHist()->GetRMS();
    Double_t maxS = evt1f.GetHist()->GetMaximum();

    MArrayF der1(hpulse->GetNbinsX());
    MArrayF der2(hpulse->GetNbinsX());

    MExtralgoSpline spline(hpulse->GetArray()+1, hpulse->GetNbinsX(),
                           der1.GetArray(), der2.GetArray());
    spline.SetRiseFallTime(rise_time_dat, fall_time_dat);
    spline.SetExtractionType(type);
    spline.SetHeightTm(heighttm);

    spline.Extract(hpulse->GetMaximumBin()-1);

    // The pulser signal is most probably around 400mV/9.5mV
    // IntegraFixed 2/48 corresponds to roughly 215mV*50slices
    Double_t scale = 1./spline.GetSignal();

    MArrayD calib(1440);
    for (int i=0; i<1440; i++)
    {
        Double_t g = gain->GetBinContent(i+1)>0.5 ? gain->GetBinContent(i+1) : 1;
        if (evt1f.GetHist()->GetBinContent(i+1)>0 && !badpixels[i].IsUnsuitable())
            calib[i] = lpref->GetBinContent(i+1) / evt1f.GetHist()->GetBinContent(i+1) / g;
    }

    gROOT->SetSelectedPad(0);
    d->AddTab("PulseTemp");
    gPad->SetGrid();
    hpulse->SetNameTitle("Pulse", "Single p.e. pulse template");
    hpulse->SetDirectory(0);
    hpulse->SetLineColor(kBlack);
    hpulse->DrawCopy();

    TAxis *ax = hpulse->GetXaxis();

    Double_t w = hpulse->GetBinWidth(1);
    Double_t T = w*(spline.GetTime()+0.5)       +ax->GetXmin();
    Double_t H = w*(hpulse->GetMaximumBin()+0.5)+ax->GetXmin();

    TLine line;
    line.SetLineColor(kRed);
    line.DrawLine(T-rise_time_dat*w, spline.GetHeight(),
                  T+fall_time_dat*w, spline.GetHeight());
    line.DrawLine(T, spline.GetHeight()/4, T, 3*spline.GetHeight()/4);
    line.DrawLine(T-rise_time_dat*w, 0,
                  T-rise_time_dat*w, spline.GetHeight());
    line.DrawLine(T+fall_time_dat*w, 0,
                  T+fall_time_dat*w, spline.GetHeight());

    TGraph gg;
    for (int ix=1; ix<=hpulse->GetNbinsX(); ix++)
        for (int i=0; i<10; i++)
        {
            Double_t x = hpulse->GetBinLowEdge(ix)+i*hpulse->GetBinWidth(ix)/10.;
            gg.SetPoint(gg.GetN(), x+w/2, spline.EvalAt(ix-1+i/10.));
        }

    gg.SetLineColor(kBlue);
    gg.SetMarkerColor(kBlue);
    gg.SetMarkerStyle(kFullDotMedium);
    gg.DrawClone("L");

    gROOT->SetSelectedPad(0);
    d->AddTab("CalConst");
    MGeomCamFACT fact;
    MHCamera hcalco(fact);
    hcalco.SetName("CalConst");
    hcalco.SetTitle(Form("Relative calibration constant [%.0f/pe]", 1./scale));
    hcalco.SetCamContent(calib);
    hcalco.SetAllUsed();
    //hcalco.Scale(scale);
    hcalco.DrawCopy();

    // ======================================================

    gLog << endl;
    gLog.Separator("Extracting random pedestal");

    MTaskList tlist3;

    MParList plist3;
    plist3.AddToList(&tlist3);
    plist3.AddToList(&drscalib300);
    plist3.AddToList(&badpixels);
    plist3.AddToList(&timecam);

    MEvtLoop loop3("DetermineRndmPed");
    loop3.SetDisplay(d);
    loop3.SetParList(&plist3);

    // ------------------ Setup the tasks ---------------

    MRawFitsRead read3;
    read3.LoadMap(map);
    read3.AddFile(pedfile);

    MFillH fill3a(&hrate);

    MContinue cont3("(MRawEvtHeader.GetTriggerID&0xff00)!=0x400", "SelectPed");

    MGeomApply apply3;

    MDrsCalibApply drsapply3;

    //---

    MExtractTimeAndChargeSpline extractor3;
    extractor3.SetRange(first_slice, last_slice);
    extractor3.SetRiseTimeHiGain(rise_time_dat);
    extractor3.SetFallTimeHiGain(fall_time_dat);
    extractor3.SetHeightTm(heighttm);
    extractor3.SetChargeType(type);
    extractor3.SetSaturationLimit(600000);
    extractor3.SetNoiseCalculation(kTRUE);

//    MHCamEvent evt2a(0, "PedRdm", "Extracted Pedestal Signal;;S");

//    MFillH fill2a(&evt2a, "MExtractedSignalCam", "FillPedRndm");

    // Use this for data, but not for calibration events
//    evt2a.SetErrorSpread(kFALSE);

    /*
     MCalibrateData conv3;
     conv3.SetCalibrationMode(MCalibrateData::kNone);
     conv3.SetPedestalFlag(MCalibrateData::kNo);
     conv3.SetCalibConvMinLimit(0);
     conv3.SetCalibConvMaxLimit(10000);
     conv3.SetScaleFactor(scale);
     */

    MCalibrateFact conv3;
    conv3.SetScale(scale);
    conv3.SetCalibConst(calib);

    MBadPixelsTreat treat3;
    treat3.SetProcessPedestalRun(kFALSE);
    treat3.SetProcessPedestalEvt(kFALSE);
    treat3.SetProcessTimes(kFALSE);

    MHCamEvent evt3b(0, "PedRdm","Interpolated random pedestal;;Signal [~phe]");
    //evt2b.SetErrorSpread(kFALSE);

    MFillH fill3b(&evt3b, "MSignalCam", "FillPedRdm");
    fill3b.SetDrawOption("gaus");

    // ------------------ Setup eventloop and run analysis ---------------

    tlist3.AddToList(&read3);
    tlist3.AddToList(&apply3);
    tlist3.AddToList(&drsapply3);
    tlist3.AddToList(&cont3);
    tlist3.AddToList(&extractor3);
//    tlist3.AddToList(&fill3a);
    tlist3.AddToList(&conv3);
    tlist3.AddToList(&treat3);
    tlist3.AddToList(&fill3b);

    if (!loop3.Eventloop(max3))
        return 14;

    if (!loop3.GetDisplay())
        return 15;

    // ======================================================

    gLog << endl;
    gLog.Separator("Extracting pedestal");

    MTaskList tlist4;

    MParList plist4;
    plist4.AddToList(&tlist4);
    plist4.AddToList(&drscalib300);
    plist4.AddToList(&badpixels);
    plist4.AddToList(&timecam);

    MEvtLoop loop4("DetermineExtractedPed");
    loop4.SetDisplay(d);
    loop4.SetParList(&plist4);

    // ------------------ Setup the tasks ---------------

    MRawFitsRead read4;
    read4.LoadMap(map);
    read4.AddFile(pedfile);

    MContinue cont4("(MRawEvtHeader.GetTriggerID&0xff00)!=0x400", "SelectPed");

    MGeomApply apply4;

    MDrsCalibApply drsapply4;

    MExtractTimeAndChargeSpline extractor4;
    extractor4.SetRange(first_slice, last_slice);
    extractor4.SetRiseTimeHiGain(rise_time_dat);
    extractor4.SetFallTimeHiGain(fall_time_dat);
    extractor4.SetHeightTm(heighttm);
    extractor4.SetChargeType(type);
    extractor4.SetSaturationLimit(600000);
    extractor4.SetNoiseCalculation(kFALSE);

    //    MHCamEvent evt3a(0, "PedExt", "Extracted Pedestal Signal;;S");

    //    MFillH fill3a(&evt3a, "MExtractedSignalCam", "FillPedExt");

    // Use this for data, but not for calibration events
//    evt3a.SetErrorSpread(kFALSE);
/*
    MCalibrateData conv4;
    conv4.SetCalibrationMode(MCalibrateData::kNone);
    conv4.SetPedestalFlag(MCalibrateData::kNo);
    conv4.SetCalibConvMinLimit(0);
    conv4.SetCalibConvMaxLimit(10000);
    conv4.SetScaleFactor(scale);
*/
    MCalibrateFact conv4;
    conv4.SetScale(scale);
    conv4.SetCalibConst(calib);

    MBadPixelsTreat treat4;
    treat4.SetProcessPedestalRun(kFALSE);
    treat4.SetProcessPedestalEvt(kFALSE);

    MHCamEvent evt4b(0, "PedExt","Interpolated extracted pedestal;;Signal [~phe]");
    //evt4b.SetErrorSpread(kFALSE);

    MFillH fill4b(&evt4b, "MSignalCam", "FillPedExt");
    fill4b.SetDrawOption("gaus");

    // ------------------ Setup eventloop and run analysis ---------------

    tlist4.AddToList(&read4);
    tlist4.AddToList(&apply4);
    tlist4.AddToList(&drsapply4);
    tlist4.AddToList(&cont4);
    tlist4.AddToList(&extractor4);
//    tlist4.AddToList(&fill4a);
    tlist4.AddToList(&conv4);
    tlist4.AddToList(&treat4);
    tlist4.AddToList(&fill4b);

    if (!loop4.Eventloop(max4))
        return 15;

    if (!loop4.GetDisplay())
        return 16;

    // ===================================================================

    gLog << endl;
    gLog.Separator("Extracting and calibration data");

    MTaskList tlist5;

    MParList plist5;
    plist5.AddToList(&tlist5);
    plist5.AddToList(&drscalib300);
    plist5.AddToList(&badpixels);
    plist5.AddToList(&timecam);

    MEvtLoop loop5("CalibratingData");
    loop5.SetDisplay(d);
    loop5.SetParList(&plist5);

    // ------------------ Setup the tasks ---------------

    MRawFitsRead read5;
    read5.LoadMap(map);
    read5.AddFiles(iter);

    MFillH fill5a(&hrate);

    MGeomApply apply5;

    MDrsCalibApply drsapply5;

    MFDataPhrase filterdat("(MRawEvtHeader.GetTriggerID&0xff00)==0",     "SelectDat");
    MFDataPhrase filtercal("(MRawEvtHeader.GetTriggerID&0xff00)==0x100", "SelectCal");
    MFDataPhrase filterped("(MRawEvtHeader.GetTriggerID&0xff00)==0x400", "SelectPed");
    MFDataPhrase filterncl("(MRawEvtHeader.GetTriggerID&0xff00)!=0x100", "SelectNonCal");

    //MContinue cont4("MRawEvtHeader.GetTriggerID!=4", "SelectData");

    // ---

    MExtractTimeAndChargeSpline extractor5dat;
    extractor5dat.SetRange(first_slice, last_slice);
    extractor5dat.SetRiseTimeHiGain(rise_time_dat);
    extractor5dat.SetFallTimeHiGain(fall_time_dat);
    extractor5dat.SetHeightTm(heighttm);
    extractor5dat.SetChargeType(type);
    extractor5dat.SetSaturationLimit(600000);
    extractor5dat.SetNoiseCalculation(kFALSE);

    MExtractTimeAndChargeSpline extractor5cal;
    extractor5cal.SetRange(first_slice, last_slice);
    extractor5cal.SetRiseTimeHiGain(rise_time_cal);
    extractor5cal.SetFallTimeHiGain(fall_time_cal);
    extractor5cal.SetHeightTm(heighttm);
    extractor5cal.SetChargeType(type);
    extractor5cal.SetSaturationLimit(600000);
    extractor5cal.SetNoiseCalculation(kFALSE);

    MExtractTimeAndChargeSpline extractor5tm("ExtractTM");
    extractor5tm.SetRange(last_slice, 294);
    extractor5tm.SetRiseTimeHiGain(1);
    extractor5tm.SetFallTimeHiGain(1);
    extractor5tm.SetHeightTm(0.5);
    extractor5tm.SetChargeType(MExtralgoSpline::kAmplitudeRel);
    extractor5tm.SetSaturationLimit(600000);
    extractor5tm.SetNoiseCalculation(kFALSE);
    extractor5tm.SetNameSignalCam("TimeMarkerAmplitude");
    extractor5tm.SetNameTimeCam("TimeMarkerTime");

    extractor5dat.SetFilter(&filterncl);
    extractor5cal.SetFilter(&filtercal);
    //extractor4tm.SetFilter(&filtercal);

    // ---
/*
    MCalibrateData conv5;
    conv5.SetCalibrationMode(MCalibrateData::kNone);
    conv5.SetPedestalFlag(MCalibrateData::kNo);
    conv5.SetCalibConvMinLimit(0);
    conv5.SetCalibConvMaxLimit(10000);
    conv5.SetScaleFactor(scale);
*/
    MCalibrateFact conv5;
    conv5.SetScale(scale);
    conv5.SetCalibConst(calib);

    MCalibrateDrsTimes calctm5;
    calctm5.SetNameUncalibrated("UncalibratedTimes");

    MCalibrateDrsTimes calctm5tm("CalibrateTimeMarker");
    calctm5tm.SetNameArrivalTime("TimeMarkerTime");
    calctm5tm.SetNameUncalibrated("UncalTimeMarker");
    calctm5tm.SetNameCalibrated("TimeMarker");
    calctm5tm.SetTimeMarker();
    //calctm4tm.SetFilter(&filtercal);

    MHCamEvent evt5m(6, "ExtTm",      "Extracted arrival times of calibration pulse;;\\Delta T [ns]");
    MHCamEvent evt5n(6, "CalTm",      "Calibrated arrival times of calibration pulse;;\\Delta T [ns]");
    MHCamEvent evt5q(6, "ExtTmShift", "Relative extracted arrival times of calibration pulse (w.r.t. event-median);;\\Delta T [ns]");
    MHCamEvent evt5r(6, "CalTmShift", "Relative calibrated arrival times of calibration pulse (w.r.t. event-median);;\\Delta T [ns]");
    MHCamEvent evt5s(6, "ExtTM",      "Extracted absolute time marker position;;T [sl]");
    MHCamEvent evt5t(6, "CalTM",      "Calibrated absolute time marker position;;T [ns]");
    MHCamEvent evt5u(6, "ExtTMshift", "Relative extracted time marker position (w.r.t. event-median);;\\Delta T [ns]");
    MHCamEvent evt5v(6, "CalTMshift", "Relative calibrated time marker position (w.r.t. event-median);;\\Delta T [ns]");
    MHCamEvent evt5w(6, "ExtDiff",    "Difference between extracted arrival time of time marker and calibration pulse;;\\Delta T [ns]");
    MHCamEvent evt5x(6, "CalDiff",    "Difference between calibrated arrival time of time marker and calibration pulse;;\\Delta T [ns]");

    evt5w.SetNameSub("UncalibratedTimes");
    evt5x.SetNameSub("MSignalCam");

    evt5q.SetMedianShift();
    evt5r.SetMedianShift();
    evt5u.SetMedianShift();
    evt5v.SetMedianShift();
    //evt4w.SetMedianShift();
    //evt4x.SetMedianShift();

    MFillH fill5m(&evt5m, "UncalibratedTimes", "FillExtTm");
    MFillH fill5n(&evt5n, "MSignalCam",        "FillCalTm");
    MFillH fill5q(&evt5q, "UncalibratedTimes", "FillExtTmShift");
    MFillH fill5r(&evt5r, "MSignalCam"       , "FillCalTmShift");
    MFillH fill5s(&evt5s, "UncalTimeMarker",   "FillExtTM");
    MFillH fill5t(&evt5t, "TimeMarker",        "FillCalTM");
    MFillH fill5u(&evt5u, "UncalTimeMarker",   "FillExtTMshift");
    MFillH fill5v(&evt5v, "TimeMarker",        "FillCalTMshift");
    MFillH fill5w(&evt5w, "UncalTimeMarker",   "FillExtDiff");
    MFillH fill5x(&evt5x, "TimeMarker",        "FillCalDiff");

    fill5m.SetDrawOption("gaus");
    fill5n.SetDrawOption("gaus");
    fill5q.SetDrawOption("gaus");
    fill5r.SetDrawOption("gaus");
    //fill5s.SetDrawOption("gaus");
    //fill5t.SetDrawOption("gaus");
    //fill5u.SetDrawOption("gaus");
    //fill5v.SetDrawOption("gaus");
    //fill5w.SetDrawOption("gaus");
    //fill5x.SetDrawOption("gaus");


    MBadPixelsTreat treat5;
    treat5.SetProcessPedestalRun(kFALSE);
    treat5.SetProcessPedestalEvt(kFALSE);

    MHSectorVsTime hist5cal("CalVsTm");
    MHSectorVsTime hist5ped("PedVsTm");
    hist5cal.SetTitle("Median calibrated calibration signal vs event number;;Signal [~phe]");
    hist5ped.SetTitle("Median calibrated pedestal signal vs event number;;Signal [~phe]");
    hist5cal.SetType(0);
    hist5ped.SetType(0);
    hist5cal.SetMinimum(0);
    hist5ped.SetMinimum(0);
    hist5cal.SetUseMedian();
    hist5ped.SetUseMedian();
    hist5cal.SetNameTime("MTime");
    hist5ped.SetNameTime("MTime");

    MFillH fill5cal(&hist5cal, "MSignalCam", "FillCalVsTm");
    MFillH fill5ped(&hist5ped, "MSignalCam", "FillPedVsTm");
    fill5cal.SetFilter(&filtercal);
    fill5ped.SetFilter(&filterped);

    MHCamEvent evt5b(0, "ExtSig",   "Extracted signal;;S [mVsl]");
    MHCamEvent evt5c(0, "CalSig",   "Calibrated and interpolated signal;;S [~phe]");
    MHCamEvent evt5d(4, "ExtSigTm", "Extracted time;;T [sl]");
    MHCamEvent evt5e(6, "CalSigTm", "Calibrated and interpolated time;;T [ns]");

    MFillH fill5b(&evt5b, "MExtractedSignalCam", "FillExtSig");
    MFillH fill5c(&evt5c, "MSignalCam",          "FillCalSig");
    MFillH fill5d(&evt5d, "MArrivalTimeCam",     "FillExtTm");
    MFillH fill5e(&evt5e, "MSignalCam",          "FillCalTm");

    fill5c.SetDrawOption("gaus");
    fill5d.SetDrawOption("gaus");
    fill5e.SetDrawOption("gaus");

    /*
    fill4b.SetFilter(&filterdat);
    fill4c.SetFilter(&filterdat);
    fill4d.SetFilter(&filterdat);
    fill4e.SetFilter(&filterdat);
    */

    //MFSoftwareTrigger swtrig;
    //MContinue contsw(&swtrig, "FilterSwTrigger", "Software trigger");
    //contsw.SetInverted();

    const TString fname(Form("s/([0-9]+_[0-9]+)[.]fits([.]gz)?$/%s\\/$1_C.root/",
                             MJob::Esc(outpath).Data()));

    // The second rule is for the case reading raw-files!
    MWriteRootFile write5(2, fname, "RECREATE", "Calibrated Data");
    write5.AddContainer("MRawRunHeader", "RunHeaders");
    write5.AddContainer("MGeomCam",      "RunHeaders");
    write5.AddContainer("MSignalCam",    "Events");
    write5.AddContainer("MTime",         "Events");
    write5.AddContainer("MRawEvtHeader", "Events");
    //write.AddContainer("MTriggerPattern", "Events");

    // ------------------ Setup histograms and fill tasks ----------------

    MContinue test;
    test.SetFilter(&filterncl);

    MTaskList tlist5tm;
    tlist5tm.AddToList(&extractor5tm);
    tlist5tm.AddToList(&calctm5tm);
    tlist5tm.AddToList(&fill5m);
    tlist5tm.AddToList(&fill5n);
    tlist5tm.AddToList(&fill5q);
    tlist5tm.AddToList(&fill5r);
    //tlist5tm.AddToList(&fill5s);
    //tlist5tm.AddToList(&fill5t);
    tlist5tm.AddToList(&fill5u);
    tlist5tm.AddToList(&fill5v);
    tlist5tm.AddToList(&fill5w);
    tlist5tm.AddToList(&fill5x);
    tlist5tm.SetFilter(&filtercal);

    MTaskList tlist5dat;
    tlist5dat.AddToList(&fill5b);
    tlist5dat.AddToList(&fill5c);
    tlist5dat.AddToList(&fill5d);
    tlist5dat.AddToList(&fill5e);
    tlist5dat.SetFilter(&filterdat);

    tlist5.AddToList(&read5);
    tlist5.AddToList(&apply5);
    tlist5.AddToList(&drsapply5);
    tlist5.AddToList(&filterncl);
    //tlist5.AddToList(&test);
    tlist5.AddToList(&filterdat);
    tlist5.AddToList(&filtercal);
    tlist5.AddToList(&filterped);
    tlist5.AddToList(&fill5a);
    tlist5.AddToList(&extractor5dat);
    tlist5.AddToList(&extractor5cal);
    tlist5.AddToList(&calctm5);
    tlist5.AddToList(&tlist5tm);
    tlist5.AddToList(&conv5);
    tlist5.AddToList(&treat5);
    tlist5.AddToList(&fill5ped);
    tlist5.AddToList(&fill5cal);
    tlist5.AddToList(&tlist5dat);
    tlist5.AddToList(&write5);

    if (!loop5.Eventloop(max4))
        return 18;

    if (!loop5.GetDisplay())
        return 19;

    TString title = "--  Calibrated signal #";
    title += seq.GetSequence();
    title += " (";
    title += drsfile;
    title += ")  --";
    d->SetTitle(title, kFALSE);

    TString path;
    path += Form("%s/20%6d_%03d-calibration.root", outpath,
                 seq.GetSequence()/1000, seq.GetSequence()%1000);

    d->SaveAs(path);

    return 0;
}
