//=//////////////////////////////////////////////////////////////////////
//=
//= camera                
//=
//= @file        camera.h
//= @desc        Header file
//= @author      J C Gonzalez
//= @email       gonzalez@mppmu.mpg.de
//= @date        Thu May  7 16:24:22 1998
//=
//=----------------------------------------------------------------------
//=
//= Created: Thu May  7 16:24:22 1998
//= Author:  Jose Carlos Gonzalez
//= Purpose: Program for reflector simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: camera.h,v $
//= $Revision: 1.1.1.1 $
//= $Author: harald $ 
//= $Date: 1999-11-05 11:59:31 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |camera.h|.

/*!@"

  This section shows the include file |camera.h|

  @"*/

//=-----------------------------------------------------------
//!@subsection Include files.

//!@{

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <libgen.h>

#include "camera-v.h"

#include "jcmacros.h"
#include "jcdebug.h"

#include "creadparam.h"
#include "atm.h"
#include "moments.h"

#include "lagrange.h"

#include "MCEventHeader.hxx"
#include "MCCphoton.hxx"

// command line options available

#define COMMAND_LINE_OPTIONS    "f:h"

/*@'

  This is C++, but RANLIB routines are written in pure ANSI C.
  In order to read easily these routines, we must include
  the following directive
  
*/

extern "C" { 
#include "ranlib.h"       
}

// version of the reflector program that can read

#define REFL_PROGRAM reflector
#define REFL_VERSION 0.4

const char REFL_SIGNATURE[] = GLUE_postp( REFL_PROGRAM, REFL_VERSION );

//!@}

//=------------------------------------------------------------
//!@subsection Macro-definitions, and constants.

//!@{

#define SLICES            100
#define WIDTH_TIMESLICE   3.3
 
#define SIN60   0.866025403784439
#define COS60   0.500000000000000

#define RandomNumber  drand48()

#define PIX_ARRAY_SIDE       40
#define PIX_ARRAY_HALF_SIDE  20
#define PIXNUM               0
#define PIXX                 1
#define PIXY                 2

#define iMAXNUMPIX  595 // total maximum possible number of pixels in the camera

//@ the trigger threshold up to which the maximum passable threshold is tested 
#define iMAX_THRESHOLD_PHE   50 

//@ number of the 1st. pixel of a sector s in a ring r (central pixel: ring=0)
#define FIRST_PIXEL(r,s)   ( ((r)>0) ? (3*(r)*((r)-1) + (r)*(s) + 1) : 0 )

//@ number of the pixels include in a camera of r pixels
#define NUMBER_PIXELS(r)   ( ((r)>0) ? FIRST_PIXEL((r)+1,0) : 1 )

//@ now we define the list CT_ITEM_LIST of possible items in the CT def. file
#define CT_ITEM_LIST  /* LIST OF ITEMS IN THE CT DEFINITION FILE */  \
T(type),              /* type of definition file */                  \
T(focal_distance),    /* std(focal distance) */                      \
T(focal_std),         /* focal distance */                           \
T(point_spread),      /* std(point spread)   */                      \
T(point_std),         /* point spread   */                           \
T(adjustment_dev),    /* std of adjustment deviation   */            \
T(black_spot),        /* radius of the black spot in center of mirror */ \
T(n_mirrors),         /* number of mirrors */                        \
T(r_mirror),          /* radius of one mirror */                     \
T(camera_width),      /* camera width */                             \
T(n_pixels),          /* total number of pixels in the camera */     \
T(n_centralpixels),   /* number of central pixels in the camera */   \
T(n_gappixels),       /* number of gap pixels in the camera */       \
T(pixel_width),       /* pixel width */                              \
T(define_mirrors)     /* this entry is followed by the def. of pixels */
  
#define T(x)  x               //@< define T() as the name as it is

enum CT_ITEM_TYPE {
  CT_ITEM_LIST
};

#undef T

#define T(x)  #x              //@< define T() as the string of x

const char *const CT_ITEM_NAMES[] = {
  CT_ITEM_LIST
};

#undef T


// TYPE=0  (CT1)
//     i   s   rho   theta   x   y   z   thetan  phin  xn   yn   zn
//
//      i : number of the mirror
//      s : arc length [cm]
//    rho : polar rho of the position of the center of the mirror [cm]
//  theta : polar angle of the position of the center of the mirror [cm]
//      x : x coordinate of the center of the mirror [cm]
//      y : y coordinate of the center of the mirror [cm]
//      z : z coordinate of the center of the mirror [cm]
// thetan : polar theta angle of the direction where the mirror points to
//   phin : polar phi angle of the direction where the mirror points to
//     xn : xn coordinate of the normal vector in the center (normalized)
//     yn : yn coordinate of the normal vector in the center (normalized)
//     zn : zn coordinate of the normal vector in the center (normalized)
//
// TYPE=1  (MAGIC)
//     i  f   sx   sy   x   y   z   thetan  phin 
//
//      i : number of the mirror
//      f : focal distance of that mirror
//     sx : curvilinear coordinate of mirror's center in X[cm]
//     sy : curvilinear coordinate of mirror's center in X[cm]
//      x : x coordinate of the center of the mirror [cm]
//      y : y coordinate of the center of the mirror [cm]
//      z : z coordinate of the center of the mirror [cm]
// thetan : polar theta angle of the direction where the mirror points to
//   phin : polar phi angle of the direction where the mirror points to
//     xn : xn coordinate of the normal vector in the center (normalized)
//     yn : yn coordinate of the normal vector in the center (normalized)
//     zn : zn coordinate of the normal vector in the center (normalized)

#define CT_I       0

#define CT_S       1
#define CT_RHO     2
#define CT_THETA   3

#define CT_FOCAL   1
#define CT_SX      2
#define CT_SY      3

#define CT_X       4
#define CT_Y       5
#define CT_Z       6
#define CT_THETAN  7
#define CT_PHIN    8
#define CT_XN      9
#define CT_YN     10
#define CT_ZN     11

#define CT_NDATA  12
 
//!@}

//=------------------------------------------------------------
//!@subsection data types

struct camera { /* camera parameters for imaging */
  int inumpixels;
  int inumcentralpixels;
  int inumgappixels;
  int inumbigpixels;
  double dpixdiameter_cm; /* diameter of the central and gap pixels in centimeters */
  double dpixsizefactor[iMAXNUMPIX]; /* size of the pixel relative to  dpixdiameter_deg */
  double dxc[iMAXNUMPIX]; /* Pixel coordinates in camera coordinate system (x points from pixel 1 to 2). */
  double dyc[iMAXNUMPIX]; /* The numbering of the pixels in these arrays starts at 0! */
  double dxpointcorr_deg; /* correction of the pixel coordinates; to be added to dxc[] to get correct value */
  double dypointcorr_deg; /* correction of the pixel coordinates; to be added to dxc[] to get correct value */
  double di[iMAXNUMPIX]; /* i coordinate in JCs bi-axis hexagonal coordinate system */
  double dj[iMAXNUMPIX]; /* j coordinate in JCs bi-axis hexagonal coordinate system */
 
};


//=------------------------------------------------------------
//!@subsection Pre-defined file names.

//!@{

#define QE_FILE     "qe.dat"

//!@}

//=------------------------------------------------------------
//!@subsection Prototypes of functions.

//!@{

//++
// prototypes
//--

#define ONoff(x)  ((x==TRUE) ? "[ ON ]" : "[ off]")

// Under Linux, the nint function does not exist, so we have to define it.
#define nint(x)  ((int)floor((x)+0.5))

void present(void);
void usage(void);
void clean(void);
void log(const char *funct, char *fmt, ...);
void error(const char *funct, char *fmt, ...);
int isA( char * s1, const char * flag );
void read_ct_file(void);
int igen_pixel_coordinates(struct camera *cam);
void read_pixels(struct camera *cam); 
int pixels_are_neig(int pix1, int pix2);
int bpoint_is_in_pix(double dx, double dy, int ipixnum, struct camera *pcam); 
float  dist_r_P(float a, float b, float c, 
                float l, float m, float n,
                float x, float y, float z);
     
//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{

/*
 *$Log: not supported by cvs2svn $
 *Revision 1.3  1999/10/22 15:32:56  petry
 *tidied-up version, really sent to H.K. and N.M., 22-10-99
 *
 *Revision 1.2  1999/10/22 15:01:28  petry
 *version sent to H.K. and N.M. on Fri Oct 22 1999
 *
 *Revision 1.1.1.1  1999/10/21 16:35:10  petry
 *first synthesised version
 *
 * Revision 1.8  1999/03/15  14:59:06  gonzalez
 * camera-1_1
 *
 * Revision 1.7  1999/03/02  09:56:11  gonzalez
 * *** empty log message ***
 *
 * Revision 1.6  1999/01/14  17:32:40  gonzalez
 * Added to camera the STDIN input option (data_from_input)
 *
 */

//!@}
//=EOF

