/*-----------------------------------------------------------------------------
mitop.h -- Top side (Host) of messaging interface to VMOD-ICAN

Copyright (c) 1994 JANZ Computer AG
All Rights Reserved

Created 95/06/16 by Soenke Hansen
Version 1.40 of 00/03/29

Functions to fill messages which can be sent to VMOD-ICAN.
Functions to format the contents of messages for printing.

The relevance of messages depends on the firmware on VMOD-ICAN.
The firmware discards messages which it does not understand.
Through preprocessor directives the set of messaging interface
functions are adapted to the firmware.

This file contains prototype declarations for functions in mitop.c.

-----------------------------------------------------------------------------*/


#ifndef mitop_DEFINED
#define mitop_DEFINED

#ifdef __cplusplus
extern "C" {
#endif

#include "defs.h"
#include "dpm.h"

/* Classes of messages. */
#define MITOP_MGR	0x0001	/* module management */
#define MITOP_ICANOS	0x0002	/* raw CAN access, SW CAN filtering, timers */
#define MITOP_CAL	0x0004	/* CAL */

#define MITOP	(MITOP_CAL + MITOP_MGR + MITOP_ICANOS)
/* Adapt the mesaging interface to the firmware capabilities.
   The default is ICANOS and VMOD-ICAN management. */
#ifndef MITOP
#define MITOP (MITOP_MGR + MITOP_ICANOS)
#endif

/* Formatting functions are implemented if and only if
   MITOP_FORMAT is defined. */
#define MITOP_FORMAT


#if MITOP & MITOP_MGR

/* Request identity and version from module */
extern void IdVersReq(Message *);
extern void IdVersSniffReq(Message *);

/* Request free memory size */
extern void CheckFreeMem(Message *, int);

/* Request host interface style */
extern void InquiryHostInterface(Message *);

/* Request CANbus bit rate, HW acceptance code/mask and termination state  */
extern void  InquiryCANBitRateMasksTerm(Message *);

/* Request CAN controllers / modules status */
extern void InquiryStatus(Message *);
extern void InquiryStatusSniff(Message *);
extern void InquiryControllerStatusSniff(Message *m);

/* Request setting of software acceptance filter */
extern void InquiryAcceptanceFilter(Message *, WORD_t);

#if MITOP & MITOP_CAL

/* Install CAL master on module.  By default the module implements
   only CAL slave functionality. */
extern void InstallCalMaster(Message *);

/* Some host-guarding stuff */
void InitHostGuarding(Message *, int, BYTE_t);
void HostIsAlive(Message *);

#endif


void SwitchCanTermination(Message *, WORD_t);
void SwitchSYSClkTermination(Message *, WORD_t);

#endif

#if MITOP & MITOP_ICANOS

/* format sja1000 error-code-capture to readable form. */
int sprintf_sja1000_ec(
	char	*dest,	/* destination for description */
	int     ec,	/* error code to format */
	int     n );	/* max length of description */

/* Start periodical clock interrupt generation. */
extern void IcStartTimer(
	Message	*,		/* message buffer */
	WORD_t	);		/* max. clock period in 100 usec units */

/* Stop periodical clock interrupt generation. */
extern void IcStopTimer(
	Message	*);		/* message buffer */

/* Unset a user-defined timer. */
extern void IcClearTimer(
	Message	*,		/* message buffer */
	WORD_t	);		/* timer Id as confirmed on set timer req. */

/* Set a timer.  The alarm time "tleft" is in units of 100 usecs. */
extern void IcSetTimer(
	Message	*,		/* message buffer */
	WORD_t	,		/* user request Id */
	WORD_t	);		/* time left until alarm */

/* Issue request to send a CAN message. */
extern void IcSendReqBCAN(
	Message	*,		/* message buffer */
	int	,		/* pos. confirm. or echo required? */
	WORD_t	,		/* application request Id */
	WORD_t	,		/* CAN descriptor (id,rtr,dlc) */
	const BYTE_t *);	/* data bytes to transmit */

/* Abort transmit request. */
extern void IcSendAbortBCAN(
	Message	*);		/* message buffer */

/* Switch CAN controller bus-on. */
extern void IcBusOnBCAN(
	Message	*);		/* message buffer */

/* Switch CAN controller bus-off. */
extern void IcBusOffBCAN(
	Message	*);		/* message buffer */

/* Set bit timing parameters in CAN controller. */
extern void IcWriteBtrBCAN(
	Message	*,		/* message buffer */
	WORD_t	);		/* BTR0 and BTR1 */

/* Set error-warining limit in CAN controller. */
void IcWriteEwlBCAN(
	Message	*m,		/* message buffer */
	BYTE_t	ewl );		/* Error warning limit to set */

/* Enable bus-error reporting. */
void IcSwitchBerrBCAN(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Configure self test mode */
void IcSwitchStmBCAN(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Configure listen-only mode */
void IcSwitchLomBCAN(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Set hardware acceptance filter in CAN controller. */
extern void IcWriteAcmBCAN(
	Message	*,		/* message buffer */
	WORD_t	);		/* acceptance code and mask */

/* Set extended hardware acceptance filter in CAN controller. */
void IcWriteExtAcmBCAN(
	Message	*m,		/* message buffer */
	int	mode,		/* 0: dual-, 1: single-filter-mode */
	LWORD_t	ac,		/* acceptance code */
	LWORD_t am );		/* acceptance mask */

/* Change software acceptance filter: accept all CAN messages
   with given Id. */
extern void IcOpenAfil(
	Message	*,		/* message buffer */
	WORD_t	);		/* this Id passes the filter */

/* Change software acceptance filter: reject all CAN messages
   with given Id. */
extern void IcCloseAfil(
	Message	*,		/* message buffer */
	WORD_t	);		/* messages with this Id are rejected */

/* Change software acceptance filter: accept all CAN messages
   with IDs in specified range. */
extern void IcRangeOpenAfil(
	Message	*,		/* message buffer */
	WORD_t	,		/* lower bound of Id range */
	WORD_t	);		/* upper bound of Id range */

/* Change software acceptance filter: reject all CAN messages
   with IDs in specified range. */
extern void IcRangeCloseAfil(
	Message	*,		/* message buffer */
	WORD_t	,		/* lower bound of Id range */
	WORD_t	);		/* upper bound of Id range */

/* Set software acceptance filter mask. */
void IcSetAfil(
	Message	*m,		/* message buffer */
	WORD_t	id,		/* this Ids filter is set ... */
	WORD_t	mask		/* ... to this value. */
);

/* Set software acceptance filter mask for a range of Ids. */
void IcRangeSetAfil(
	Message	*m,		/* message buffer */
	WORD_t	idl,		/* this Ids filter is set ... */
	WORD_t	idh,		/* this Ids filter is set ... */
	WORD_t	mask		/* ... to this value. */
);

void IcCreateCycSendList(
	Message		*m,	/* message buffer */
	WORD_t		cyctim 	/* cycle-time to send data */
);

void IcDeleteCycSendList(
	Message		*m,	/* message buffer */
	BYTE_t		listNo	/* number of list to delete */
);

void IcAddMsgToCycSendList(
	Message		*m,	/* message buffer */
	BYTE_t		listId,	/* list number where the message should */
				/* be added to */
	WORD_t		id,	/* CANbus id */
	BYTE_t		len,	/* CANbus message length */
	BYTE_t		rtr,	/* rtr bit [0,1] */
	BYTE_t		*data	/* pointer to data */
);

/* Set priority boundaries. */
void IcSetPrioBnd(
	Message	*m,			/* message buffer */
	WORD_t	hi_mid_boundary,	/* boundary between hi and mid prior. */
	WORD_t	mid_low_boundary	/* boundary between mid and low prior.*/
);


/* Set host endianess. */
void SetHostEndianess(
	Message	*m,		/* message buffer */
	BYTE_t	endianess	/* endianess */
);

/* Init bulk buffer with timeout only */
void InitBulkBufferTimeout(
	Message	*m,		/* message buffer */
	WORD_t	timeout		/* time after bulk buffer message will be sent */
);

/* Init bulk buffer with timeout and sizes */
void InitBulkBuffer(
	Message	*m,		 /* message buffer */
	WORD_t	timeout,	 /* time after bulk buffer message will be sent */
	BYTE_t	maxMsgInBulkMsg, /* max. number of bulk msgs. in one bulk message */
	BYTE_t	maxBulkMsgPreAlloc /* max. number of preallocated bulk messages */
);

/* Init sniff buffer */
void InitSniffBufferTimeout(
	Message	*m,		/* message buffer */
	WORD_t	timeout		/* time after sniff buffer message will be sent */
);
void InitSniffBuffer(
	Message	*m,		   /* message buffer */
	WORD_t	timeout,	   /* time after bulk buffer message will be sent */
	BYTE_t	maxByteInSniffMsg,  /* max. number of msg.-bytes in one sniff message */
	BYTE_t	maxSniffMsgPreAlloc /* max. number of preallocated sniff messages */
);

/* Config sniff buffer */
void ConfigSniffBufferEcho(
	Message	*m,		/* message buffer */
	WORD_t	echoFromQueue	/* specifies the queues, a sent message should be echoed from */
);

void ConfigSniffBufferAfilStd(
	Message	*m,		/* message buffer */
	WORD_t	idl,		/* lower bound of Id range */
	WORD_t	idh,		/* upper bound of Id range */
	BYTE_t	mask		/* filter mask to use ([0..1] */
);

void ConfigSniffBufferAfilXtd(
	Message	*m,		/* message buffer */
	WORD_t	idl,		/* lower bound of Id range (13 MSB!) */
	WORD_t	idh,		/* upper bound of Id range (13 MSB!) */
	BYTE_t	mask		/* filter mask to use ([0..1] */
);

void InitBusLoadStatistic(
	Message	*m,		/* message buffer */
	WORD_t	intTime,	/* integration time (in times of 10 ms) */
	BYTE_t	flags		/* some flags */
);

void InitSniffBusLoadStatistic(
	Message	*m,		/* message buffer */
	WORD_t	intTime,	/* integration time (in times of 10 ms) */
	BYTE_t	flags		/* some flags */
);

void RequestBusLoadStatistic(
	Message	*m,		/* message buffer */
	BYTE_t	choice		/* choose the kind of statistics */
);

#endif /* MITOP & MITOP_ICANOS */


#if MITOP & MITOP_CAL

/* Create Node request */
extern void NsCreateNodeReq(
	Message	*,		/* message buffer */
	int,			/* module Id */
	const char *,		/* module name string (strlen==7) */
	int);			/* CAL node class */

/* Create Node request */
extern void NsCreateCOPNodeReq(
	Message	*,		/* message buffer */
	int,			/* module Id */
	const char *,		/* module name string (strlen==7) */
	int);			/* CAL node class */

/* Config NMT of CANopen slave node request */
extern void NsConfigCOPNodeReq(
	Message	*m,		/* message buffer */
	WORD_t	guard_time,	/* guard time in ms */
	int	life_fac,	/* life time factor */
	int	guard_start);	/* timepoint of guarding start */

/* Config CAL NMT slave node request */
extern void NsConfigNodeReq(
	Message *m,		/* message buffer */
	WORD_t	mux,		/* multiplexor */
	WORD_t	flags);		/* flags */

/* Configuration download acknowledge response */
extern void NsConfigAck(
	Message *m);		/* message buffer */

/* NMT slave configuration transfer abort request */
extern void NsConfigAbort(
	Message *m,			/* message buffer */
	int 	failureReason,		/* failure reason */
	int	applSpecificReason); 	/* appl. specific failure reason */

/* NMT slave configuration transfer verify response */
extern void NsConfigVerify(
	Message *m,			/* message buffer */
	int 	verificationResult,	/* result of verification */
	int	reason);		/* failure reason */

/* Delete Node request */
extern void NsDeleteNodeReq(
	Message	*);		/* message buffer */

/* Identify Node request */
extern void NsIdentifyNodeReq(
	Message	*);		/* message buffer */

/* Connect Node request */
extern void NsConnectNodeReq(
	Message	*,		/* message buffer */
	int	,		/* download requested? */
	WORD_t	,		/* guard time in ms */
	int	);		/* life time factor */

/* Disconnect Node request */
extern void NsDisconnectNodeReq(
	Message	*);		/* message buffer */

/* Prepare Remote Node Response */
extern void NsPrepareRemoteNodeRes(
	Message	*,		/* message buffer */
	int	);		/* 0 or error code of CAL protocol */

/* Delete User Definition request */
extern void DsDeleteUserDefReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Create all User Definitions request:
   Initiate the Create User Definition service for the COB's
   used by by the currently defined CMS objects. */
extern void DsCreateAllUserDefReq(
	Message	*);		/* message buffer */

/* Get Checksum request */
extern void DsGetChecksumReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Delete User Definition request */
extern void DsVerifyCobClassReq(
	Message	*);		/* message buffer */

/* Set Bit Timing and Attach CAN controller to CAN bus */
extern void LsBusOnReq(
	Message	*,		/* message buffer */
	WORD_t	);		/* Bit timing parameters */

/* Detach CAN controller from CAN bus */
extern void LsBusOffReq(
	Message	*);		/* message buffer */

/* Control bus error generation */
void LsSwitchBerrReq(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Set Error warning limit threshold */
void LsSetEwlReq(
	Message	*m,		/* message buffer */
	BYTE_t	ewl );		/* Error warning limit to set */

/* Control self-test mode */
void LsSwitchStmReq(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Control listen-only mode */
void LsSwitchLomReq(
	Message	*m,		/* message buffer */
	BYTE_t	state );	/* 0: disabled, 1: enabled */

/* Add Remote Node request */
extern void NmAddRemoteNodeReq(
	Message	*,		/* message buffer */
	int,			/* module Id */
	const char *);		/* module name string (strlen==7) */

/* Add Remote CANopen Node request */
extern void NmAddRemoteCOPNodeReq(
	Message	*,		/* message buffer */
	int,			/* module Id */
	const char *,		/* module name string (strlen==7) */
	WORD_t,			/* default guard time */
	int );			/* default node class */

/* Add Remote CANopen Node request with user given guarding IDs */
extern void NmAddRemoteCOPNodeGidReq(
	Message	*,		/* message buffer */
	int,			/* module Id */
	const char *,		/* module name string (strlen==7) */
	WORD_t,			/* default guard time */
	int,			/* default node class */
	WORD_t);		/* user given guarding ID */

/* Remove Remote Node request */
extern void NmRemoveRemoteNodeReq(
	Message	*,		/* message buffer */
	int	,		/* module Id */
	const char *);		/* module name string (strlen==7) */

/* Connect Remote Node request */
extern void NmConnectRemoteNodeReq(
	Message	*,		/* message buffer */
	int	,		/* module Id */
	const char *);		/* module name string (strlen==7) */

/* Prepare Remote Node request */
extern void NmPrepareRemoteNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Start Remote Node request */
extern void NmStartRemoteNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Stop Remote Node request */
extern void NmStopRemoteNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Disconnect Remote Node request */
extern void NmDisconnectRemoteNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Enter preoperational state request */
extern void NmEnterPreOperationalStateReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Reset node request */
extern void NmResetNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

void NmStartGuardRemoteCOPNodeReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Reset communication request */
extern void NmResetCommunicationReq(
	Message	*,		/* message buffer */
	int	);		/* node Id */

/* Identify Remote Nodes request */
extern void NmIdentifyRemoteNodesReq(
	Message	*,		/* message buffer */
	int	,		/* min. node Id in range */
	int	);		/* max. node Id in range */

/* Get NMT Remote Nodes request */
extern void NmGetRemNodes(
	Message	*,		/* message buffer */
	int	);		/* node Id (0 to get all) */

/* Configure NMT master behaviour */
extern void NmConfig(
        Message *m,             /* message buffer */
        WORD_t  timeout,        /* service timeout (in milliseconds) */
        WORD_t  gt,             /* guard time (in milliseconds) */
        int     lf,             /* life time factor */
        WORD_t  flags           /* configuration flags */
);


/* Start Nmt configuration download request */
extern void NmConfigDownloadStartReq(
	Message		*m,	/* message buffer */
	int		nid,	/* node ID of slave node to dwonload to */
	int		size,	/* size of whole configuration data chunk */
	int		dblk,	/* size of configuration data in this message */
	int		more,	/* set to one if other datablocks will follow */
	const void	*data	/* data block */
);

/* Start subsequesnt Nmt configuration download continue response */
extern void NmConfigDownloadContRes(
	Message		*m,	/* message buffer */
	int		nid,	/* node ID of slave node to dwonload to */
	int		dblk,	/* size of configuration data in this message */
	int		more,	/* set to one if other datablocks will follow */
	const void	*data	/* data block */
);

/* NMT master configuration transfer abort request */
extern void NmConfigAbort(
	Message *m,			/* message buffer */
	int 	nid,			/* node id of peer node */
	int 	failureReason,		/* failure reason */
	int	applSpecificReason);	/* appl. specific failure reason */

extern void NmConfigVerify(
	Message *m,		/* message buffer */
	int 	nid,		/* remote node of interest */
	int	checksum);	/* checksum */

/* Create COB Definition request */
extern void DmCreateCobDefReq(
	Message	*,		/* message buffer */
	WORD_t 	,		/* low COB-ID */
	WORD_t 	,		/* high COB-ID */
	int	);		/* min. inhibit time */

/* Delete COB Definition request */
extern void DmDeleteCobDefReq(
	Message	*,		/* message buffer */
	WORD_t 	,		/* low COB-ID */
	WORD_t 	);		/* high COB-ID */

/* Create Predefinition request */
extern void DmCreatePreDefReq(
	Message	*,		/* message buffer */
	WORD_t,			/* COB-ID */
	const char *);		/* COB-Name string (strlen==14) */

/* Delete Predefinition request */
extern void DmDeletePreDefReq(
	Message	*,		/* message buffer */
	const char *);		/* COB-Name string (strlen==14) */

/* Get COB Database request */
extern void DmGetCobDbReq(
	Message	*);		/* message buffer */

void DmConfig(
	Message	*m,		/* message buffer */
	WORD_t	timeout,	/* service timeout (in milliseconds) */
	WORD_t	flags		/* configuration flags */
);

void CmsPatchHostId(
	Message *m,		/* message buffer */
	WORD_t	hid		/* ID to be assigned to object */
);

/* Define a Domain */
extern void CmsDomain(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* muxlen ? ? */
	int			/* priority */
);

/* Define a multipexed Domain */
extern void CmsMultiplexedDomain(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* muxlen ? ? */
	int			/* priority */
);

/* Define a multipexed Domain for CANopen SDO */
extern void CmsMultiplexedDomainSDO(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* muxlen ? ? */
	int	,		/* priority */
	WORD_t,			/* Tx_SDO_ID as seen by node */
	WORD_t,			/* Rx_SDO ID as seen by node */
	BYTE_t			/* dbt_cap - DBT capibility of CANopen node */
);

/* Change a multipexed Domain's ID for CANopen SDO */
extern void CmsMultiplexedDomainChangeIdSDO(
	Message	*,		/* message buffer */
	WORD_t	,		/* object id */
	WORD_t,			/* Tx_SDO_ID as seen by node */
	WORD_t			/* Rx_SDO ID as seen by node */
);

/* Define a Basic Variable */
extern void CmsBasicVariable(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	WORD_t	,		/* CMS_READ_ONLY,... (see: cms.h) */
	int	,		/* length of data type */
	int	);		/* priority */

/* Define a Basic Variable with predefined CobId */
extern void CmsBasicVariablePredefCobId(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	WORD_t	,		/* CMS_READ_ONLY,... (see: cms.h) */
	int	,		/* length of data type */
	int	,		/* priority */
	WORD_t	,
	WORD_t	,
	BYTE_t	);

/* Define a Multiplexed Variable */
extern void CmsMultiplexedVariable(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	WORD_t	,		/* CMS_READ_ONLY,... (see: cms.h) */
	int	,		/* length of data type */
	int	);		/* priority */

/* Define an Uncontrolled Event */
extern void CmsUncontrolledEvent(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* length of data type */
	int	);		/* priority */

/* Define a Controlled Event */
extern void CmsControlledEvent(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* length of data type */
	int	);		/* priority */

/* Define a Stored Event */
extern void CmsStoredEvent(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* length of data type */
	int	);		/* priority */

/* Define a Stored Event with predefined CobId */
extern void CmsStoredEventPredefCobId(
	Message	*,		/* message buffer */
	const char *,		/* CMS name string (strlen==13) */
	WORD_t	,		/* CMS_CLIENT, CMS_SERVER (see: cms.h) */
	int	,		/* length of data type */
	int	,		/* priority */
	WORD_t	,
	WORD_t	,
	BYTE_t	);

void ConfigDomainObject(
	Message *m,		/* message buffer */
	WORD_t	cid,		/* object id */
	WORD_t  mux,		/* multiplexor */
	WORD_t  flags		/* parameter flags */
);

void ConfigCmsObject(
	Message *m,		/* message buffer */
	WORD_t	cid,		/* object id */
	WORD_t  mux,		/* multiplexor */
	WORD_t  param		/* parameter */
);

void CmsConfig(
	Message *m,		/* message buffer */
	WORD_t  flags		/* parameter flags */
);

/* Update Variable in server of a Read-Only Variable */
extern void CmsUpdateVariable(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* length of data type */
	const BYTE_t *);	/* value matching data type */

/* Write Variable request by client of a Basic Variable */
extern void CmsWriteVariable(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* length of data type */
	const BYTE_t *);	/* value matching data type */

/* Read Variable request by client of a Basic Variable */
extern void CmsReadVariable(
	Message	*,		/* message buffer */
	WORD_t 	);		/* Id of CMS Variable */

/* Write Multiplexed Variable request by client of a Basic Variable */
extern void CmsWriteMultiplexedVariable(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* length of data type */
	int	,		/* multiplexor */
	const BYTE_t *);	/* value matching data type */

/* Read Mulitplexed Variable request by client of a Basic Variable */
extern void CmsReadMultiplexedVariable(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	);		/* multiplexor */

void CmsWriteVariableRes(
	Message *m,		/* message buffer */
	WORD_t 	cid,		/* Id of CMS Variable */
	int	dlen,		/* length of data type */
	int	stat,		/* operation status */
	const BYTE_t *value	/* value matching data type */
);

void CmsReadVariableRes(
	Message *m,		/* message buffer */
	WORD_t 	cid,		/* Id of CMS Variable */
	int	dlen,		/* length of data type */
	int	stat,		/* operation status */
	const BYTE_t *value	/* value matching data type */
);


/* Notify Event to clients */
extern void CmsNotifyEvent(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* length of data type */
	const BYTE_t *);	/* value matching data type */

/* Store and Notify Eventy to clients */
extern void CmsStoreEvent(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* length of data type */
	const BYTE_t *,		/* value matching data type */
	int	);		/* !=0 for immediate notification */

/* Read Event */
extern void CmsReadEvent(
	Message	*,		/* message buffer */
	WORD_t 	);		/* Id of CMS Variable */

/* Set Event Control State Request */
extern void CmsSetControlEventReq(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	);		/* !=0 to enable event in server */

/* Set Event Control State Response */
extern void CmsSetControlEventRes(
	Message	*,		/* message buffer */
	WORD_t 	,		/* Id of CMS Variable */
	int	,		/* !=0 to indicate error */
	int	,		/* !=0 to enable event in server */
	const BYTE_t	*	/* reason in case of error */
);


extern void CmsDomDownloadStart(
        Message *,
        WORD_t  ,  		/* Id of CMS Domain */
        int     ,		/* option flags */
        int     ,		/* domain multiplexor */
        int     ,		/* whole data length */
        int     ,		/* data length of current message */
        const void    *		/* data block */
);

extern void CmsDomDownloadCont(
	Message *,
	WORD_t  ,		/* Id of CMS Domain */
	int	,		/* option flags */
	int	,		/* data length of current message */
	const void *		/* data block */
);

extern void CmsDomUploadStart(
        Message *m,
        WORD_t  cid,            /* Id of CMS Domain */
        int     mux             /* Multiplexor type */
);

void CmsDomUpload(
        Message *m,
        WORD_t  cid,            /* Id of CMS Domain */
        int     cs,             /* option flags */
        int     size,           /* whole data length */
        int     dblk,           /* data length of current message */
        const void    *data	/* data block */
);

void CmsDomUploadCont(
	Message *m,
	WORD_t  cid,            /* Id of CMS Domain */
	int	cs,		/* option flags */
	int	dblk,		/* data length of current message */
	const void *data	/* data block */
);

void CmsDomainAck(
	Message *m,
	WORD_t	cid
);

void CmsDomainRcptAck(
	Message *m,
	WORD_t	cid
);

void CmsDomAbort(
	Message *m,
	WORD_t  cid,            /* Id of CMS Domain */
	const void	*data	/* data block pointer or zero */
);

#endif /* MITOP & MITOP_CAL */

#ifdef MITOP_FORMAT

/* Format the contents of messages for display. */
#if MITOP & MITOP_MGR
extern void format_modulmgr(char *, int, const Message *);
#endif /* MITOP & MITOP_MGR */
#if MITOP & MITOP_ICANOS
extern void format_icanos(char *, int, const Message *);
#endif /* MITOP & MITOP_ICANOS */
#if MITOP & MITOP_CAL
extern void format_nmts(char *, int, const Message *);
extern void format_dbts(char *, int, const Message *);
extern void format_lmts(char *, int, const Message *);
extern void format_nmtm(char *, int, const Message *);
extern void format_dbtm(char *, int, const Message *);
extern void format_cmsdef(char *, int, const Message *);
extern void format_cmsdata(char *, int, const Message *);

/* Formatted output */
extern void printf_cob_db(FILE *, int, const BYTE_t *);
extern void printf_nmtm(FILE *, int, const BYTE_t *);
#endif /* MITOP & MITOP_CAL */

#endif /* MITOP_FORMAT */


void Cust1Init(Message *, int);
void Cust1StartTimestampPeriod(Message *, WORD_t);

void COPStartPeriodical(Message *, BYTE_t, WORD_t);
void COPStartSingleShot(Message *, BYTE_t);
void COPMapHiResTimestamp(Message *, WORD_t);

/* 
 * CANopen process picture interface
 */
extern void COPDeleteAllPDOMappingEntries(Message *, WORD_t);
extern void COPAddPDOMappingEntry(Message	*, WORD_t, WORD_t, BYTE_t, BYTE_t, LWORD_t);
extern void COPAddPDOLength(Message *, WORD_t, BYTE_t);
extern void COPAddPDOTransmType(Message *, WORD_t, BYTE_t);
extern void COPAddPDOInhibTime(Message *, WORD_t, WORD_t);
extern void COPControlPDO(Message *, WORD_t, BYTE_t);
extern void COPAdjustCobIdPDO(Message *, WORD_t, LWORD_t);
extern void COPSetSyncToHostSending(Message	*, BYTE_t);
extern void COPTriggerTPDO(Message	*, WORD_t);
extern void COPTriggerRPDO(Message	*, WORD_t);

#ifdef __cplusplus
}
#endif

#endif /* mitop_DEFINED */
