#!/bin/bash

# this script has been written to run on La Palma on the machine daq
#   i.e. paths are only working on this machine

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting "$0

logfile=$runlogpath"/RsyncAuxLP-"$datetime".log"
date >> $logfile

# check if /daq is mounted on data
if ! mount | grep data >> $logfile 2>&1
then 
   printprocesslog "ERROR /data is not mounted on daq => please mount it"
   echo `date`": /data is not mounted on daq => please mount it"
   finish
fi

# check if paths are available
if ! ls /data/aux >/dev/null 2>&1
then 
   printprocesslog "ERROR /data/aux is not available."
   finish
fi
if ! ls /loc_data/aux >/dev/null 2>&1
then 
   printprocesslog "ERROR /loc_data/aux is not available."
   finish
fi

# do the rsync for the last 6 days
dirs=( `ssh newdaq "find /loc_data/aux/ -mindepth 3 -type d | sort | tail -6"` )

# do rsync for auxfiles in these directories
for dir in ${dirs[@]}
do 
   echo "" >> $logfile 2>&1
   # directory on daq
   if ! [ -d $dir ]
   then 
      mkdir -pv $dir >> $logfile 2>&1
   fi
   # directory on data
   dirdata=`echo $dir | sed -e 's/loc_//'`
   if ! [ -d $dirdata ]
   then 
      mkdir -pv $dirdata >> $logfile 2>&1
   fi
   printprocesslog "INFO processing files in "$dir >> $logfile 2>&1
   echo `date`": processing files in "$dir >> $logfile 2>&1
   
   # get current hour
   hour=`date +%k`
   # define bwlimit for rsync depending on the time: from 19-7h reduced bwlimit for rsync
   if [ $hour -le 6 ] || [ $hour -ge 19 ]
   then
      # limit bw for rsync to 20 MB/s during night
      bwlimit="--bwlimit=20000"
      bwlimit="--bwlimit=5000"
      printprocesslog "INFO rsync files with "$bwlimit >> $logfile 2>&1
      echo "rsync files with "$bwlimit >> $logfile 2>&1
   else 
      # no bw limit during day
      bwlimit=""
      printprocesslog "INFO rsync files without bwlimit" >> $logfile 2>&1
      echo "rsync files without bwlimit" >> $logfile 2>&1
   fi

   # rsync from newdaq to daq
   if ! /usr/bin/rsync -avxHP -T $rsynctempdir $bwlimit newdaq:$dir/ $dir >> $logfile 2>&1
   then
      printprocesslog "WARN problem rsyncing auxiliary files in "$dir" from newdaq to daq"
      echo `date`": problem rsyncing auxiliary files in "$dir" from newdaq to daq"
   fi
   
   #rsync from daq to data
   if ! /usr/bin/rsync -avxHP -T $rsynctempdir $bwlimit $dir/ $dirdata >> $logfile 2>&1
   then
      printprocesslog "WARN problem rsyncing auxiliary files in "$dir" from daq to data"
      echo `date`": problem rsyncing auxiliary files in "$dir" from daq to data"
   fi
done

