#!/bin/bash

# this script has been written to run on La Palma on the machine data
#   i.e. paths are only working on this machine

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting "$0

logfile=$runlogpath"/RsyncAuxLP-"$datetime".log"
date >> $logfile

# check if /daq is mounted on data
if ! mount | grep daq >> $logfile 2>&1
then 
   printprocesslog "ERROR /daq is not mounted on data => please mount it"
   echo `date`": /daq is not mounted on data => please mount it"
   finish
fi

# check if paths are available
if ! ls /daq/aux >/dev/null 2>&1
then 
   printprocesslog "ERROR /daq/aux is not available."
   finish
fi
if ! ls /loc_data/aux >/dev/null 2>&1
then 
   printprocesslog "ERROR /loc_data/aux is not available."
   finish
fi


# get last 3 nights
dates=( `date +%Y/%m/%d --date="-12hour"` `date +%Y/%m/%d --date="-36hour"` `date +%Y/%m/%d --date="-60hour"` )

# do rsync for rawfiles of these dates
for date in ${dates[@]}
do 
   echo "" >> $logfile 2>&1
   # auxiliary data directories on data 
   auxdirdaq=/daq/aux/$date/ # /loc_data from daq via nfs on data
   if ! [ -d $auxdirdaq ]
   then 
      auxdirdata=/loc_data/aux/$date/
      if ! [ -d $auxdirdata ]
      then 
         printprocesslog "INFO "$auxdirdaq" and "$auxdirdata" not found. " >> $logfile 2>&1
      else
         printprocesslog "WARN "$auxdirdaq" not found, but "$auxdirdata". Data were probably taken on data." >> $logfile 2>&1
      fi
      echo `date`": "$auxdirdaq" not found. " >> $logfile 2>&1
      continue
   fi
   auxdirdata=/loc_data/aux/$date 
   if ! [ -d $auxdirdata ]
   then 
      mkdir -pv $auxdirdata >> $logfile 2>&1
   fi
   printprocesslog "INFO processing files in "$auxdirdaq >> $logfile 2>&1
   echo `date`": processing files in "$auxdirdaq >> $logfile 2>&1
   
   # get current hour
   hour=`date +%k`
   # define bwlimit for rsync depending on the time: from 19-7h reduced bwlimit for rsync
   if [ $hour -le 6 ] || [ $hour -ge 19 ]
   then
      # limit bw for rsync to 20 MB/s during night
      bwlimit="--bwlimit=20000"
      printprocesslog "INFO rsync data with "$bwlimit >> $logfile 2>&1
      echo "rsync data with "$bwlimit >> $logfile 2>&1
   else 
      # no bw limit during day
      printprocesslog "INFO rsync data without bwlimit" >> $logfile 2>&1
      echo "rsync data without bwlimit" >> $logfile 2>&1
   fi
   
   #rsync from daq to data
   #if ! /usr/bin/rsync -avxHPu $bwlimit $auxdirdaq $auxdirdata >> $logfile 2>&1 # in case of adding checksum, only update files with rsync
   if ! /usr/bin/rsync -avxHP -T $rsynctempdir $bwlimit $auxdirdaq $auxdirdata >> $logfile 2>&1
   then
      printprocesslog "WARN problem rsyncing auxiliary data for "$date" from daq to data"
      echo `date`": problem rsyncing auxiliary data for "$date" from daq to data"
   fi
done

