#!/bin/bash

# option whether to fill all row or only those where information is missing
# $doupdate might be given as environment variable
if [ "$doupdate" = "" ]
then
   doupdate="yes" # update all entries (needed when new fields have been added)
   doupdate="no" # fill only entries which are not yet existing (default)
fi

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting $0 with option doupdate="$doupdate

logfile=$runlogpath"/FillCurrents-"$datetime".log"
date >> $logfile

# setup to use ftools
source $HEADAS/headas-init.sh

# check if software is available
if ! ls $factpath/fitsdump >/dev/null 2>&1
then 
   printprocesslog "ERROR "$factpath"/fitsdump is not available."
   finish
fi

# get dates
if [ "$certaindate" != "" ]
then
   checkstring=`echo $certaindate | grep -E -o '^20[0-9][0-9]\/[01][0-9]\/[0-3][0-9]$'`
   if [ "$checkstring" = "" ]
   then
      echo "Please give the variable certaindate in the correct format (YYYY/MM/DD)"
      finish
   fi
   getdates $certaindate
else
   # get all night
   #getdates "all"
   # get last 6 nights
   getdates 6
fi

printprocesslog "INFO processing the following night(s): "${dates[@]}
echo  `date`": processing the following night(s): "${dates[@]} >> $logfile 2>&1

cd $mars

# do filling of aux data 
for date in ${dates[@]}
do 
   auxdir=$auxdata/$date
   runnumber=`echo $date | sed -e 's/\///g'`
   
   #if [ $runnumber -lt 20130301 ]
   #then
   #   continue
   #fi
   
   # get file numbers from DB
   #   but only for not-corrupted files
   query="SELECT fRunID from RunInfo WHERE fNight="$runnumber" AND fFitsFileErrors=0 "
   if [ "$doupdate" = "no" ]
   then
      query=$query" AND ISNULL(fCurrentsMedMean) "
   fi
   printprocesslog "DEBUG get filenumbers from DB: QUERY: "$query
   filenumbers=( `sendquery $query` )
   # proceed only if there are files available
   if [ ${#filenumbers} -eq 0 ]
   then
      printprocesslog "INFO No files found in the DB for night "$date
      continue
   fi

   # check if aux files are available from that night
   if ! [ -d $auxdir ]
   then
      printprocesslog "INFO no data available in "$auxdir
      continue
   else
      printprocesslog "INFO processing files in "$auxdir
   fi

   biasvoltagefile=$auxdir/$runnumber.BIAS_CONTROL_VOLTAGE.fits
   if ! [ -e $biasvoltagefile ]
   then 
      printprocesslog "WARN "$biasvoltagefile" not found."
      continue
   else
      biasnumerrors=`fverify $biasvoltagefile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $biasnumerrors -gt 0 ]
      then 
         printprocesslog "WARN for $biasvoltagefile fverify returned "$biasnumerrors" error(s)."
      fi
   fi

   biascurrentfile=$auxdir/$runnumber.BIAS_CONTROL_CURRENT.fits
   if ! [ -e $biascurrentfile ]
   then 
      printprocesslog "WARN "$biascurrentfile" not found."
      continue
   else
      biascurrnumerrors=`fverify $biascurrentfile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $biascurrnumerrors -gt 0 ]
      then 
         printprocesslog "WARN for $biascurrentfile fverify returned "$biascurrnumerrors" error(s)."
      fi
   fi

   # this file is needed for the calibration of the currents
   feedbackcalfile=$auxdir/$runnumber.FEEDBACK_CALIBRATION.fits
   if ! [ -e $feedbackcalfile ]
   then 
      printprocesslog "WARN "$feedbackcalfile" not found."
      continue
   else
      feedbacknumerrors=`fverify $feedbackcalfile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $feedbacknumerrors -gt 0 ]
      then 
         printprocesslog "WARN for $feedbackcalfile fverify returned "$feedbacknumerrors" error(s)."
      fi
   fi

   calcurrentsfile=$auxdir/$runnumber.FEEDBACK_CALIBRATED_CURRENTS.fits
   if ! [ -e $calcurrentsfile ]
   then 
      calcurrentsfile=/gpfs/scratch/fact/calibrated_currents/$runnumber.CALIBRATED_CURRENTS.fits
      if ! [ -e $calcurrentsfile ]
      then 
         printprocesslog "INFO run calibrate.C for night "$runnumber >> $logfile 2>&1
         root -q -b -l fact/processing/calibrate.C\($runnumber\)
      fi
   fi
   printprocesslog "INFO using calibrated currents from file "$calcurrentsfile

   #calcurrentsfile=$auxdir/$runnumber.FEEDBACK_CALIBRATED_CURRENTS.fits
   #calcurrentsfile=/scratch_nfs/calibrated_currents/$runnumber.CALIBRATED_CURRENTS.fits
   #calcurrentsfile=/gpfs/scratch/fact/calibrated_currents/$runnumber.CALIBRATED_CURRENTS.fits
   if ! [ -e $calcurrentsfile ]
   then 
      printprocesslog "WARN "$calcurrentsfile" not found."
      continue
   else
      calnumerrors=`fverify $calcurrentsfile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $calnumerrors -gt 0 ]
      then 
         printprocesslog "WARN for $calcurrentsfile fverify returned "$calnumerrors" error(s)."
      fi
   fi
   
   # fill auxiliary information for files
   for filenum in ${filenumbers[@]}
   do
      printprocesslog "INFO processing file number "$runnumber"_"`printf %03d $filenum`
      echo  `date`": processing file number "$runnumber"_"`printf %03d $filenum` >> $logfile 2>&1
      # get information from rawfile
      rawfile=$ziprawdata/$date/$runnumber"_"`printf %03d $filenum`.fits.fz
      if ! [ -e $rawfile ]
      then 
         printprocesslog "ERROR: "$rawfile" not found."
         continue
      fi
      runtype=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep RUNTYPE | grep -E -o "['][a-z-]+[']" | sed -e "s/'//g"`
      mjdrefraw=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'MJDREF' | grep -E -o '[0-9]{5}'`
      tstarti=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTI' | grep -E -o '[0-9]{5}'`
      tstartf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTF' | grep -E -o '0[.][0-9]+'`
      tstopi=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPI' | grep -E -o '[0-9]{5}'`
      tstopf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPF' | grep -E -o '0[.][0-9]+'`
      if [ "$tstarti" == "" ] || [ "$tstopi" == "" ] || [ "$tstartf" == "" ] || [ "$tstopf" == "" ]
      then 
         printprocesslog "WARN: "$rawfile": one of the following keywords is empty or 0: TSTARTI TSTARTF TSTOPI TSTOPF "
         continue
      fi
      # assuming that at least TSTARTI and TSTOPI are consistent
      #echo $rawfile
      #echo $tstarti
      #echo $tstopi
      #echo $tstartf
      #echo $tstopf
      if [ $tstarti -gt 30000 ]
      then 
         tstart=`echo " $tstarti + $tstartf - 40587 " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 40587 - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 40587 - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf - 40587 " | bc -l`
      else
         tstart=`echo " $tstarti + $tstartf " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf " | bc -l`
      fi

      # build query to update runinfo in DB
      query="UPDATE RunInfo SET "

      # get information from fsc: T[31]
      if [ -e $calcurrentsfile ] && [ $calnumerrors -eq 0 ]
      then 
         #root -q -b -l fact/curavg.C\("\"$calcurrentsfile\""\,$tstart\,$tstop\) # | grep "result" | grep -E -o '[0-9]+[.]?[0-9]*'
         #currents=( `root -q -b -l fact/curavg.C\("\"$calcurrentsfile\""\,$tstart\,$tstop\) | grep "result" | grep -E -o '[0-9]+[.]?[0-9]*'` )
         currents=( `root -q -b -l fact/processing/currents.C\("\"$calcurrentsfile\""\,$tstart\,$tstop\) | grep "result" | grep -E -o '[-]?[0-9]+[.]?[0-9]*'` )
         if [ "${currents[0]}" == "" ]
         then 
            query=$query"fCurrentsMedMean=NULL"
         else
            query=$query"fCurrentsMedMean="${currents[0]}
         fi
         if [ "${currents[1]}" == "" ]
         then 
            query=$query", fCurrentsMedRms=NULL"
         else
            query=$query", fCurrentsMedRms="${currents[1]}
         fi
         if [ "${currents[2]}" == "" ]
         then 
            query=$query", fCurrentsDevMean=NULL"
         else
            query=$query", fCurrentsDevMean="${currents[2]}
         fi
         if [ "${currents[3]}" == "" ]
         then 
            query=$query", fCurrentsDevRms=NULL"
         else
            query=$query", fCurrentsDevRms="${currents[3]}
         fi
         if [ "${currents[4]}" == "" ]
         then 
            query=$query", fCurrentsMedMeanBeg=NULL"
         else
            query=$query", fCurrentsMedMeanBeg="${currents[4]}
         fi
         if [ "${currents[5]}" == "" ]
         then 
            query=$query", fCurrentsMedMeanEnd=NULL"
         else
            query=$query", fCurrentsMedMeanEnd="${currents[5]}
         fi
         if [ "${currents[6]}" == "" ]
         then
            query=$query", fCurrentsDiffToPrediction=NULL"
         else
            query=$query", fCurrentsDiffToPrediction="${currents[6]}
         fi
         if [ "${currents[7]}" == "" ]
         then
            query=$query", fCurrentsRelDiffToPrediction=NULL"
         else
            query=$query", fCurrentsRelDiffToPrediction="${currents[7]}
         fi
         if [ "${currents[8]}" == "" ]
         then
            query=$query", fCurrentsLineRms=NULL"
         else
            query=$query", fCurrentsLineRms="${currents[8]}
         fi
         if [ "${currents[9]}" == "" ]
         then
            query=$query", fCurrentsRelLineRms=NULL"
         else
            query=$query", fCurrentsRelLineRms="${currents[9]}
         fi
      else
         query=$query" fCurrentsMedMean=NULL"
         query=$query", fCurrentsMedRms=NULL"
         query=$query", fCurrentsDevMean=NULL"
         query=$query", fCurrentsDevRms=NULL"
         query=$query", fCurrentsMedMeanBeg=NULL"
         query=$query", fCurrentsMedMeanEnd=NULL"
         query=$query", fCurrentsDiffToPrediction=NULL"
         query=$query", fCurrentsRelDiffToPrediction=NULL"
         query=$query", fCurrentsLineRms=NULL"
         query=$query", fCurrentsRelLineRms=NULL"
      fi
      
      # add where condition
      query=$query" WHERE fNight="$runnumber" AND fRunID="$filenum
      
      #echo $query
      # send query to DB
      sendquery >/dev/null
   done
done

finish


