#!/bin/bash

# option whether to fill all row or only those where information is missing
# $doupdate might be given as environment variable
if [ "$doupdate" = "" ]
then
   doupdate="yes" # update all entries (needed when new fields have been added)
   doupdate="no" # fill only entries which are not yet existing (default)
fi

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting $0 with option doupdate="$doupdate

logfile=$runlogpath"/EffectiveOn-"$datetime".log"
date >> $logfile

# setup to use ftools
source $HEADAS/headas-init.sh

# check if software is available
if ! ls $factpath/fitsdump >/dev/null 2>&1
then 
   printprocesslog "ERROR "$factpath"/fitsdump is not available."
   finish
fi

# get dates
if [ "$certaindate" != "" ]
then
   getdates $certaindate
else
   # get all night
   #getdates "all"
   # get last 6 nights
   getdates 6
fi


printprocesslog "INFO processing the following night(s): "${dates[@]}
echo  `date`": processing the following night(s): "${dates[@]} >> $logfile 2>&1

cd $mars

# do filling of aux data 
for date in ${dates[@]}
do 
   auxdir=$auxdata/$date
   runnumber=`echo $date | sed -e 's/\///g'`

   # get file numbers from DB
   #   but only for not-corrupted files
   query="SELECT fRunID from RunInfo WHERE fNight="$runnumber" AND fFitsFileErrors=0 "
   if [ "$doupdate" = "no" ]
   then
      query=$query" AND ISNULL(fEffectiveOn) "
   fi
   printprocesslog "DEBUG get filenumbers from DB: QUERY: "$query
   filenumbers=( `sendquery $query` )
   # proceed only if there are files available
   if [ ${#filenumbers} -eq 0 ]
   then
      printprocesslog "INFO No files found in the DB for night "$date
      continue
   fi

   # check if aux files are available from that night
   if ! [ -d $auxdir ]
   then
      printprocesslog "INFO no data available in "$auxdir
      continue
   else
      printprocesslog "INFO processing files in "$auxdir
   fi
   
   ftmcontrolfile=$auxdir/$runnumber.FTM_CONTROL_TRIGGER_RATES.fits
   if ! [ -e $ftmcontrolfile ]
   then 
      printprocesslog "WARN "$ftmcontrolfile" not found."
      continue
   else
      ftmnumerrors=`fverify $ftmcontrolfile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $ftmnumerrors -gt 0 ]
      then 
         printprocesslog "WARN for $ftmcontrolfile fverify returned "$ftmnumerrors" error(s)."
      fi
   fi

   # fill auxiliary information for files
   for filenum in ${filenumbers[@]}
   do
      printprocesslog "INFO processing file number "$runnumber"_"`printf %03d $filenum`
      echo  `date`": processing file number "$runnumber"_"`printf %03d $filenum` >> $logfile 2>&1
      # get information from rawfile
      rawfile=$ziprawdata/$date/$runnumber"_"`printf %03d $filenum`.fits.fz
      if ! [ -e $rawfile ]
      then 
         printprocesslog "ERROR: "$rawfile" not found."
         continue
      fi
      runtype=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep RUNTYPE | grep -E -o "['][a-z-]+[']" | sed -e "s/'//g"`
      mjdrefraw=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'MJDREF' | grep -E -o '[0-9]{5}'`
      tstarti=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTI' | grep -E -o '[0-9]{5}'`
      tstartf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTF' | grep -E -o '0[.][0-9]+'`
      tstopi=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPI' | grep -E -o '[0-9]{5}'`
      tstopf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPF' | grep -E -o '0[.][0-9]+'`
      if [ "$tstarti" == "" ] || [ "$tstopi" == "" ] || [ "$tstartf" == "" ] || [ "$tstopf" == "" ]
      then 
         printprocesslog "WARN: "$rawfile": one of the following keywords is empty or 0: TSTARTI TSTARTF TSTOPI TSTOPF "
         continue
      fi
      # assuming that at least TSTARTI and TSTOPI are consistent
      if [ $tstarti -gt 30000 ]
      then 
         tstart=`echo " $tstarti + $tstartf - 40587 " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 40587 - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 40587 - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf - 40587 " | bc -l`
      else
         tstart=`echo " $tstarti + $tstartf " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf " | bc -l`
      fi

      # build query to update runinfo in DB
      query="UPDATE RunInfo SET "

      # get information from fsc: T[31]
      if [ -e $ftmcontrolfile ] && [ $ftmnumerrors -eq 0 ]
      then 
         effectiveon=( `root -q -b -l fact/processing/ontime.C\("\"$ftmcontrolfile\""\,$tstart\,$tstop\) | grep "result" | grep -E -o '[0-9]+[.]?[0-9]*'` )
         if [ "${effectiveon[0]}" == "" ]
         then 
            query=$query"fEffectiveOn=NULL"
         else
            query=$query"fEffectiveOn="${effectiveon[0]}
         fi
         if [ "${effectiveon[1]}" == "" ]
         then 
            query=$query", fEffectiveOnRms=NULL"
         else
            query=$query", fEffectiveOnRms="${effectiveon[1]}
         fi
         if [ "${effectiveon[2]}" == "" ]
         then
            query=$query", fOnTime=NULL"
         else
            query=$query", fOnTime="${effectiveon[2]}
         fi
         if [ "${effectiveon[3]}" == "" ]
         then
            query=$query", fTriggerRateTimeOver100=NULL"
         else
            query=$query", fTriggerRateTimeOver100="${effectiveon[3]}
         fi
         if [ "${effectiveon[4]}" == "" ]
         then
            query=$query", fTriggerRateTimeOver125=NULL"
         else
            query=$query", fTriggerRateTimeOver125="${effectiveon[4]}
         fi
         if [ "${effectiveon[5]}" == "" ]
         then
            query=$query", fTriggerRateTimeOver150=NULL"
         else
            query=$query", fTriggerRateTimeOver150="${effectiveon[5]}
         fi
         if [ "${effectiveon[6]}" == "" ]
         then
            query=$query", fTriggerRateTimeOver175=NULL"
         else
            query=$query", fTriggerRateTimeOver175="${effectiveon[6]}
         fi
         if [ "${effectiveon[7]}" == "" ]
         then
            query=$query", fTriggerRateRms=NULL"
         else
            query=$query", fTriggerRateRms="${effectiveon[7]}
         fi
      else
         query=$query" fEffectiveOn=NULL"
         query=$query", fEffectiveOnRms=NULL"
         query=$query", fOnTime=NULL"
         query=$query", fTriggerRateTimeOver100=NULL"
         query=$query", fTriggerRateTimeOver125=NULL"
         query=$query", fTriggerRateTimeOver150=NULL"
         query=$query", fTriggerRateTimeOver175=NULL"
         query=$query", fTriggerRateRms=NULL"

      fi
      
      # add where condition
      query=$query" WHERE fNight="$runnumber" AND fRunID="$filenum

      #send query to DB
      sendquery >/dev/null

   done
done

finish


