#!/bin/bash

# option whether to fill all row or only those where information is missing
# $doupdate might be given as environment variable
if [ "$doupdate" = "" ]
then
   doupdate="yes" # update all entries (needed when new fields have been added)
   doupdate="no" # fill only entries which are not yet existing (default)
fi

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting $0 with option doupdate="$doupdate

logfile=$runlogpath"/FillDrsTemp-"$datetime".log"
date >> $logfile

# setup to use ftools
source $HEADAS/headas-init.sh

# check if software is available
if ! ls $factpath/fitsdump >/dev/null 2>&1
then 
   printprocesslog "ERROR "$factpath"/fitsdump is not available."
   finish
fi

# get dates
if [ "$certaindate" != "" ]
then
   getdates $certaindate
else
   # get all night
   #getdates "all"
   # get last 6 nights
   getdates 6
fi


printprocesslog "INFO processing the following night(s): "${dates[@]}
echo  `date`": processing the following night(s): "${dates[@]} >> $logfile 2>&1

cd $mars

# do filling of aux data 
for date in ${dates[@]}
do 
   auxdir=$auxdata/$date
   runnumber=`echo $date | sed -e 's/\///g'`
   if [ $runnumber -lt 20120328 ]
   then
      continue
   fi

   # check if aux files are available from that night
   if ! [ -d $auxdir ]
   then
      printprocesslog "INFO no data available in "$auxdir
      continue
   else
      printprocesslog "INFO processing files in "$auxdir
   fi

   # get file numbers from DB
   #   but only for not-corrupted files
   query="SELECT fRunID from RunInfo WHERE fNight="$runnumber" AND fFitsFileErrors=0 "
   if [ "$doupdate" = "no" ]
   then
      query=$query" AND ISNULL(fDrsTempMinMean) "
   fi
   printprocesslog "DEBUG get filenumbers from DB: QUERY: "$query
   filenumbers=( `sendquery $query` )
   # proceed only if there are files available
   if [ ${#filenumbers} -eq 0 ]
   then
      printprocesslog "INFO No files found in the DB for night "$date
      continue
   fi

   drstempfile=$auxdir/$runnumber.FAD_CONTROL_TEMPERATURE.fits
   if ! [ -e $drstempfile ]
   then 
      printprocesslog "WARN "$drstempfile" not found."
      #echo "WARN "$drstempfile" not found."
   else
      tempnumerrors=`fverify $drstempfile 2>/dev/null | grep -o '[0-9][ ]error(s)'  | grep -E -o '[0-9]'`
      if [ $tempnumerrors -gt 0 ]
      then 
         printprocesslog "WARN for "$drstempfile" fverify returned "$tempnumerrors" error(s)."
      fi
   fi

   # fill auxiliary information for files
   for filenum in ${filenumbers[@]}
   do
      printprocesslog "INFO processing file number "$runnumber"_"`printf %03d $filenum`
      echo  `date`": processing file number "$runnumber"_"`printf %03d $filenum` >> $logfile 2>&1
      # get information from rawfile
      rawfile=$ziprawdata/$date/$runnumber"_"`printf %03d $filenum`.fits.fz
      if ! [ -e $rawfile ]
      then 
         printprocesslog "ERROR: "$rawfile" not found."
         continue
      fi
      #checkfitsfile=`fverify $rawfile  2>/dev/null | grep '0 error(s)'`
      #if [ "$checkfitsfile" == "" ]
      #then
      #   numfitserrors=1
      #   printprocesslog "WARN: "$rawfile" probably corrupted."
      #   continue
      #fi
      runtype=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep RUNTYPE | grep -E -o "['][a-z-]+[']" | sed -e "s/'//g"`
      mjdrefraw=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'MJDREF' | grep -E -o '[0-9]{5}'`
      tstarti=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTI' | grep -E -o '[0-9]{5}'`
      tstartf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTF' | grep -E -o '0[.][0-9]+'`
      tstopi=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPI' | grep -E -o '[0-9]{5}'`
      tstopf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPF' | grep -E -o '0[.][0-9]+'`
      if [ "$tstarti" == "" ] || [ "$tstopi" == "" ] || [ "$tstartf" == "" ] || [ "$tstopf" == "" ]
      then 
         printprocesslog "WARN: "$rawfile": one of the following keywords is empty or 0: TSTARTI TSTARTF TSTOPI TSTOPF "
         continue
      fi
      # assuming that at least TSTARTI and TSTOPI are consistent
      if [ $tstarti -gt 30000 ]
      then 
         tstart=`echo " $tstarti + $tstartf - 40587 " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 40587 - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 40587 - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf - 40587 " | bc -l`
      else
         tstart=`echo " $tstarti + $tstartf " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf " | bc -l`
      fi

      # build query to update runinfo in DB
      query="UPDATE RunInfo SET "

      # get information from fsc: T[31]
      if [ -e $drstempfile ] && [ $tempnumerrors -eq 0 ]
      then 
         drstemps=( `root -q -b -l fact/processing/drstemp.C\("\"$drstempfile\""\,$tstart\,$tstop\) | grep "result" | grep -E -o '[0-9]+[.]?[0-9]*'` )
         if [ "${drstemps[0]}" == "" ]
         then 
            query=$query"fDrsTempMinMean=NULL"
         else
            query=$query"fDrsTempMinMean="${drstemps[0]}
         fi
         if [ "${drstemps[1]}" == "" ]
         then 
            query=$query", fDrsTempMaxMean=NULL"
         else
            query=$query", fDrsTempMaxMean="${drstemps[1]}
         fi
         if [ "${drstemps[2]}" == "" ]
         then 
            query=$query", fDrsTempMinRmsMean=NULL"
         else
            query=$query", fDrsTempMinRmsMean="${drstemps[2]}
         fi
         if [ "${drstemps[3]}" == "" ]
         then 
            query=$query", fDrsTempMaxRmsMean=NULL"
         else
            query=$query", fDrsTempMaxRmsMean="${drstemps[3]}
         fi
      else
         query=$query" fDrsTempMinMean=NULL"
         query=$query", fDrsTempMaxMean=NULL"
         query=$query", fDrsTempMinRmsMean=NULL"
         query=$query", fDrsTempMaxRmsMean=NULL"
      fi
      #fCameraTempMeanRms: mean of rms of single sensors
      
      # add where condition
      query=$query" WHERE fNight="$runnumber" AND fRunID="$filenum
      
      # send query to DB
      sendquery >/dev/null
   done
done

finish


