#!/bin/bash

# this script has been written to run on La Palma on the machine daq
#   i.e. paths are only working on this machine

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting "$0

logfile=$runlogpath"/RsyncAuxLP-"$datetime".log"
date >> $logfile

# check if /daq is mounted on data
if ! mount | grep data >> $logfile 2>&1
then 
   printprocesslog "ERROR /data is not mounted on daq => please mount it"
   echo `date`": /data is not mounted on daq => please mount it"
   finish
fi

# check if paths are available
paths="/newdaq/aux /data1/aux /data2/aux /scratch/aux"
for path in ${paths[@]}
do 
   if ! ls $path >/dev/null 2>&1
   then 
      printprocesslog "ERROR "$path" is not available."
      finish
   fi
done

# first get some settings
# get current hour
hour=`date +%k`
# define bwlimit for rsync depending on the time: from 19-7h reduced bwlimit for rsync
if [ $hour -le 6 ] || [ $hour -ge 19 ]
then
   # limit bw for rsync to 20 MB/s during night
   bwlimit="--bwlimit=20000"
   bwlimit="--bwlimit=5000"
   printprocesslog "INFO rsync files with "$bwlimit 
   echo "rsync files with "$bwlimit >> $logfile 2>&1
   searchwindow="-cmin -840"
   printprocesslog "INFO treat the last 14 hours."
   echo "INFO treat the last 14 hours."  >> $logfile 2>&1
else 
   # no bw limit during day
   bwlimit=""
   printprocesslog "INFO rsync files without bwlimit"
   echo "rsync files without bwlimit" >> $logfile 2>&1
   searchwindow="-ctime -6"
   printprocesslog "INFO treat the last 14 hours."
   echo "INFO treat the last 14 hours."  >> $logfile 2>&1
fi

# find directories to be rsynced
dirs=( `ssh newdaq "find /loc_data/aux/ -mindepth 3 $searchwindow -type d | sort "` )

# do rsync for auxfiles in these directories
for dir in ${dirs[@]}
do 
   echo "" >> $logfile 2>&1
   printprocesslog "INFO processing files in "$dir 
   echo `date`": processing files in "$dir >> $logfile 2>&1
   #echo `date`": processing files in "$dir
   
   # first take care of SSD /scratch
   scratch=`echo $dir | sed -e 's/loc_data/scratch/'`
   if ! [ -d $scratch ]
   then 
      mkdir -pv $scratch >> $logfile 2>&1
   fi
   # rsync from newdaq to newdata
   if ! /usr/bin/rsync -avu -T $rsynctempdir $bwlimit newdaq:$dir/ $scratch >> $logfile 2>&1
   then
      printprocesslog "CONNECTION problem rsyncing auxiliary files in "$dir" from newdaq to newdata"
      echo `date`": problem rsyncing auxiliary files in "$dir" from newdaq to newdata" >> $logfile 2>&1
   fi
   
   # now make copies on newdata
   data1=`echo $dir | sed -e 's/loc_data/data1/'`
   if ! [ -d $data1 ]
   then 
      mkdir -pv $data1 >> $logfile 2>&1
   fi
   data2=`echo $dir | sed -e 's/loc_data/data2/'`
   if ! [ -d $data2 ]
   then 
      mkdir -pv $data2 >> $logfile 2>&1
   fi
   # distribute to several disk on newdata
   if ! /usr/bin/rsync -avu -T $rsynctempdir $bwlimit $scratch/ $data1 >> $logfile 2>&1
   then
      printprocesslog "WARN problem rsyncing auxiliary files in "$dir" from scratch to data1"
      echo `date`": problem rsyncing auxiliary files in "$dir" from scratch to data1" >> $logfile 2>&1
   fi
   if ! /usr/bin/rsync -avu -T $rsynctempdir $bwlimit $scratch/ $data2 >> $logfile 2>&1
   then
      printprocesslog "WARN problem rsyncing auxiliary files in "$dir" from scratch to data2"
      echo `date`": problem rsyncing auxiliary files in "$dir" from scratch to data2" >> $logfile 2>&1
   fi
done

finish
