#include "MainWindow.h"

#include <iostream>
#include <sstream>

#include <QTimer>

#include "src/Dim.h"

#include "DockWindow.h"
#include "HtmlDelegate.h"
#include "CheckBoxDelegate.h"

using namespace std;

MainWindow::MainWindow(QWidget *p) : QMainWindow(p)
{
    // setupUi MUST be called before the DimNetwork is initilized
    // In this way it can be ensured that nothing from the
    // DimNetwork arrives before all graphical elements are
    // initialized. This is a simple but very powerfull trick.
    setupUi(this);

    // Now here we can do further setup which should be done
    // before the gui is finally displayed.
    fDimCmdServers->setItemDelegate(new CheckBoxDelegate);
    fDimCmdCommands->setItemDelegate(new CheckBoxDelegate);
    fDimCmdDescription->setItemDelegate(new HtmlDelegate);

    fDimSvcServers->setItemDelegate(new CheckBoxDelegate);
    fDimSvcServices->setItemDelegate(new CheckBoxDelegate);
    fDimSvcDescription->setItemDelegate(new HtmlDelegate);

    // Set a default string to be displayed in a the status bar at startup
    fStatusBar->showMessage(PACKAGE_STRING"   |   "PACKAGE_URL"   |   report bugs to <"PACKAGE_BUGREPORT">");

    // Initialize the 40 FTU Leds as a copy of the prototype LED
    fFtuLED[0] = fFtuLEDPrototype;
    fFtuLED[0]->setToolTip("Crate 0, Board 0, Index 0");

    for (int i=1; i<40; i++)
    {
        QPushButton *b = new QPushButton(static_cast<QWidget*>(fFtuLEDPrototype->parent()));

        b->setEnabled(fFtuLEDPrototype->isEnabled());
        b->setSizePolicy(fFtuLEDPrototype->sizePolicy());
        b->setMaximumSize(fFtuLEDPrototype->maximumSize());
        b->setIcon(fFtuLEDPrototype->icon());
        b->setIconSize(fFtuLEDPrototype->iconSize());
        b->setCheckable(fFtuLEDPrototype->isCheckable());
        b->setFlat(fFtuLEDPrototype->isFlat());

        ostringstream str;
        str << "Crate " << i/10 << ", Board " << i%10 << ", Index " << i;
        b->setToolTip(str.str().c_str());

        fFtuLedLayout->addWidget(b, i/10+1, i%10+1, 1, 1);

        fFtuLED[i] = b;
    }

    for (int i=0; i<40; i++)
    {
        fFtuLED[i]->setObjectName(QString::fromUtf8("fFtuLED")+QString::number(i));
        QObject::connect(fFtuLED[i], SIGNAL(clicked()), this, SLOT(slot_fFtuLED_clicked()));
    }

    // Initialize a timer to update the displayed UTC time
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(slot_TimeUpdate()));
    timer->start(100);
}

void MainWindow::slot_TimeUpdate()
{
    // Used toTUC to support also older Qt versions
    fUTC->setDateTime(QDateTime::currentDateTime().toUTC());
}

void MainWindow::SelectTab(const QString &name)
{
    for (int i=0; i<fTabWidget->count(); i++)
        if (fTabWidget->tabText(i)==name)
        {
            fTabWidget->setCurrentIndex(i);
            break;
        }
}


void MainWindow::on_fShutdown_clicked()
{
    Dim::SendCommand("DIS_DNS/KILL_SERVERS");
}


void MainWindow::on_fShutdownAll_clicked()
{
    Dim::SendCommand("DIS_DNS/KILL_SERVERS");
    Dim::SendCommand("DIS_DNS/EXIT", int(1));
}

void MainWindow::on_fTabWidget_tabCloseRequested(int which)
{
    // To get the correct size we have to switch to this tab
    // An alternative would be to take the size of the current tab
    fTabWidget->setCurrentIndex(which);

    QWidget *w = fTabWidget->currentWidget(); //fTabWidget->widget(which);
    if (!w)
    {
        cout << "Weird... the tab requested to be closed doesn't exist!" << endl;
        return;
    }

    QDockWidget *d = w->findChild<QDockWidget*>();
    if (!d)
    {
        cout << "Sorry, tab requested to be closed contains no QDockWidget!" << endl;
        return;
    }

    new DockWindow(d, fTabWidget->tabText(which));
    fTabWidget->removeTab(which);

    if (fTabWidget->count()==1)
        fTabWidget->setTabsClosable(false);
}

void MainWindow::SetTriggerSequence()
{
    const uint16_t d[3] =
    {
        uint16_t(fTriggerSeqPed->value()),
        uint16_t(fTriggerSeqLPext->value()),
        uint16_t(fTriggerSeqLPint->value())
    };

    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_SEQUENCE", d);
}

void on_fEnableTrigger_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_TRIGGER", b);
}

void on_fEnableExt1_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_EXT1", b);
}

void on_fEnableExt2_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_EXT2", b);
}

void on_fEnableVeto_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_VETO", b);
}

void MainWindow::on_fPhysicsCoincidence_valueChanged(int v)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_COINCIDENCE", v);
}

void MainWindow::on_fPhysicsWindow_valueChanged(int v)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_WINDOW", v/4-2);
}

void MainWindow::on_fCalibCoincidence_valueChanged(int v)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_CALIBRATION_COINCIDENCE", v);
}

void MainWindow::on_fCalibWindow_valueChanged(int v)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_CALIBRATION_WINDOW", v/4-2);
}

void MainWindow::on_fThresholdVal_valueChanged(int v)
{
    fThresholdVolt->setValue(2500./4095*v);

    const int32_t d[2] = { fThresholdIdx->value(), v };

    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_THRESHOLD", d);
}

void MainWindow::on_fTriggerInterval_valueChanged(int val)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_INTERVAL", val);
}

void MainWindow::on_fTriggerDelay_valueChanged(int val)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_DELAY", val/4-2);
}

void MainWindow::on_fTimeMarkerDelay_valueChanged(int val)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_TIME_MARKER_DELAY", val/4-2);
}

void MainWindow::on_fDeadTime_valueChanged(int val)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_DEAD_TIME", val/4-2);
}

void MainWindow::on_fPrescalingVal_valueChanged(int val)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/SET_PRESCALING", val);
}

void MainWindow::on_fPixelEnable_stateChanged(int b)
{
    cout << "PIXEL: " << fPixelIdx->value() << " " << b << endl;
    if (!fInHandler)
        Dim::SendCommand(b==Qt::Unchecked ?
                         "FTM_CONTROL/DISABLE_PIXEL" : "FTM_CONTROL/ENABLE_PIXEL",
                         uint16_t(fPixelIdx->value()));
}

void MainWindow::on_fEnableTrigger_stateChanged(int b)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/ENABLE_TRIGGER", b==Qt::Checked);
}

void MainWindow::on_fEnableExt1_stateChanged(int b)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/ENABLE_EXT1", b==Qt::Checked);
}

void MainWindow::on_fEnableExt2_stateChanged(int b)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/ENABLE_EXT2", b==Qt::Checked);
}

void MainWindow::on_fEnableClockCond_stateChanged(int b)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/ENABLE_CLOCK_CONDITIONER", b==Qt::Checked);
}

void MainWindow::on_fEnableVeto_stateChanged(int b)
{
    if (!fInHandler)
        Dim::SendCommand("FTM_CONTROL/ENABLE_VETO", b==Qt::Checked);
}

void MainWindow::slot_fFtuLED_clicked()
{
    for (int32_t i=0; i<40; i++)
        if (sender()==fFtuLED[i])
        {
            Dim::SendCommand("FTM_CONTROL/TOGGLE_FTU", i);
            break;
        }
}

void MainWindow::on_fPing_toggled(bool checked)
{
    if (checked)
        Dim::SendCommand("FTM_CONTROL/PING");
}

void MainWindow::on_fChatSend_clicked()
{
    if (Dim::SendCommand("CHAT/MSG", fChatMessage->displayText().constData()))
        fChatMessage->clear();
}

void MainWindow::on_fStatusLoggerLed_clicked()
{
    SelectTab("Logger");
}

void MainWindow::on_fStatusChatLed_clicked()
{
    SelectTab("Chat");
}

void MainWindow::on_fStatusFTMLed_clicked()
{
    SelectTab("Trigger");
}

void MainWindow::on_fStatusFTULed_clicked()
{
    SelectTab("FTUs");
}

void MainWindow::on_fStatusFADLed_clicked()
{
    SelectTab("FAD");
}
