/* ************************************************************************

   This is the analysis query. It returns two columns for all
   signal/background events.

     - `Weight` (Positive for signal (typ. 1),
                 negative for background (typ. -0.2))
     - `LogEnergyEst` logarithm of estimated energy in GeV

   In additon, all columns provided by the 100-clause must be
   returned. (Note that you must not add a comma behind it)

   100| %100:columns
   101| %101:files:: table containing the `FileId`s to analyze.
   102| %102:runinfo:: table with the run info data
   103| %103:events:: table with the image parameters
   104| %104:positions:: table with the source positions in the camera
   105| %105:estimator:: estimator for log10 energy

*************************************************************************** */

WITH Table0 AS
(
   SELECT
      %100:columns  -- this could be removed if we can join events via the same columns (without CorsikaNumResuse)
      Weight,
      Size,
      NumUsedPixels,
      NumIslands,
      Leakage1,
      MeanX,
      MeanY,
      CosDelta,
      SinDelta,
      M3Long,
      SlopeLong,
      Width/Length      AS WdivL,
      PI()*Width*Length AS Area,
      cosa*X - sina*Y   AS PX,
      cosa*Y + sina*X   AS PY
   FROM
      %101:files
   LEFT JOIN
      %102:runinfo USING (FileId)
   LEFT JOIN 
      %103:events USING (FileId)  -- This could be replaced by a user uploaded temporary table
   LEFT JOIN 
      %104:positions USING (FileId, EvtNumber)
   CROSS JOIN 
      Wobble
   WHERE
      NumUsedPixels>5.5
   AND
      NumIslands<3.5
   AND
      Leakage1<0.1
),

Table1 AS
(
   SELECT
      %100:columns
      Weight,
      Size, CosDelta, SinDelta, M3Long, SlopeLong, Leakage1, WdivL,
      MeanX - PX/1.02e0 AS DX,
      MeanY - PY/1.02e0 AS DY
   FROM
      Table0
   WHERE
      Area < LOG10(Size)*898e0 - 1535e0
),

Table2 AS
(
   SELECT
      %100:columns
      Weight,
      Size, CosDelta, SinDelta, DX, DY, M3Long, SlopeLong, Leakage1, WdivL,
      SQRT(DX*DX + DY*DY) AS Norm
   FROM
      Table1
),

Table3 AS
(
   SELECT
      %100:columns
      Weight,
      Size, M3Long, SlopeLong, Leakage1, WdivL, Norm,
      LEAST(GREATEST((CosDelta*DY - SinDelta*DX)/Norm, -1), 1) AS LX,
      SIGN(CosDelta*DX + SinDelta*DY) AS Sign
   FROM
      Table2
),

Table5 AS
(
   SELECT
      %100:columns
      Weight,
      Size, Leakage1, WdivL, LX,
      Norm          *0.0117193246260285378e0 AS Dist,
      M3Long   *Sign*0.0117193246260285378e0 AS M3L,
      SlopeLong*Sign/0.0117193246260285378e0 AS Slope
   FROM
      Table3
),

Table6 AS
(
   SELECT
      %100:columns
      Weight,
      Size, WdivL, Dist, LX, M3L, Slope,
      1.39252e0 + 0.154247e0*Slope + 1.67972e0*(1-1/(1+4.86232e0*Leakage1)) AS Xi
   FROM
      Table5
),

Table7 AS
(
   SELECT
      %100:columns
      Weight,
      Size, Dist, LX,
      IF (M3L<-0.07 OR (Dist-0.5e0)*7.2e0-Slope<0, -Xi, Xi) * (1-WdivL) AS Disp
   FROM
      Table6
)

SELECT
   %100:columns
   Weight,
   (Disp*Disp + Dist*Dist - 2*Disp*Dist*SQRT(1-LX*LX)) AS ThetaSq,
   %105:estimator AS LogEnergyEst
FROM
   Table7
HAVING
   ThetaSq<0.024
