#ifndef FACT_Connection
#define FACT_Connection

#include <deque>
#include <string>
#include <boost/asio.hpp>
#include <boost/function.hpp>
#include <boost/asio/deadline_timer.hpp>

#include "MessageImp.h"

class Connection : public MessageImp, public boost::asio::ip::tcp::socket
{
private:
    MessageImp *fLog;

    std::string fAddress;
    std::string fPort;
protected:
    boost::asio::deadline_timer   fInTimeout;

private:
    boost::asio::deadline_timer   fOutTimeout;
    boost::asio::deadline_timer   fConnectionTimer;
    std::deque<std::vector<char>> fOutQueue; // Shell we directly put ba:buffers into the queue?

    int fConnectionStatus;   // 0=offline, 1=connecting, 2=connected

    std::string fErrConnect;
    std::string fMsgConnect;

protected:
    char fReadBuffer[1000];


public:
    void SetLogStream(MessageImp *log) { fLog = log; }

    // -------- Abbreviations for starting async tasks ---------

    int Write(const Time &t, const char *txt, int qos=kInfo);

    void AsyncRead(boost::asio::mutable_buffers_1 buffers, int type=0);
    void AsyncWrite(boost::asio::mutable_buffers_1 buffers);
    void AsyncWait(boost::asio::deadline_timer &timer, int millisec,
                   void (Connection::*handler)(const boost::system::error_code&));
    void AsyncConnect(boost::asio::ip::tcp::resolver::iterator iterator);

    std::string URL() const { return fAddress + ":" + fPort; }

    void CloseImp(bool restart=true);

    void ConnectImp(const boost::system::error_code& error,
                    boost::asio::ip::tcp::resolver::iterator endpoint_iterator);

public:

    // ------------------------ close --------------------------
    // close from another thread
    void PostClose(bool restart=true);

    // ------------------------ write --------------------------
    void HandleConnectionTimer(const boost::system::error_code &error);
    void HandleWriteTimeout(const boost::system::error_code &error);
    void HandleSentData(const boost::system::error_code& error, size_t);
    void SendMessageImp(const std::vector<char> &msg);
    void PostMessage(std::vector<uint16_t> msg);
    void PostMessage(const std::vector<char> &msg);
    void PostMessage(const void *msg, size_t s=0);
    void PostMessage(const std::string &cmd, size_t s=-1);

    template<std::size_t N>
        void PostMessage(boost::array<uint16_t, N> arr)
    {
        // Convert to network byte order
        for_each(arr.begin(), arr.end(), htons);
        PostMessage(std::vector<uint16_t>(arr.begin(), arr.end()));
    }

    // ------------------------ connect --------------------------

    virtual void ConnectionEstablished() { }

    void StartConnect();

    Connection(boost::asio::io_service& io_service, std::ostream &out);
//    Connection(boost::asio::io_service& io_service, const std::string &addr="localhost", int port=5000);
//    Connection(boost::asio::io_service& io_service, const std::string &addr, const std::string &port);

    void SetEndpoint(const std::string &addr, int port);
    void SetEndpoint(const std::string &addr, const std::string &port);
    void SetEndpoint(const std::string &addr);

    // ------------------------ others --------------------------

    virtual void HandleReceivedData(const boost::system::error_code&, size_t, int = 0) { }
    virtual void HandleReadTimeout(const boost::system::error_code&) { }

    int IsClosed() const { return !is_open(); }

    bool IsConnected()  const { return fConnectionStatus==2; }
    bool IsConnecting() const { return fConnectionStatus==1; }
};

#endif
