#ifndef FACT_LocalControl
#define FACT_LocalControl

#include <ostream>

class StateMachineImp;

// **************************************************************************
/** @class LocalControl

@brief Implements a local control for a StateMachine based on a Readline class

This template implements all functions which overwrite any function from the
Readline class needed for a local control of a state machien. Since
several derivatives of the Readline class implement different kind of
Readline access, this class can be derived by any of them due to its
template argument. However, the normal case will be deriving it from
either Console or Shell.

@tparam T
   The base class for RemoteControl. Either Readlien or a class
   deriving from it. This is usually either Console or Shell.

**/
// **************************************************************************
#include <boost/filesystem.hpp>

#include "WindowLog.h"
#include "StateMachineImp.h"

using namespace std;

template <class T>
class LocalControl : public T
{
private:
    char **Completion(const char *text, int pos, int)
    {
        return pos>0 ? 0 : T::Complete(fStateMachine->GetEventNames(), text);
    }

protected:
    StateMachineImp *fStateMachine;

    std::ostream &lout;

    std::string fName;

    LocalControl(const char *name) : T(name), 
        fStateMachine(0), lout(T::GetStreamIn()),
        fName(boost::filesystem::path(name).filename())
    { }

    //bool PrintGeneralHelp()
    //bool PrintKeyBindings()
    bool PrintCommands()
    {
        lout << endl << kBold << "List of commands:" << endl;
        fStateMachine->PrintListOfEvents(lout);
        //lout << endl;

        return true;
    }

    bool Process(const std::string &str)
    {
        if (T::Process(str))
            return true;

        return !fStateMachine->PostEvent(lout, str);
    }

public:

    void SetReceiver(StateMachineImp &imp) { fStateMachine = &imp; }

    void Run(const char * = 0)
    {
        lout << endl;
        lout << kBlue << kBold << "You are on the " << fName << " terminal of the MCP -" << endl;
        lout << kBlue << kBold << "the Master Control Program." << endl;
        lout << endl;
        lout << kBlue << kBold << "Hello Flynn..." << endl;
        lout << endl;

        T::Run();
    }
};

// **************************************************************************
/** @class LocalConsole

@brief Derives the LocalControl from Control and adds prompt

This is basically a LocalControl, which derives through the template
argument from the Console class. It enhances the functionality of
the local control with a proper updated prompt.

 */
// **************************************************************************
#include "Console.h"
#include "tools.h"

class LocalConsole : public LocalControl<Console>
{
public:
    LocalConsole(const char *name, bool continous=false)
        : LocalControl<Console>(name)
    {
        SetContinous(continous);
    }

    string GetUpdatePrompt() const;
};

// **************************************************************************
/** @class LocalShell

@brief Derives the LocalControl from Shell and adds a colored prompt

This is basically a LocalControl, which derives through the template
argument from the Shell class. It enhances the functionality of
the local control with a proper updated prompt.

 */
// **************************************************************************
#include "Shell.h"

class LocalShell : public LocalControl<Shell>
{
public:
    LocalShell(const char *name, bool = false)
        : LocalControl<Shell>(name) { }

    string GetUpdatePrompt() const;
};

#endif
