#include "MessageDim.h"

#include "tools.h"
#include "Time.h"

using namespace std;

// **************************************************************************
/** @class MessageDimTX

@brief Based on MessageImp, redirects log-output to a Dim service MESSAGE

This is a special DimService which offers SERVER/MESSAGE to the DimNetwork
and redirects output issued via its base-class MessageImp to the Dim
service. The severity of the message is send as qualiy of service of
the service message.

@section Examples

 - A simple and usefull example can be found in \ref log.cc and \ref logtime.cc

**/
// **************************************************************************

// --------------------------------------------------------------------------
//
//! Constructs a DimService with the name SERVER/MESSAGE. And passes the
//! given ostream down to the MessageImp base.
//!
//! @param name
//!    Name of the message server to which we want to subscribe, e.g. DRIVE
//!
//! @param out
//!    ostream passed to MessageImp. It is used to redirect the output to.
//
MessageDimTX::MessageDimTX(const std::string &name, std::ostream &out)
: DimService((name + "/MESSAGE").c_str(), const_cast<char*>("")),
MessageImp(out), fDebug(false)
{
    // This is a message which will never arrive because
    // the time to establish a client-sever connection is
    // too short.
    Message("MessageDimTX started.");
}

// --------------------------------------------------------------------------
//
//!
//
MessageDimTX::~MessageDimTX()
{
    Message("MessageDimTX shutting down..");
}

// --------------------------------------------------------------------------
//
//! First calls MessageImp::Write to output the message tobe transmitted
//! also to a local logging stream. Then the Dim service is updated.
//! If sending of the message failed a message is written to the
//! logging stream stored in MessageImp. It is intentionally not
//! output through Update to make it look different than usual
//! transmitted messages.
//
int MessageDimTX::Write(const Time &t, const string &txt, int qos)
{
    MessageImp::Write(t, txt, qos);

    // We have to use setData to make sure the DimService will
    // hold a local copy of the data.
    setQuality(qos);
    setData(const_cast<char*>(txt.c_str()));
    const int rc = updateService();

    if (rc==0 && fDebug)
        Out() << " !! " << t.GetAsStr() << " - Sending failed!" << endl;

    return rc;
}



// **************************************************************************
/** @class MessageDimRX

@brief Based on MessageImp, subscribes to a MESSAGE service in the Dim network

This is a special DimInfoHandler. It subscribes to a service SERVER/MESSAGE
on the DimNetwork and redirects all received output to its base class
MessageImp view MessageImp::Write. the quality of service received with
each service update is passed as severity.

@section Examples

 - A simple and usefull example can be found in \ref log.cc and \ref logtime.cc

 @todo Maybe it is not a good idea that MessageImp is a base class,
 maybe it should be a reference given in the constructor

**/
// **************************************************************************

// --------------------------------------------------------------------------
//
//! Setup a DimStamedInfo service subscription for SERVER/MESSAGE
//!
//! @param name
//!    the name of the SERVER
//!
//! @param imp
//!    A reference to MessageImo to which messages will be redirected
//
MessageDimRX::MessageDimRX(const std::string &name, MessageImp &imp)
: fMsg(imp),
fDimMessage(Form("%s/MESSAGE", name.c_str()).c_str(), const_cast<char*>(""), this),
fConnected(false)
{
    fMinLogLevel = 0;
}

// --------------------------------------------------------------------------
//
//! If the server has been disconnected write a simple log-message.
//! Skip all received messages which have a severity smaller than
//! fMinLogLevel. Write any other message with MessageImp::Write.
//
void MessageDimRX::infoHandler()
{
    if (getInfo()!=&fDimMessage)
        return;

    // The server is diconnected. Do nothing
    if (getInfo()->getTimestamp()==0 || getInfo()->getSize()==1)
    {
        fConnected=false;
        fMsg.Message("Disconnected.");
        return;
    }

    fConnected=true;

    // skip all messages with a severity smaller than the minimum log level
    if (getInfo()->getQuality()<fMinLogLevel)
        return;

    const string n = getInfo()->getName();

    stringstream msg;
    msg << n.substr(0, n.find_first_of('/')) << ": " << getInfo()->getString();

    // Write the received message to the output
    fMsg.Write(Time(getInfo()->getTimestamp(), getInfo()->getTimestampMillisecs()),
               msg.str().c_str(), getInfo()->getQuality());
}
