#ifndef FACT_EventImp
#define FACT_EventImp

#include <string>
#include <vector>

#include <boost/function.hpp>

#include "Time.h"

class EventImp
{   int              fTargetState;   /// Target state of an event
    std::vector<int> fAllowedStates; /// List of states in which this event is allowed

    /// http://www.boost.org/doc/libs/1_45_0/libs/bind/bind.html
    boost::function<int(const EventImp &)> fFunction;

public:
    /// Constructor. Stores the target state given.
    EventImp(int target=-1) : fTargetState(target) { }
    /// Copy constructor
    EventImp(const EventImp &cmd);

    // Description
    virtual void SetDescription(const std::string &) { }
    virtual std::string GetDescription() const { return ""; }

    // Function handling
    EventImp &AssignFunction(const boost::function<int(const EventImp &)> &func) { fFunction = func; return *this; }
    bool HasFunc() const { return !fFunction.empty(); }
    int ExecFunc() const { return HasFunc() ? fFunction(*this) : -1; }

    // Configuration helper
    EventImp &operator()(const boost::function<int(const EventImp &)> &func) { return AssignFunction(func); }
    EventImp &operator()(const std::string str) { SetDescription(str); return *this; }
    EventImp &operator()(const char *str) { SetDescription(str); return *this; }

    // Print contents
    virtual void Print(std::ostream &out, bool strip=false) const;
    virtual void Print(bool strip=false) const;

    // Handling of the states
    void AddAllowedState(int state);
    void AddAllowedStates(const char *states);

    bool IsStateAllowed(int state) const;

    int  GetTargetState() const { return fTargetState; }

    // virtual function to return the data as stored in the derived classes
    virtual std::string GetName() const   { return ""; }
    virtual std::string GetFormat() const { return ""; }

    virtual const void *GetData() const { return 0; }
    virtual int         GetSize() const { return 0; }

    virtual Time GetTime() const { return Time::None; }

    // Getter for all the data contained (name, format, data and time)
    short             GetShort()  const { return *reinterpret_cast<const short*>(GetData()); }
    unsigned short    GetUShort() const { return *reinterpret_cast<const unsigned short*>(GetData()); }
    int               GetInt() const    { return *reinterpret_cast<const int*>(GetData()); }
    unsigned int      GetUInt() const   { return *reinterpret_cast<const unsigned int*>(GetData()); }
    float             GetFloat() const  { return *reinterpret_cast<const float*>(GetData()); }
    double            GetDouble() const { return *reinterpret_cast<const double*>(GetData()); }
    const char       *GetText() const   { return  reinterpret_cast<const char*>(GetData()); }
    std::string       GetString() const { return std::string(GetText(), GetSize()); }
    std::vector<char> GetVector() const { return std::vector<char>(GetText(), GetText()+GetSize()); }

};

#endif
