#include <iostream>

#include <boost/bind.hpp>

#include "EventImp.h"
#include "WindowLog.h"
#include "Configuration.h"
#include "StateMachineDim.h"

using namespace std;

void SetupConfiguration(Configuration &conf)
{
    const string n = conf.GetName()+".log";

    po::options_description config("Program options");
    config.add_options()
        ("dns",       var<string>("localhost"), "Dim nameserver host name (Overwites DIM_DNS_NODE environment variable)")
        ("log,l",     var<string>(n), "Write log-file")
        ;

    conf.AddEnv("dns", "DIM_DNS_NODE");

    conf.AddOptions(config);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "The ftmctrl controls the FTM (FACT Trigger Master) board.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: ftmctrl [-c type] [OPTIONS]\n"
        "  or:  ftmctrl [OPTIONS]\n"
        "\n"
        "Options:\n"
        "The following describes the available commandline options. "
        "For further details on how command line option are parsed "
        "and in which order which configuration sources are accessed "
        "please refer to the class reference of the Configuration class.";
    cout << endl;

}

void PrintHelp()
{
}

/*
 The first line of the --version information is assumed to be in one
 of the following formats:

   <version>
   <program> <version>
   {GNU,Free} <program> <version>
   <program> ({GNU,Free} <package>) <version>
   <program> - {GNU,Free} <package> <version>

 and separated from any copyright/author details by a blank line.

 Handle multi-line bug reporting sections of the form:

   Report <program> bugs to <addr>
   GNU <package> home page: <url>
   ...
*/
void PrintVersion(const char *name)
{
    cout <<
        name << " - FACT++ 1.0 - Trigger master control\n"
        "\n"
        "Written by Thomas Bretz <thomas.bretz@epfl.ch> et al.\n"
        "\n"
        "Report bugs to Thomas Bretz <thomas.bretz@epfl.ch>\n"
        "FACT++ home page: http://www.xxx.com\n"
        "\n"
        "Copyright (C) 2011 by the FACT Collaboration.\n"
        "This is free software; see the source for copying conditions.\n"
        << endl;
}

class ChatServer : public StateMachineDim
{
private:
    int HandleMsg(const EventImp &evt)
    {
        Message(evt.GetString());
        return GetCurrentState();
    }
public:
    ChatServer(ostream &lout) : StateMachineDim(lout, "CHAT")
    {
        AddConfiguration("MSG", "C")
            (boost::bind(&ChatServer::HandleMsg, this, _1))
            ("|msg[string]:message to be distributed");
    }
};

int main(int argc, const char *argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    SetupConfiguration(conf);

    po::variables_map vm;
    try
    {
        vm = conf.Parse(argc, argv);
    }
#if BOOST_VERSION > 104000
    catch (po::multiple_occurrences &e)
    {
        cout << "Error: " << e.what() << " of '" << e.get_option_name() << "' option." << endl;
        cout << endl;
        return -1;
    }
#endif
    catch (std::exception &e)
    {
        cout << "Error: " << e.what() << endl;
        cout << endl;

        return -1;
    }

    if (conf.HasPrint())
        return -1;

    if (conf.HasVersion())
    {
        PrintVersion(argv[0]);
        return -1;
    }

    if (conf.HasHelp())
    {
        PrintHelp();
        return -1;
    }

    // To allow overwriting of DIM_DNS_NODE set 0 to 1
    setenv("DIM_DNS_NODE", conf.Get<string>("dns").c_str(), 1);


    WindowLog log;

    if (conf.Has("log"))
        if (!log.OpenLogFile(conf.Get<string>("log")))
            cerr << "ERROR - Couldn't open log-file " << conf.Get<string>("log") << ": " << strerror(errno) << endl;

    ChatServer serv(log);
    serv.Run(true);

    return 0;
}

// **************************************************************************
/** @example chatserv.cc

The program is stopped by CTRL-C

*/
// **************************************************************************
