#include <boost/bind.hpp>
#include <boost/array.hpp>
#if BOOST_VERSION < 104400
#if (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 4))
#undef BOOST_HAS_RVALUE_REFS
#endif
#endif
#include <boost/thread.hpp>
#include <boost/asio/error.hpp>
#include <boost/asio/deadline_timer.hpp>

#include "Event.h"
#include "Shell.h"
#include "StateMachineDim.h"
#include "Connection.h"
#include "Configuration.h"
#include "Timers.h"
#include "Console.h"
#include "Converter.h"

#include "tools.h"

#include "LocalControl.h"
#include "HeadersFTM.h"


namespace ba = boost::asio;
namespace bs = boost::system;

using namespace std;
using namespace FTM;

// ------------------------------------------------------------------------

class ConnectionFTM : public Connection
{
    vector<uint16_t> fBuffer;

    bool fHasHeader;
    int  fState;

    bool fIsVerbose;
    bool fIsDynamicOut;
    bool fIsHexOutput;

    // --verbose
    // --hex-out
    // --dynamic-out
    // --load-file
    // --leds
    // --trigger-interval
    // --physcis-coincidence
    // --calib-coincidence
    // --physcis-window
    // --physcis-window
    // --trigger-delay
    // --time-marker-delay
    // --dead-time
    // --clock-conditioner-r0
    // --clock-conditioner-r1
    // --clock-conditioner-r8
    // --clock-conditioner-r9
    // --clock-conditioner-r11
    // --clock-conditioner-r13
    // --clock-conditioner-r14
    // --clock-conditioner-r15
    // ...

protected:
    map<uint16_t, int> fCounter;

    FTM::Header      fHeader;
    FTM::FtuList     fFtuList;
    FTM::StaticData  fStaticData;
    FTM::DynamicData fDynamicData;
    FTM::Error       fError;

    virtual void UpdateFirstHeader()
    {
        // FIXME: Message() ?
        Out() << endl << kBold << "First header received:" << endl;
        Out() << fHeader;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fHeader, sizeof(fHeader), 16) << endl;
    }

    virtual void UpdateHeader()
    {
        // emit service with trigger counter from header
        if (!fIsVerbose)
            return;

        if (fHeader.fType==kDynamicData && !fIsDynamicOut)
            return;

        Out() << endl << kBold << "Header received:" << endl;
        Out() << fHeader;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fHeader, sizeof(fHeader), 16) << endl;
    }

    virtual void UpdateFtuList()
    {
        if (!fIsVerbose)
            return;

        Out() << endl << kBold << "FtuList received:" << endl;
        Out() << fFtuList;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fFtuList, 16) << endl;
    }

    virtual void UpdateStaticData()
    {
        if (!fIsVerbose)
            return;

        Out() << endl << kBold << "Static data received:" << endl;
        Out() << fStaticData;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fStaticData, 16) << endl;
    }

    virtual void UpdateDynamicData()
    {
        if (!fIsDynamicOut)
            return;

        Out() << endl << kBold << "Dynamic data received:" << endl;
        Out() << fDynamicData;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fDynamicData, 16) << endl;
    }

    virtual void UpdateError()
    {
        if (!fIsVerbose)
            return;

        Out() << endl << kRed << "Error received:" << endl;
        Out() << fError;
        if (fIsHexOutput)
            Out() << Converter::GetHex<uint16_t>(fError, 16) << endl;
    }

    virtual void UpdateCounter()
    {
        if (!fIsVerbose)
            return;

        if (!fIsDynamicOut)
            return;

        Out() << "Received: ";
        Out() << "H=" << fCounter[kHeader] << "  ";
        Out() << "S=" << fCounter[kStaticData] << "  ";
        Out() << "D=" << fCounter[kDynamicData] << "  ";
        Out() << "F=" << fCounter[kFtuList] << "  ";
        Out() << "E=" << fCounter[kErrorList] << "  ";
        Out() << "R=" << fCounter[kRegister] << endl;
    }

    void CheckConsistency()
    {
        if (fStaticData.IsEnabled(StaticData::kPedestal) != (fStaticData.GetSequencePed()  >0) ||
            fStaticData.IsEnabled(StaticData::kLPint)    != (fStaticData.GetSequenceLPint()>0) ||
            fStaticData.IsEnabled(StaticData::kLPext)    != (fStaticData.GetSequenceLPext()>0))
        {
            Warn("GeneralSettings not consistent with trigger sequence.");
        }

        const uint16_t ref = fStaticData[0].fPrescaling;
        for (int i=1; i<40; i++)
        {
            if (fStaticData[i].fPrescaling != ref)
            {
                Warn("Prescaling not consistent for all boards.");
                break;
            }
        }

    }

private:
    void HandleReceivedData(const bs::error_code& err, size_t bytes_received, int /*type*/)
    {
        cout << "Data received " << err << " " << bytes_received << endl;

        // Do not schedule a new read if the connection failed.
        if (bytes_received==0 || err)
        {
            if (err==ba::error::eof)
                Warn("Connection closed by remote host (FTM).");

            // 107: Transport endpoint is not connected (bs::error_code(107, bs::system_category))
            // 125: Operation canceled
            if (err && err!=ba::error::eof &&                     // Connection closed by remote host
                err!=ba::error::basic_errors::not_connected &&    // Connection closed by remote host
                err!=ba::error::basic_errors::operation_aborted)  // Connection closed by us
            {
                stringstream str;
                str << "Reading from " << URL() << ": " << err.message() << " (" << err << ")";// << endl;
                Error(str);
            }
            PostClose(err!=ba::error::basic_errors::operation_aborted);
            return;
        }

        // If we have not yet received a header we expect one now
        // This could be moved to a HandleReceivedHeader function
        if (!fHasHeader)
        {
            if (bytes_received!=sizeof(FTM::Header))
            {
                stringstream str;
                str << "Excepted " << sizeof(FTM::Header) << " bytes (FTM::Header) but received " << bytes_received << ".";
                Error(str);
                PostClose(false);
                return;
            }

            fHeader = fBuffer;

            // Check the data integrity
            if (fHeader.fDelimiter!=kDelimiterStart)
            {
                stringstream str;
                str << "Invalid header received: start delimiter wrong, received " << hex << fHeader.fDelimiter << " expected " << kDelimiterStart << ".";
                Error(str);
                PostClose(false);
                return;
            }

            fHasHeader = true;

            // Convert FTM state into FtmCtrl state
            switch (fHeader.fState)
            {
            case FTM::kFtmIdle:
            case FTM::kFtmConfig:
                fState = FTM::kIdle;
                break;

            case FTM::kFtmCalib:
            case FTM::kFtmRunning:
                fState = FTM::kTakingData;
                break;
            }

            if (++fCounter[kHeader]==1)
                UpdateFirstHeader();

            UpdateCounter();
            UpdateHeader();

            // Start reading of data
            switch (fHeader.fType)
            {
            case kStaticData:
            case kDynamicData:
            case kFtuList:
            case kRegister:
            case kErrorList:
                // This is not very efficient because the space is reallocated
                // maybe we can check if the capacity of the std::vector
                // is ever decreased. If not, everythign is fine.
                fBuffer.resize(fHeader.fDataSize);
                AsyncRead(ba::buffer(fBuffer));
                AsyncWait(fInTimeout, 50, &Connection::HandleReadTimeout);
                return;

            default:
                stringstream str;
                str << "Unknonw type " << fHeader.fType << " in received header." << endl;
                Error(str);
                PostClose(false);
                return;
            }

            return;
        }

        // Check the data integrity (check end delimiter)
        if (ntohs(fBuffer.back())!=FTM::kDelimiterEnd)
        {
            stringstream str;
            str << "Invalid data received: end delimiter wrong, received ";
            str << hex << ntohs(fBuffer.back()) << " expected " << kDelimiterEnd << ".";
            Error(str);
            PostClose(false);
            return;
        }

        // Remove end delimiter
        fBuffer.pop_back();

        try
        {
            // If we have already received a header this is the data now
            // This could be moved to a HandleReceivedData function

            fCounter[fHeader.fType]++;
            UpdateCounter();

            cout << "TYPE=" << fHeader.fType << endl;

            switch (fHeader.fType)
            {
            case kFtuList:
                fFtuList = fBuffer;
                UpdateFtuList();
                break;

            case kStaticData:
                fStaticData = fBuffer;

                if (fCounter[kStaticData]==1)
                    CheckConsistency();

                UpdateStaticData();
                break;

            case kDynamicData:
                fDynamicData = fBuffer;
                UpdateDynamicData();
                break;

            case kRegister:
                if (fIsVerbose)
                {
                    Out() << endl << kBold << "Register received: " << endl;
                    Out() << "Addr:  " << ntohs(fBuffer[0]) << endl;
                    Out() << "Value: " << ntohs(fBuffer[1]) << endl;
                }
                break;

            case kErrorList:
                fError = fBuffer;
                UpdateError();
                break;

            default:
                stringstream str;
                str << "Unknonw type " << fHeader.fType << " in header." << endl;
                Error(str);
                PostClose(false);
                return;
            }
        }
        catch (const logic_error &e)
        {
            stringstream str;
            str << "Exception converting buffer into data structure: " << e.what();
            Error(str);
            PostClose(false);
            return;
        }

        fInTimeout.cancel();

        fHeader.clear();
        fHasHeader = false;
        fBuffer.resize(sizeof(FTM::Header)/2);
        AsyncRead(ba::buffer(fBuffer));
    }

    // This is called when a connection was established
    void ConnectionEstablished()
    {
        fState = FTM::kConnected;
        fCounter.clear();

        fHeader.clear();
        fHasHeader = false;
        fBuffer.resize(sizeof(FTM::Header)/2);
        AsyncRead(ba::buffer(fBuffer));

        // Get a header and configdata!
        CmdReqStatDat();

        // get the DNA of the FTUs
        CmdPing();
    }

    void HandleReadTimeout(const bs::error_code &error)
    {
        if (error && error!=ba::error::basic_errors::operation_aborted)
        {
            stringstream str;
            str << "Read timeout of " << URL() << ": " << error.message() << " (" << error << ")";// << endl;
            Error(str);

            PostClose();
            return;

        }

        if (!is_open())
        {
            // For example: Here we could schedule a new accept if we
            // would not want to allow two connections at the same time.
            return;
        }

        // Check whether the deadline has passed. We compare the deadline
        // against the current time since a new asynchronous operation
        // may have moved the deadline before this actor had a chance
        // to run.
        if (fInTimeout.expires_at() > ba::deadline_timer::traits_type::now())
            return;

        Error("Timeout reading data from "+URL());

        PostClose();
    }


    template<size_t N>
    void PostCmd(boost::array<uint16_t, N> dat, uint16_t u1=0, uint16_t u2=0, uint16_t u3=0, uint16_t u4=0)
    {
        boost::array<uint16_t, 5> cmd = {{ '@', u1, u2, u3, u4 }};

        stringstream msg;
        msg << "Sending command:" << hex;
        msg << " 0x" << setw(4) << setfill('0') << cmd[0];
        msg << " 0x" << setw(4) << setfill('0') << u1;
        msg << " 0x" << setw(4) << setfill('0') << u2;
        msg << " 0x" << setw(4) << setfill('0') << u3;
        msg << " 0x" << setw(4) << setfill('0') << u4;
        msg << " (+" << dec << dat.size() << " words)";
        Message(msg);

        vector<uint16_t> out(cmd.size()+dat.size());

        transform(cmd.begin(), cmd.end(), out.begin(), htons);
        transform(dat.begin(), dat.end(), out.begin()+cmd.size(), htons);

        PostMessage(out);
    }

    void PostCmd(vector<uint16_t> dat, uint16_t u1=0, uint16_t u2=0, uint16_t u3=0, uint16_t u4=0)
    {
        boost::array<uint16_t, 5> cmd = {{ '@', u1, u2, u3, u4 }};

        stringstream msg;
        msg << "Sending command:" << hex;
        msg << " 0x" << setw(4) << setfill('0') << cmd[0];
        msg << " 0x" << setw(4) << setfill('0') << u1;
        msg << " 0x" << setw(4) << setfill('0') << u2;
        msg << " 0x" << setw(4) << setfill('0') << u3;
        msg << " 0x" << setw(4) << setfill('0') << u4;
        msg << " (+" << dec << dat.size() << " words)";
        Message(msg);

        vector<uint16_t> out(cmd.size()+dat.size());

        transform(cmd.begin(), cmd.end(), out.begin(), htons);
        copy(dat.begin(), dat.end(), out.begin()+cmd.size());

        PostMessage(out);
    }

    void PostCmd(uint16_t u1=0, uint16_t u2=0, uint16_t u3=0, uint16_t u4=0)
    {
        PostCmd(boost::array<uint16_t, 0>(), u1, u2, u3, u4);
    }
public:

    static const uint16_t kMaxAddr;

public:
    ConnectionFTM(ba::io_service& ioservice, MessageImp &imp) : Connection(ioservice, imp()),
        fIsVerbose(true), fIsDynamicOut(true), fIsHexOutput(true)
    {
        SetLogStream(&imp);
    }

    void CmdToggleLed()
    {
        PostCmd(kCmdToggleLed);
    }

    void CmdPing()
    {
        PostCmd(kCmdPing);
    }

    void CmdReqDynDat()
    {
        PostCmd(kCmdRead, kReadDynamicData);
    }

    void CmdReqStatDat()
    {
        PostCmd(kCmdRead, kReadStaticData);
    }

    void CmdSendStatDat()
    {
        PostCmd(fStaticData.HtoN(), kCmdWrite, kWriteStaticData);

        // Request the changed configuration to ensure the
        // change is distributed in the network
        //CmdReqStatDat();
    }

    void CmdStartRun()
    {
        PostCmd(kCmdStartRun, kStartRun);

        // Update state information by requesting a new header
        CmdGetRegister(0);
    }

    void CmdStopRun()
    {
        PostCmd(kCmdStopRun);

        // Update state information by requesting a new header
        CmdGetRegister(0);
    }

    void CmdTakeNevents(uint32_t n)
    {
        const boost::array<uint16_t, 2> data = {{ uint16_t(n>>16), uint16_t(n&0xffff) }};
        PostCmd(data, kCmdStartRun, kTakeNevents);

        // Update state information by requesting a new header
        CmdGetRegister(0);
    }

    bool CmdSetRegister(uint16_t addr, uint16_t val)
    {
        if (addr>kMaxAddr)
            return false;

        const boost::array<uint16_t, 2> data = {{ addr, val }};
        PostCmd(data, kCmdWrite, kWriteRegister);

        // Request the changed configuration to ensure the
        // change is distributed in the network
        CmdReqStatDat();

        return true;
    }

    bool CmdGetRegister(uint16_t addr)
    {
        if (addr>kMaxAddr)
            return false;

        const boost::array<uint16_t, 1> data = {{ addr }};
        PostCmd(data, kCmdRead, kReadRegister);

        return true;
    }

    bool CmdDisableReports(bool b)
    {
        PostCmd(kCmdDisableReports, b ? uint16_t(0) : uint16_t(1));
        return true;
    }

    void SetVerbose(bool b)
    {
        fIsVerbose = b;
    }

    void SetHexOutput(bool b)
    {
        fIsHexOutput = b;
    }

    void SetDynamicOut(bool b)
    {
        fIsDynamicOut = b;
    }

    bool LoadStaticData(string name)
    {
        if (name.rfind(".bin")!=name.length()-5)
            name += ".bin";

        ifstream fin(name);
        if (!fin)
            return false;

        FTM::StaticData data;

        fin.read(reinterpret_cast<char*>(&data), sizeof(FTM::StaticData));

        if (fin.gcount()<streamsize(sizeof(FTM::StaticData)))
            return false;

        if (fin.fail() || fin.eof())
            return false;

        if (fin.peek()!=-1)
            return false;

        fStaticData = data;

        for (int i=0; i<100; i++)
            CmdSendStatDat();

        return true;
    }

    bool SaveStaticData(string name) const
    {
        if (name.rfind(".bin")!=name.length()-5)
            name += ".bin";

        ofstream fout(name);
        if (!fout)
            return false;

        fout.write(reinterpret_cast<const char*>(&fStaticData), sizeof(FTM::StaticData));

        return !fout.bad();
    }

    bool SetThreshold(int32_t patch, int32_t value)
    {
        if (patch>159)
            return false;

        if (value<0 || value>0xffff)
            return false;

        if (patch<0)
        {
            bool ident = true;
            for (int i=0; i<160; i++)
                if (fStaticData[i/4].fDAC[patch%4] != value)
                {
                    ident = false;
                    break;
                }

            if (ident)
                return true;

            for (int i=0; i<160; i++)
                fStaticData[i/4].fDAC[i%4] = value;
        }
        else
        {
            if (fStaticData[patch/4].fDAC[patch%4] == value)
                return true;

            fStaticData[patch/4].fDAC[patch%4] = value;
        }

        // Maybe move to a "COMMIT" command?
        CmdSendStatDat();

        return true;
    }

    bool SetPrescaling(uint32_t value)
    {
        if (value>0xffff)
            return false;


        bool ident = true;
        for (int i=0; i<40; i++)
            if (fStaticData[i].fPrescaling != value)
            {
                ident = false;
                break;
            }

        if (ident)
            return true;

        for (int i=0; i<40; i++)
            fStaticData[i].fPrescaling = value;

        // Maybe move to a "COMMIT" command?
        CmdSendStatDat();

        return true;
    }

    bool EnableFTU(int32_t board, bool enable)
    {
        if (board>39)
            return false;

        if (board<0)
        {
            if (enable)
                fStaticData.EnableAllFTU();
            else
                fStaticData.DisableAllFTU();
        }
        else
        {
            if (enable)
                fStaticData.EnableFTU(board);
            else
                fStaticData.DisableFTU(board);

        }

        // Maybe move to a "COMMIT" command?
        CmdSendStatDat();

        return true;
    }

    bool ToggleFTU(uint32_t board)
    {
        if (board>39)
            return false;

        fStaticData.ToggleFTU(board);

        // Maybe move to a "COMMIT" command?
        CmdSendStatDat();

        return true;
    }

    bool SetVal(uint16_t *dest, uint32_t val, uint32_t max)
    {
        if (val>max)
            return false;

        if (*dest==val)
            return true;

        *dest = val;

        CmdSendStatDat();

        return true;
    }

    bool SetTriggerInterval(uint32_t val)
    {
        return SetVal(&fStaticData.fTriggerInterval, val, StaticData::kMaxTriggerInterval);
    }

    bool SetTriggerDelay(uint32_t val)
    {
        return SetVal(&fStaticData.fDelayTrigger, val, StaticData::kMaxDelayTrigger);
    }

    bool SetTimeMarkerDelay(uint32_t val)
    {
        return SetVal(&fStaticData.fDelayTimeMarker, val, StaticData::kMaxDelayTimeMarker);
    }

    bool SetDeadTime(uint32_t val)
    {
        return SetVal(&fStaticData.fDeadTime, val, StaticData::kMaxDeadTime);
    }

    void Enable(FTM::StaticData::GeneralSettings type, bool enable)
    {
        fStaticData.Enable(type, enable);
    }

    bool SetTriggerSeq(const uint8_t d[3])
    {
        const uint16_t oldset = fStaticData.fGeneralSettings;
        const uint16_t oldseq = fStaticData.fTriggerSequence;

        fStaticData.Enable(StaticData::kPedestal, d[0]>0);
        fStaticData.Enable(StaticData::kLPext,    d[1]>0);
        fStaticData.Enable(StaticData::kLPint,    d[2]>0);

        if (d[0]>StaticData::kMaxSequence ||
            d[1]>StaticData::kMaxSequence ||
            d[2]>StaticData::kMaxSequence)
            return false;

        fStaticData.fTriggerSequence =
            (d[0]<<10) | (d[1]<<5) || d[2];

        if (oldseq!=fStaticData.fTriggerSequence || oldset!=fStaticData.fGeneralSettings)
            CmdSendStatDat();

        return true;
    }

    bool SetTriggerCoincidence(uint16_t n, uint16_t win)
    {
        if (n==0 || n>StaticData::kMaxCoincidence || win>StaticData::kMaxWindow)
            return false;

        if (n  ==fStaticData.fCoincidencePhysics &&
            win==fStaticData.fWindowPhysics)
            return true;

        fStaticData.fCoincidencePhysics = n;
        fStaticData.fWindowPhysics      = win;

        CmdSendStatDat();

        return true;
    }

    bool SetCalibCoincidence(uint16_t n, uint16_t win)
    {
        if (n==0 || n>StaticData::kMaxCoincidence || win>StaticData::kMaxWindow)
            return false;

        if (n  ==fStaticData.fCoincidenceCalib &&
            win==fStaticData.fWindowCalib)
            return true;

        fStaticData.fCoincidenceCalib = n;
        fStaticData.fWindowCalib      = win;

        CmdSendStatDat();

        return true;
    }

    int GetState() const { return IsConnected() ? fState : (int)FTM::kDisconnected; }
};

const uint16_t ConnectionFTM::kMaxAddr = 0xfff;

// ------------------------------------------------------------------------

#include "DimDescriptionService.h"

class ConnectionDimFTM : public ConnectionFTM
{
private:

    DimDescribedService fDimPassport;
    DimDescribedService fDimTriggerCounter;
    DimDescribedService fDimError;
    DimDescribedService fDimFtuList;
    DimDescribedService fDimStaticData;
    DimDescribedService fDimDynamicData;
    DimDescribedService fDimCounter;

    template<class T>
        void Update(DimDescribedService &svc, const T &data) const
    {
        //cout << "Update: " << svc.getName() << " (" << sizeof(T) << ")" << endl;
        svc.setData(const_cast<T*>(&data), sizeof(T));
        svc.updateService();
    }

    void UpdateFirstHeader()
    {
        ConnectionFTM::UpdateFirstHeader();

        const DimPassport data(fHeader);
        Update(fDimPassport, data);
    }

    void UpdateHeader()
    {
        ConnectionFTM::UpdateHeader();

        const DimTriggerCounter data(fHeader);
        Update(fDimTriggerCounter, data);
    }

    void UpdateFtuList()
    {
        ConnectionFTM::UpdateFtuList();

        const DimFtuList data(fHeader, fFtuList);
        Update(fDimFtuList, data);
    }

    void UpdateStaticData()
    {
        ConnectionFTM::UpdateStaticData();

        const DimStaticData data(fHeader, fStaticData);
        Update(fDimStaticData, data);
    }

    void UpdateDynamicData()
    {
        ConnectionFTM::UpdateDynamicData();

        const DimDynamicData data(fHeader, fDynamicData);
        Update(fDimDynamicData, data);
    }

    void UpdateError()
    {
        ConnectionFTM::UpdateError();

        const DimError data(fHeader, fError);
        Update(fDimError, data);
    }

    void UpdateCounter()
    {
        ConnectionFTM::UpdateCounter();

        const uint32_t counter[6] =
        {
            fCounter[kHeader],
            fCounter[kStaticData],
            fCounter[kDynamicData],
            fCounter[kFtuList],
            fCounter[kErrorList],
            fCounter[kRegister],
        };

        Update(fDimCounter, counter);
    }

public:
    ConnectionDimFTM(ba::io_service& ioservice, MessageImp &imp) :
        ConnectionFTM(ioservice, imp),
        fDimPassport      ("FTM_CONTROL/PASSPORT",        "X:1;S:1",      NULL, 0, ""),
        fDimTriggerCounter("FTM_CONTROL/TRIGGER_COUNTER", "X:1;L:1",      NULL, 0, ""),
        fDimError         ("FTM_CONTROL/ERROR",           "X:1;S:1;S:28", NULL, 0, ""),
        fDimFtuList       ("FTM_CONTROL/FTU_LIST",        "X:1;X:1;S:1;C:4;X:40;C:40;C:40",  NULL, 0, ""),
        fDimStaticData    ("FTM_CONTROL/STATIC_DATA",     "X:1;S:1;S:1;X:1;S:1;S:3;S:1;S:1;S:1;S:1;S:1;S:1;I:1;S:8;S:80;S:160;S:40;S:40", NULL, 0, ""),
        fDimDynamicData   ("FTM_CONTROL/DYNAMIC_DATA",    "X:1;X:1;F:4;I:160;I:40;S:40;S:40", NULL, 0, ""),
        fDimCounter       ("FTM_CONTROL/COUNTER",         "I:6",          NULL, 0, "")
    {
    }

    // A B [C] [D] E [F] G H [I] J K [L] M N O P Q R [S] T U V W [X] Y Z
};

// ------------------------------------------------------------------------

template <class T, class S>
class StateMachineFTM : public T, public ba::io_service, public ba::io_service::work
{
    int Wrap(boost::function<void()> f)
    {
        f();
        return T::GetCurrentState();
    }

    boost::function<int(const EventImp &)> Wrapper(boost::function<void()> func)
    {
        return boost::bind(&StateMachineFTM::Wrap, this, func);
    }

private:
    S fFTM;

    enum states_t
    {
        kStateDisconnected = FTM::kDisconnected,
        kStateConnected    = FTM::kConnected,
        kStateIdle         = FTM::kIdle,
        kStateTakingData   = FTM::kTakingData,

        kCmdTest
    };

    bool CheckEventSize(size_t has, const char *name, size_t size)
    {
        if (has==size)
            return true;

        stringstream msg;
        msg << name << " - Received event has " << has << " bytes, but expected " << size << ".";
        T::Fatal(msg);
        return false;
    }

    int SetRegister(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetRegister", 8))
            return T::kSM_FatalError;

        const unsigned int *dat = reinterpret_cast<const unsigned int*>(evt.GetData());

        if (dat[1]>uint16_t(-1))
        {
            stringstream msg;
            msg << hex << "Value " << dat[1] << " out of range.";
            T::Error(msg);
            return T::GetCurrentState();
        }


        if (dat[0]>uint16_t(-1) || !fFTM.CmdSetRegister(dat[0], dat[1]))
        {
            stringstream msg;
            msg << hex << "Address " << dat[0] << " out of range.";
            T::Error(msg);
        }

        return T::GetCurrentState();
    }

    int GetRegister(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "GetRegister", 4))
            return T::kSM_FatalError;

        const unsigned int addr = evt.GetInt();
        if (addr>uint16_t(-1) || !fFTM.CmdGetRegister(addr))
        {
            stringstream msg;
            msg << hex << "Address " << addr << " out of range.";
            T::Error(msg);
        }

        return T::GetCurrentState();
    }

    int TakeNevents(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "TakeNevents", 4))
            return T::kSM_FatalError;

        const unsigned int dat = evt.GetUInt();

        /*
        if (dat[1]>uint32_t(-1))
        {
            stringstream msg;
            msg << hex << "Value " << dat[1] << " out of range.";
            T::Error(msg);
            return T::GetCurrentState();
        }*/

        fFTM.CmdTakeNevents(dat);

        return T::GetCurrentState();
    }

    int DisableReports(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "DisableReports", 1))
            return T::kSM_FatalError;

        fFTM.CmdDisableReports(evt.GetText()[0]!=0);

        return T::GetCurrentState();
    }

    int SetVerbosity(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetVerbosity", 1))
            return T::kSM_FatalError;

        fFTM.SetVerbose(evt.GetText()[0]!=0);

        return T::GetCurrentState();
    }

    int SetHexOutput(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetHexOutput", 1))
            return T::kSM_FatalError;

        fFTM.SetHexOutput(evt.GetText()[0]!=0);

        return T::GetCurrentState();
    }

    int SetDynamicOut(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetDynamicOut", 1))
            return T::kSM_FatalError;

        fFTM.SetDynamicOut(evt.GetText()[0]!=0);

        return T::GetCurrentState();
    }

    int LoadStaticData(const EventImp &evt)
    {
        if (fFTM.LoadStaticData(evt.GetString()))
            return T::GetCurrentState();

        stringstream msg;
        msg << "Loading static data from file '" << evt.GetString() << "' failed ";

        if (errno)
            msg << "(" << strerror(errno) << ")";
        else
            msg << "(wrong size, expected " << sizeof(FTM::StaticData) << " bytes)";

        T::Warn(msg);

        return T::GetCurrentState();
    }

    int SaveStaticData(const EventImp &evt)
    {
        if (fFTM.SaveStaticData(evt.GetString()))
            return T::GetCurrentState();

        stringstream msg;
        msg << "Writing static data to file '" << evt.GetString() << "' failed ";
        msg << "(" << strerror(errno) << ")";

        T::Warn(msg);

        return T::GetCurrentState();
    }

    int SetThreshold(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetThreshold", 8))
            return T::kSM_FatalError;

        const int32_t *data = reinterpret_cast<const int32_t*>(evt.GetData());

        if (!fFTM.SetThreshold(data[0], data[1]))
            T::Warn("SetThreshold - Maximum allowed patch number 159, valid value range 0-0xffff");

        return T::GetCurrentState();
    }

    int EnableFTU(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "EnableFTU", 5))
            return T::kSM_FatalError;

        const int32_t &board  = *reinterpret_cast<const int32_t*>(evt.GetText());
        const int8_t  &enable = *reinterpret_cast<const int8_t*>(evt.GetText()+4);

        if (!fFTM.EnableFTU(board, enable))
            T::Warn("EnableFTU - Board number must be <40.");

        return T::GetCurrentState();
    }

    int ToggleFTU(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "ToggleFTU", 4))
            return T::kSM_FatalError;

        if (!fFTM.ToggleFTU(evt.GetInt()))
            T::Warn("ToggleFTU - Allowed range of boards 0-39.");

        return T::GetCurrentState();
    }

    int SetTriggerInterval(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetTriggerInterval", 4))
            return T::kSM_FatalError;

        if (!fFTM.SetTriggerInterval(evt.GetInt()))
            T::Warn("SetTriggerInterval - Value out of range.");

        return T::GetCurrentState();
    }

    int SetTriggerDelay(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetTriggerDelay", 4))
            return T::kSM_FatalError;

        if (!fFTM.SetTriggerDelay(evt.GetInt()))
            T::Warn("SetTriggerDealy -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetTimeMarkerDelay(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetTimeMarkerDelay", 4))
            return T::kSM_FatalError;

        if (!fFTM.SetTimeMarkerDelay(evt.GetInt()))
            T::Warn("SetTimeMarkerDelay -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetPrescaling(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetPrescaling", 4))
            return T::kSM_FatalError;

        if (!fFTM.SetPrescaling(evt.GetInt()))
            T::Warn("SetPrescaling -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetTriggerSeq(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetTriggerSeq", 3))
            return T::kSM_FatalError;

        const uint8_t *data = reinterpret_cast<const uint8_t*>(evt.GetData());

        if (!fFTM.SetTriggerSeq(data))
            T::Warn("SetTriggerSeq -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetDeadTime(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetDeadTime", 4))
            return T::kSM_FatalError;

        if (!fFTM.SetDeadTime(evt.GetInt()))
            T::Warn("SetDeadTime -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetTriggerCoincidence(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetTriggerCoincidence", 4))
            return T::kSM_FatalError;

        const uint16_t *d = reinterpret_cast<const uint16_t*>(evt.GetText());;

        if (!fFTM.SetTriggerCoincidence(d[0], d[1]))
            T::Warn("SetTriggerCoincidence -  Value out of range.");

        return T::GetCurrentState();
    }

    int SetCalibCoincidence(const EventImp &evt)
    {
        if (!CheckEventSize(evt.GetSize(), "SetCalibCoincidence", 4))
            return T::kSM_FatalError;

        const uint16_t *d = reinterpret_cast<const uint16_t*>(evt.GetText());;

        if (!fFTM.SetCalibCoincidence(d[0], d[1]))
            T::Warn("SetCalibCoincidence -  Value out of range.");

        return T::GetCurrentState();
    }

    int Enable(const EventImp &evt, FTM::StaticData::GeneralSettings type)
    {
        if (!CheckEventSize(evt.GetSize(), "Enable", 1))
            return T::kSM_FatalError;

        fFTM.Enable(type, evt.GetText()[0]!=0);

        return T::GetCurrentState();
    }

    int Disconnect()
    {
        // Close all connections
        fFTM.PostClose(false);

        /*
         // Now wait until all connection have been closed and
         // all pending handlers have been processed
         poll();
         */

        return T::GetCurrentState();
    }

    int Reconnect(const EventImp &evt)
    {
        // Close all connections to supress the warning in SetEndpoint
        fFTM.PostClose(false);

        // Now wait until all connection have been closed and
        // all pending handlers have been processed
        poll();

        if (evt.GetText()[0]!=0)
            fFTM.SetEndpoint(evt.GetString());

        // Now we can reopen the connection
        fFTM.PostClose(true);

        return T::GetCurrentState();
    }

    /*
    int Transition(const Event &evt)
    {
        switch (evt.GetTargetState())
        {
        case kStateDisconnected:
        case kStateConnected:
        }

        return T::kSM_FatalError;
    }*/

    int Execute()
    {
        // Dispatch (execute) at most one handler from the queue. In contrary
        // to run_one(), it doesn't wait until a handler is available
        // which can be dispatched, so poll_one() might return with 0
        // handlers dispatched. The handlers are always dispatched/executed
        // synchronously, i.e. within the call to poll_one()
        poll_one();

        return fFTM.GetState();
    }

public:
    StateMachineFTM(ostream &out=cout) :
        T(out, "FTM_CONTROL"), ba::io_service::work(static_cast<ba::io_service&>(*this)),
        fFTM(*this, *this)
    {
        // ba::io_service::work is a kind of keep_alive for the loop.
        // It prevents the io_service to go to stopped state, which
        // would prevent any consecutive calls to run()
        // or poll() to do nothing. reset() could also revoke to the
        // previous state but this might introduce some overhead of
        // deletion and creation of threads and more.

        // State names
        AddStateName(kStateDisconnected, "Disconnected",
                     "FTM board not connected via ethernet.");

        AddStateName(kStateConnected, "Connected",
                     "Ethernet connection to FTM established (no state received yet).");

        AddStateName(kStateIdle, "Idle",
                     "Ethernet connection to FTM established, FTM in idle state.");

        AddStateName(kStateTakingData, "TakingData",
                     "Ethernet connection to FTM established, FTM is in taking data state.");

        // FTM Commands
        AddConfiguration("TOGGLE_LED", kStateIdle)
            (Wrapper(boost::bind(&ConnectionFTM::CmdToggleLed, &fFTM)))
            ("toggle led");

        AddConfiguration("PING", kStateIdle)
            (Wrapper(boost::bind(&ConnectionFTM::CmdPing, &fFTM)))
            ("send ping");

        AddConfiguration("REQUEST_DYNAMIC_DATA", kStateIdle)
            (Wrapper(boost::bind(&ConnectionFTM::CmdReqDynDat, &fFTM)))
            ("request transmission of dynamic data block");

        AddConfiguration("REQUEST_STATIC_DATA", kStateIdle)
            (Wrapper(boost::bind(&ConnectionFTM::CmdReqStatDat, &fFTM)))
            ("request transmission of static data from FTM to memory");

        AddConfiguration("GET_REGISTER", "I", kStateIdle)
            (boost::bind(&StateMachineFTM::GetRegister, this, _1))
            ("read register from address addr"
            "|addr[short]:Address of register");

        AddConfiguration("SET_REGISTER", "I:2", kStateIdle)
            (boost::bind(&StateMachineFTM::SetRegister, this, _1))
            ("set register to value"
            "|addr[short]:Address of register"
            "|val[short]:Value to be set");

        AddConfiguration("START_RUN", kStateIdle)
            (Wrapper(boost::bind(&ConnectionFTM::CmdStartRun, &fFTM)))
            ("start a run (start distributing triggers)");

        AddConfiguration("STOP_RUN", kStateTakingData)
            (Wrapper(boost::bind(&ConnectionFTM::CmdStopRun, &fFTM)))
            ("stop a run (stop distributing triggers)");

        AddConfiguration("TAKE_N_EVENTS", "I", kStateIdle)
            (boost::bind(&StateMachineFTM::TakeNevents, this, _1))
            ("take n events (distribute n triggers)|number[int]:Number of events to be taken");

        AddConfiguration("DISABLE_REPORTS", "B", kStateIdle)
            (boost::bind(&StateMachineFTM::DisableReports, this, _1))
            ("disable sending rate reports"
             "|status[bool]:disable or enable that the FTM sends rate reports (yes/no)");

        AddConfiguration("SET_THRESHOLD", "I:2", kStateIdle)
            (boost::bind(&StateMachineFTM::SetThreshold, this, _1))
            ("Set the comparator threshold"
             "|Patch[idx]:Index of the patch (0-159), -1 for all"
             "|Threshold[counts]:Threshold to be set in binary counts");

        AddConfiguration("SET_PRESCALING", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::SetPrescaling, this, _1))
            (""
             "|[]:");

        AddConfiguration("ENABLE_FTU", "I:1;B:1", kStateIdle)
            (boost::bind(&StateMachineFTM::EnableFTU, this, _1))
            ("Enable or disable FTU"
             "|Board[idx]:Index of the board (0-39), -1 for all"
             "|Enable[bool]:Whether FTU should be enabled or disabled (yes/no)");

        AddConfiguration("TOGGLE_FTU", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::ToggleFTU, this, _1))
            ("Toggle status of FTU (this is mainly meant to be used in the GUI)"
             "|Board[idx]:Index of the board (0-39)");

        AddConfiguration("SET_TRIGGER_INTERVAL", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::SetTriggerInterval, this, _1))
            (""
             "|[]:");

        AddConfiguration("SET_TRIGGER_DELAY", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::SetTriggerDelay, this, _1))
            (""
             "|[]:");

        AddConfiguration("SET_TIME_MARKER_DELAY", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::SetTimeMarkerDelay, this, _1))
            (""
             "|[]:");

        AddConfiguration("SET_DEAD_TIME", "I:1", kStateIdle)
            (boost::bind(&StateMachineFTM::SetDeadTime, this, _1))
            (""
             "|[]:");

        AddConfiguration("ENABLE_TRIGGER", "B:1", kStateIdle)
            (boost::bind(&StateMachineFTM::Enable, this, _1, FTM::StaticData::kTrigger))
            (""
             "|[]:");

        // FIXME: Switch on/off depending on sequence
        AddConfiguration("ENABLE_EXT1", "B:1", kStateIdle)
            (boost::bind(&StateMachineFTM::Enable, this, _1, FTM::StaticData::kExt1))
            (""
             "|[]:");

        // FIXME: Switch on/off depending on sequence
        AddConfiguration("ENABLE_EXT2", "B:1", kStateIdle)
            (boost::bind(&StateMachineFTM::Enable, this, _1, FTM::StaticData::kExt2))
            (""
             "|[]:");

        AddConfiguration("ENABLE_VETO", "B:1", kStateIdle)
            (boost::bind(&StateMachineFTM::Enable, this, _1, FTM::StaticData::kVeto))
            (""
             "|[]:");

        AddConfiguration("SET_TRIGGER_SEQUENCE", "C:3", kStateIdle)
            (boost::bind(&StateMachineFTM::SetTriggerSeq, this, _1))
            (""
             "|[]:");

        AddConfiguration("SET_TRIGGER_COINCIDENCE", "S:2", kStateIdle)
            (boost::bind(&StateMachineFTM::SetTriggerCoincidence, this, _1))
            (""
             "|[]:");

        AddConfiguration("SET_CALIBRATION_COINCIDENCE", "S:2", kStateIdle)
            (boost::bind(&StateMachineFTM::SetCalibCoincidence, this, _1))
            (""
             "|[]:");


        // Load/save static data block
        T::AddConfiguration("SAVE", "C", kStateIdle)
            (boost::bind(&StateMachineFTM::SaveStaticData, this, _1))
            ("Saves the static data (FTM configuration) from memory to a file"
             "|filename[string]:Filename (can include a path), .bin is automatically added");

        T::AddConfiguration("LOAD", "C", kStateIdle)
            (boost::bind(&StateMachineFTM::LoadStaticData, this, _1))
            ("Loads the static data (FTM configuration) from a file into memory and sends it to the FTM"
             "|filename[string]:Filename (can include a path), .bin is automatically added");



        // Verbosity commands
        T::AddConfiguration("SET_VERBOSE", "B")
            (boost::bind(&StateMachineFTM::SetVerbosity, this, _1))
            ("set verbosity state"
             "|verbosity[bool]:disable or enable verbosity for received data (yes/no)");

        T::AddConfiguration("SET_HEX_OUTPUT", "B")
            (boost::bind(&StateMachineFTM::SetHexOutput, this, _1))
            ("enable or disable hex output for received data"
             "|hexout[bool]:disable or enable hex output for verbose and received data (yes/no)");

        T::AddConfiguration("SET_DYNAMIC_OUTPUT", "B")
            (boost::bind(&StateMachineFTM::SetDynamicOut, this, _1))
            ("enable or disable output for received dynamic data (data is still broadcasted via Dim)"
             "|dynout[bool]:disable or enable output for dynamic data (yes/no)");


        // Conenction commands
        AddConfiguration("DISCONNECT", kStateConnected, kStateIdle)
            (boost::bind(&StateMachineFTM::Disconnect, this))
            ("disconnect from ethernet");

        AddConfiguration("RECONNECT", "O", kStateDisconnected, kStateConnected, kStateIdle)
            (boost::bind(&StateMachineFTM::Reconnect, this, _1))
            ("(Re)connect ethernet connection to FTM, a new address can be given"
             "|[host][string]:new ethernet address in the form <host:port>");

        // Other
        AddTransition(kCmdTest, "TEST", "O")
            (boost::bind(&StateMachineFTM::Test, this, _1))
            ("Just for test purpose, do not use");

        fFTM.StartConnect();
    }

        /// Just for test purpose, do not touch
    int Test(const Event &evt)
    {
        const Converter conv(T::Out(), evt.GetFormat(), false);
        T::Out() << kBlue << evt.GetName();
        T::Out() << " " << conv.GetString(evt.GetData(), evt.GetSize());
        T::Out() << endl;

        return T::GetCurrentState();
    }

    void SetEndpoint(const string &url)
    {
        fFTM.SetEndpoint(url);
    }

    bool SetConfiguration(const Configuration &conf)
    {
        SetEndpoint(conf.Get<string>("addr"));
        return true;
    }
};

// ------------------------------------------------------------------------

void RunThread(StateMachineImp *io_service)
{
    // This is necessary so that the StateMachien Thread can signal the
    // Readline to exit
    io_service->Run();
    Readline::Stop();
}

template<class S, class T>
int RunDim(Configuration &conf)
{
    WindowLog wout;

    /*
    static Test shell(conf.GetName().c_str(), conf.Get<int>("console")!=1);

    WindowLog &win  = shell.GetStreamIn();
    WindowLog &wout = shell.GetStreamOut();
    */

    if (conf.Has("log"))
        if (!wout.OpenLogFile(conf.Get<string>("log")))
            wout << kRed << "ERROR - Couldn't open log-file " << conf.Get<string>("log") << ": " << strerror(errno) << endl;

    // Start io_service.Run to use the StateMachineImp::Run() loop
    // Start io_service.run to only use the commandHandler command detaching
    StateMachineFTM<S, T> io_service(wout);
    if (!io_service.SetConfiguration(conf))
        return -1;

    io_service.Run();

    /*
    shell.SetReceiver(io_service);

    boost::thread t(boost::bind(RunThread, &io_service));
    // boost::thread t(boost::bind(&StateMachineFTM<S>::Run, &io_service));

    shell.Run();                 // Run the shell
    io_service.Stop();           // Signal Loop-thread to stop
    // io_service.Close();       // Obsolete, done by the destructor

    // Wait until the StateMachine has finished its thread
    // before returning and destroying the dim objects which might
    // still be in use.
    t.join();
    */

    return 0;
}

template<class T, class S, class R>
int RunShell(Configuration &conf)
{
    static T shell(conf.GetName().c_str(), conf.Get<int>("console")!=1);

    WindowLog &win  = shell.GetStreamIn();
    WindowLog &wout = shell.GetStreamOut();

    if (conf.Has("log"))
        if (!wout.OpenLogFile(conf.Get<string>("log")))
            win << kRed << "ERROR - Couldn't open log-file " << conf.Get<string>("log") << ": " << strerror(errno) << endl;

    StateMachineFTM<S, R> io_service(wout);
    if (!io_service.SetConfiguration(conf))
        return -1;

    shell.SetReceiver(io_service);

    boost::thread t(boost::bind(RunThread, &io_service));
    // boost::thread t(boost::bind(&StateMachineFTM<S>::Run, &io_service));

    shell.Run();                 // Run the shell
    io_service.Stop();           // Signal Loop-thread to stop
    // io_service.Close();       // Obsolete, done by the destructor

    // Wait until the StateMachine has finished its thread
    // before returning and destroying the dim objects which might
    // still be in use.
    t.join();

    return 0;
}

void SetupConfiguration(Configuration &conf)
{
    const string n = conf.GetName()+".log";

    po::options_description config("Program options");
    config.add_options()
        ("dns",       var<string>("localhost"), "Dim nameserver host name (Overwites DIM_DNS_NODE environment variable)")
        ("log,l",     var<string>(n), "Write log-file")
        ("no-dim,d",  po_switch(),    "Disable dim services")
        ("console,c", var<int>(),     "Use console (0=shell, 1=simple buffered, X=simple unbuffered)")
        ;

    po::options_description control("FTM control options");
    control.add_options()
        ("addr",      var<string>("localhost:5000"),  "Network address of FTM")
        ;

    conf.AddEnv("dns", "DIM_DNS_NODE");

    conf.AddOptions(config);
    conf.AddOptions(control);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "The ftmctrl controls the FTM (FACT Trigger Master) board.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: ftmctrl [-c type] [OPTIONS]\n"
        "  or:  ftmctrl [OPTIONS]\n"
        "\n"
        "Options:\n"
        "The following describes the available commandline options. "
        "For further details on how command line option are parsed "
        "and in which order which configuration sources are accessed "
        "please refer to the class reference of the Configuration class.";
    cout << endl;

}

void PrintHelp()
{
}

/*
 The first line of the --version information is assumed to be in one
 of the following formats:

   <version>
   <program> <version>
   {GNU,Free} <program> <version>
   <program> ({GNU,Free} <package>) <version>
   <program> - {GNU,Free} <package> <version>

 and separated from any copyright/author details by a blank line.

 Handle multi-line bug reporting sections of the form:

   Report <program> bugs to <addr>
   GNU <package> home page: <url>
   ...
*/
void PrintVersion(const char *name)
{
    cout <<
        name << " - "PACKAGE_STRING"\n"
        "\n"
        "Written by Thomas Bretz et al.\n"
        "\n"
        "Report bugs to <"PACKAGE_BUGREPORT">\n"
        "Home page: "PACKAGE_URL"\n"
        "\n"
        "Copyright (C) 2011 by the FACT Collaboration.\n"
        "This is free software; see the source for copying conditions.\n"
        << endl;
}
/*
string GetLocalIp()
{
    const char *kDnsIp = getenv("DIM_DNS_NODE");

    struct addrinfo hints, *servinfo, *p;

    memset(&hints, 0, sizeof hints);
    hints.ai_family   = AF_INET; //AF_UNSPEC; // use AF_INET6 to force IPv6
    hints.ai_socktype = SOCK_STREAM;

    int rv;
    if ((rv = getaddrinfo(kDnsIp, NULL, &hints, &servinfo)) != 0)
    {
        cout << "WARNING - getaddrinfo: " << gai_strerror(rv) << endl;
        return kDnsIp;
    }

    // loop through all the results and connect to the first we can
    for (p=servinfo; p; p=p->ai_next)
    {
        const int sock = socket(AF_INET, SOCK_DGRAM, 0);
        if (sock==-1)
            continue;

        if (connect(sock, p->ai_addr, p->ai_addrlen)==-1)
        {
            cout << "WARNING - connect: " << strerror(errno) << endl;
            close(sock);
            continue;
        }

        sockaddr_in name;
        socklen_t namelen = sizeof(name);
        if (getsockname(sock, (sockaddr*)&name, &namelen)==-1)
        {
            cout << "WARNING - getsockname: " << strerror(errno) << endl;
            close(sock);
            continue;
        }

        char buffer[16];
        if (!inet_ntop(AF_INET, &name.sin_addr, buffer, 16))
        {
            cout << "WARNING - inet_ntop: " << strerror(errno) << endl;
            close(sock);
            continue;
        }

        close(sock);

        freeaddrinfo(servinfo); // all done with this structure

        cout << "DIM_HOST_NODE=" << buffer << endl;
        return buffer;
    }

    freeaddrinfo(servinfo); // all done with this structure
    return kDnsIp;
}
*/

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    SetupConfiguration(conf);

    po::variables_map vm;
    try
    {
        vm = conf.Parse(argc, argv);
    }
#if BOOST_VERSION > 104000
    catch (po::multiple_occurrences &e)
    {
        cout << "Error: " << e.what() << " of '" << e.get_option_name() << "' option." << endl;
        cout << endl;
        return -1;
    }
#endif
    catch (std::exception &e)
    {
        cout << "Error: " << e.what() << endl;
        cout << endl;

        return -1;
    }

    if (conf.HasPrint())
        return -1;

    if (conf.HasVersion())
    {
        PrintVersion(argv[0]);
        return -1;
    }

    if (conf.HasHelp())
    {
        PrintHelp();
        return -1;
    }

    // To allow overwriting of DIM_DNS_NODE set 0 to 1
    setenv("DIM_DNS_NODE", conf.Get<string>("dns").c_str(), 1);
    //setenv("DIM_HOST_NODE", GetLocalIp().c_str(), 1);

    //try
    {
        // No console access at all
        if (!conf.Has("console"))
        {
            if (conf.Get<bool>("no-dim"))
                return RunDim<StateMachine, ConnectionFTM>(conf);
            else
                return RunDim<StateMachineDim, ConnectionDimFTM>(conf);
        }
        // Cosole access w/ and w/o Dim
        if (conf.Get<bool>("no-dim"))
        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell, StateMachine, ConnectionFTM>(conf);
            else
                return RunShell<LocalConsole, StateMachine, ConnectionFTM>(conf);
        }
        else
        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell, StateMachineDim, ConnectionDimFTM>(conf);
            else
                return RunShell<LocalConsole, StateMachineDim, ConnectionDimFTM>(conf);
        }
    }
    /*catch (std::exception& e)
    {
        cerr << "Exception: " << e.what() << endl;
        return -1;
    }*/

    return 0;
}
