#ifndef FACT_EventBuilderWrapper
#define FACT_EventBuilderWrapper

/*
#if BOOST_VERSION < 104400
#if (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 4))
#undef BOOST_HAS_RVALUE_REFS
#endif
#endif
#include <boost/thread.hpp>

using namespace std;
*/

#include <boost/date_time/posix_time/posix_time_types.hpp>

#include <CCfits/CCfits>

#include "EventBuilder.h"

extern "C" {
    extern void StartEvtBuild();
    extern int CloseRunFile(uint32_t runId, uint32_t closeTime);
}

class DataFileImp
{
    uint32_t fRunId;

public:
    DataFileImp(uint32_t id) : fRunId(id) { }

    virtual bool OpenFile(uint32_t runid, RUN_HEAD* h) = 0;
    virtual bool Write(EVENT *) = 0;
    virtual bool Close(RUN_TAIL * = 0) = 0;

    uint32_t GetRunId() const { return fRunId; }
};


#include "FAD.h"

class DataFileRaw : public DataFileImp
{
public:
    DataFileRaw(uint32_t id) : DataFileImp(id)  { }
    ~DataFileRaw() { Close(); }

    virtual bool OpenFile(uint32_t, RUN_HEAD* ) { return true; }
    virtual bool Write(EVENT *) { return true; }
    virtual bool Close(RUN_TAIL * = 0) { return true; }
};


class DataFileFits : public DataFileImp
{
    CCfits::FITS*  fFile;        /// The pointer to the CCfits FITS file
    CCfits::Table* fTable;       /// The pointer to the CCfits binary table

    uint64_t fNumRows;                ///the number of rows that have been written already to the FITS file.
    uint32_t fRoi;                ///the number of rows that have been written already to the FITS file.

public:
    DataFileFits(uint32_t runid) : DataFileImp(runid), fFile(0)
    {
    }

    // --------------------------------------------------------------------------
    //
    //! Default destructor
    //! The Fits file SHOULD have been closed already, otherwise the informations
    //! related to the RUN_TAIL will NOT be written to the file.
    //
    ~DataFileFits() { Close(); }

    // --------------------------------------------------------------------------
    //
    //! Add a new column to the vectors storing the column data.
    //! @param names the vector of string storing the columns names
    //! @param types the vector of string storing the FITS data format
    //! @param numElems the number of elements in this column
    //! @param type the char describing the FITS data format
    //! @param name the name of the particular column to be added.
    //
    inline void AddColumnEntry(vector<string>& names, vector<string>& types, int numElems, char type, string name)
    {
        names.push_back(name);

        ostringstream str;
        if (numElems != 1)
            str << numElems;
        str << type;
        types.push_back(str.str());
    }

    // --------------------------------------------------------------------------
    //
    //! Writes a single header key entry
    //! @param name the name of the key
    //! @param value its value
    //! @param comment the comment associated to that key
    //
    //FIXME this function is a duplicate from the class Fits. should we try to merge it ?
    template <typename T>
    void WriteKey(const string &name, const T &value, const string &comment)
    {
        try
        {
            fTable->addKey(name, value, comment);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not add header key ";
            //TODO pipe the error message somewhere
        }
    }

    template <typename T>
    void WriteKey(const string &name, const int idx, const T &value, const string &comment)
    {
        ostringstream str;
        str << name << idx;

        WriteKey(str.str(), value, comment);
    }

    // --------------------------------------------------------------------------
    //
    //! This creates an appropriate file name for a particular run number and type
    //! @param runNumber the run number for which a filename is to be created
    //! @param runType an int describing the kind of run. 0=Data, 1=Pedestal, 2=Calibration, 3=Calibrated data
    //! @param extension a string containing the extension to be appened to the file name
    //
    string FormFileName(uint32_t runNumber, uint32_t runType, string extension)
    {
        //TODO where am I supposed to get the base directory from ?
        //TODO also, for creating subsequent directories, should I use the functions from the dataLogger ?
        string baseDirectory = "./Run";
        ostringstream result;
        result << baseDirectory;
        result << Time::fmt("/%Y/%m/%d/") << (Time() - boost::posix_time::time_duration(12,0,0));
        result << setfill('0') << setw(8) << runNumber;
        result << "_001_";
        switch (runType)
        {
        case 0:
            result << 'D';
            break;
        case 1:
            result << 'P';
            break;
        case 2:
            result << 'C';
            break;
        case 3:
            result << 'Y';
            break;
        default:
            //TODO pipe this error message to the appropriate error stream
            cout << "Error unexpected event" << endl;
        };
        result << "_data." << extension;

        return result.str();
    }

    // --------------------------------------------------------------------------
    //
    //! DataFileFits constructor. This is the one that should be used, not the default one (parameter-less)
    //! @param runid This parameter should probably be removed. I first thought it was the run number, but apparently it is not
    //! @param h a pointer to the RUN_HEAD structure that contains the informations relative to this run
    //
    bool OpenFile(uint32_t /*runid*/, RUN_HEAD* h)
    {
        //Form filename, based on runid and run-type
        string fileName = FormFileName(h->FADhead[0].runnumber, h->RunType, "fits");

        //create the FITS object
        try
        {
            fFile = new CCfits::FITS(fileName, CCfits::RWmode::Write);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not open FITS file " << fileName << " reason: " << e.message();
            //TODO display the message somewhere

            return false;
        }

        //create columns according to header
        ostringstream arrayTypes;

        // uint32_t EventNum ;       // EventNumber as from FTM
        // uint16_t TriggerType ;    // Trigger Type from FTM
        // uint32_t SoftTrig ;       // SoftTrigger Info (TBD)
        // uint32_t PCTime ;         // when did event start to arrive at PC
        // uint32_t BoardTime[NBOARDS];//
        // int16_t  StartPix[NPIX];  // First Channel per Pixel (Pixels sorted according Software ID)  ; -1 if not filled
        // int16_t  StartTM[NTMARK]; // First Channel for TimeMark (sorted Hardware ID) ; -1 if not filled
        // uint16_t Adc_Data[];      // final length defined by malloc ....

        vector<string> colNames;
        vector<string> dataTypes;
        AddColumnEntry(colNames, dataTypes, 1,              'V', "EventNum");
        AddColumnEntry(colNames, dataTypes, 1,              'U', "TriggerType");
        AddColumnEntry(colNames, dataTypes, 1,              'V', "SoftTrig");
        AddColumnEntry(colNames, dataTypes, 1,              'V', "PCTime");
        AddColumnEntry(colNames, dataTypes, NBOARDS,        'V', "BoardTime");
        AddColumnEntry(colNames, dataTypes, NPIX,           'I', "StartPix");
        AddColumnEntry(colNames, dataTypes, NTMARK,         'I', "StartTM");
        AddColumnEntry(colNames, dataTypes, NPIX*h->Nroi,   'U', "Data");

        fRoi = h->Nroi;

        //actually create the table
        try
        {
            fTable = fFile->addTable("Events", 0, colNames, dataTypes);
            if (fTable->rows() != 0)
            {
                ostringstream str;
                str << "Error: table created on the fly looks non-empty.";
                //TODO giev the error text to some error handler
                //FIXME I guess that this error checking is useless. remove it for performances.
            }
        }
        catch(CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not create FITS table " << "Events" << " in file " << fileName << " reason: " << e.message();
            //TODO give the error text to some error handler

            Close();
            return false;
        }

        //write header data
        //first the "standard" keys
        string stringValue;
        WriteKey("EXTREL",   1.0f,   "Release Number");
        WriteKey("TELESCOP", "FACT", "Telescope that acquired this data");
        WriteKey("ORIGIN",   "ISDC", "Institution that wrote the file");
        WriteKey("CREATOR",  "FACT++ Event Builder", "Program that wrote this file");

        stringValue = Time().GetAsStr();
        stringValue[10]= 'T';
        WriteKey("DATE",     stringValue, "File creation data");
        WriteKey("TIMESYS",  "TT",        "Time frame system");
        WriteKey("TIMEUNIT", "d",         "Time unit");
        WriteKey("TIMEREF",  "UTC",       "Time reference frame");
        //FIXME should we also put the start and stop time of the received data ?
        //now the events header related variables
        WriteKey("VERSION",  h->Version,  "Builder version");
        WriteKey("RUNTYPE",  h->RunType,  "Type of run");
        WriteKey("NBOARD",   h->NBoard,   "Number of acquisition boards");
        WriteKey("NPIX",     h->NPix,     "Number of pixels");
        WriteKey("NTM",      h->NTm,      "Number of Time marks");
        WriteKey("NROI",     h->Nroi,     "Number of slices per pixels");

        //now the boards related keywords
        for (int i=0; i<h->NBoard; i++)
        {
            const PEVNT_HEADER &hh = h->FADhead[i];

            WriteKey("STPKGFG", i, hh.start_package_flag,
                     "Start package flag");

            WriteKey("PKGLEN",  i, hh.package_length,
                     "Package length");

            WriteKey("VERNO",   i, hh.version_no,
                     "Version number");

            WriteKey("PLLLCK",  i, hh.PLLLCK,
                     "");
/*
            WriteKey("TRIGCRC", i, hh.trigger_crc,
                     "Trigger CRC");

            WriteKey("TRIGTYP", i, hh.trigger_type,
                     "Trigger type");

            WriteKey("TRIGID",  i, hh.trigger_id,
                     "Trigger ID");

            WriteKey("EVTCNTR", i, hh.fad_evt_counter,
                     "FAD Event Counter");
*/
            WriteKey("REFCLK",  i, hh.REFCLK_frequency,
                     "Reference Clock Frequency");

            WriteKey("BOARDID", i, hh.board_id,
                     "Board ID");

            WriteKey("PHASESH", i, hh.adc_clock_phase_shift,
                     "ADC clock phase shift");

            //WriteKey("TRGGEN",  i, hh.number_of_triggers_to_generate,
            //         "Number of triggers to generate");

            WriteKey("PRESC",   i, hh.trigger_generator_prescaler,
                    "Trigger generator prescaler");

            WriteKey("DNA",     i, hh.DNA,       "DNA");
            WriteKey("TIME",    i, hh.time,      "Time");
            WriteKey("RUNNB",   i, hh.runnumber, "Run number");
/*
            for (int j=0;j<NTemp;j++)
            {
                str.str(""); str2.str("");
                str << "DRS_T" << i << j;
                str2 << "DRS temperature #" << i << " " << j;
                WriteKey(str.str(), h->FADhead[i].drs_temperature[j], str2.str());
            }
*/
            for (int j=0;j<NDAC;j++)
                WriteKey("DAC", i*NDAC+j, hh.dac[j], "DAC");

            //Last but not least, add header keys that will be updated when closing the file
            WriteFooter(NULL);
        }

        return true;
    }


    int WriteColumns(size_t &start, size_t size, void *e)
    {
        int status = 0;
        fits_write_tblbytes(fFile->fitsPointer(), fNumRows, start, size, reinterpret_cast<unsigned char*>(e), &status);
        start += size;
        return status;
    }

    // --------------------------------------------------------------------------
    //
    //! This writes one event to the file
    //! @param e the pointer to the EVENT
    //
    virtual bool Write(EVENT *e)
    {
        //FIXME As discussed earlier, we do not swap the bytes yet.
        fTable->makeThisCurrent();

        //insert a new row
        int status(0);
        if (fits_insert_rows(fTable->fitsPointer(), fNumRows, 1, &status))
        {
            ostringstream str;
            str << "Could not insert a row in fits file";
            //TODO pipe this error message to the appropriate error stream
        }
        fNumRows++;

        const int sh = sizeof(PEVNT_HEADER)-1;

        // column size pointer
        size_t col = 1;
        if (!WriteColumns(col, sh + NPIX*fRoi*2, e))
            return true;

        //TODO output an error
        return false;

        /*
        //write the data, chunk by chunk
        //FIXME hard-coded size corresponds to current variables of the event, in bytes.
        //FIXME no padding was taken into account. Because smallest member is 2 bytes, I don't think that this should be a problem.
        const long sizeInBytesOfEventBeforePointers = 16;

        long col = 1;
        if (FitsWriteTblBytes(col, sizeInBytesOfEventBeforePointers, e))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NBOARDS*2, e->BoardTime))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*2, e->StartPix))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NTMARK*2, e->StartTM))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*fRoi*2, e->Adc_Data))
        {
            //TODO output an error
            return false;
        }
        return true;*/
    }

    void WriteFooter(RUN_TAIL *rt)
    {
        //write final header keys
        fTable->makeThisCurrent();

        WriteKey("NBEVTOK",  rt ? rt->nEventsOk  : uint32_t(0),
                 "How many events were written");

        WriteKey("NBEVTREJ", rt ? rt->nEventsRej : uint32_t(0),
                 "How many events were rejected by SW-trig");

        WriteKey("NBEVTBAD", rt ? rt->nEventsBad : uint32_t(0),
                 "How many events were rejected by Error");

        //FIXME shouldn't we convert start and stop time to MjD first ?
        //FIXME shouldn't we also add an MjD reference ?

        WriteKey("TSTART",   rt ? rt->PCtime0    : uint32_t(0),
                 "Time when first event received");

        WriteKey("TSTOP",    rt ? rt->PCtimeX    : uint32_t(0),
                 "Time when last event received");
    }

    // --------------------------------------------------------------------------
    //
    //! Closes the file, and before this it write the TAIL data
    //! @param rt the pointer to the RUN_TAIL data structure
    //
    virtual bool Close(RUN_TAIL *rt = 0)
    {
        if (!fFile)
            return false;

        WriteFooter(rt);

        delete fFile;
        fFile = NULL;

        return true;
    }

};

class EventBuilderWrapper
{
public:
    // FIXME
    static EventBuilderWrapper *This;

private:
    boost::thread fThread;

    enum CommandStates_t // g_runStat
    {
        kAbort      = -2,  // quit as soon as possible ('abort')
        kExit       = -1,  // stop reading, quit when buffered events done ('exit')
        kInitialize =  0,  // 'initialize' (e.g. dim not yet started)
        kHybernate  =  1,  // do nothing for long time ('hybernate') [wakeup within ~1sec]
        kSleep      =  2,  // do nothing ('sleep')                   [wakeup within ~10msec]
        kModeFlush  = 10,  // read data from camera, but skip them ('flush')
        kModeTest   = 20,  // read data and process them, but do not write to disk ('test')
        kModeFlag   = 30,  // read data, process and write all to disk ('flag')
        kModeRun    = 40,  // read data, process and write selected to disk ('run')
    };

    MessageImp &fMsg;

    enum
    {
        kCurrent = 0,
        kTotal   = 1
    };

    bool fFitsFormat;

    uint32_t fMaxRun;
    uint32_t fNumEvts[2];

    DimDescribedService fDimFiles;
    DimDescribedService fDimRuns;
    DimDescribedService fDimEvents;
    DimDescribedService fDimCurrentEvent;

public:
    EventBuilderWrapper(MessageImp &msg) : fMsg(msg), 
        fFitsFormat(false), fMaxRun(0),
        fDimFiles ("FAD_CONTROL/FILES",         "X:1", ""),
        fDimRuns  ("FAD_CONTROL/RUNS",          "I:1", ""),
        fDimEvents("FAD_CONTROL/EVENTS",        "I:2", ""),
        fDimCurrentEvent("FAD_CONTROL/CURRENT_EVENT", "I:1", "")
    {
        if (This)
            throw logic_error("EventBuilderWrapper cannot be instantiated twice.");

        This = this;

        memset(fNumEvts, 0, sizeof(fNumEvts));

        Update(fDimRuns,         uint32_t(0));
        Update(fDimCurrentEvent, uint32_t(0));
        Update(fDimEvents,       fNumEvts);
    }
    ~EventBuilderWrapper()
    {
        Abort();
        // FIXME: Used timed_join and abort afterwards
        //        What's the maximum time the eb need to abort?
        fThread.join();
        //fMsg.Info("EventBuilder stopped.");

        for (vector<DataFileImp*>::iterator it=fFiles.begin(); it!=fFiles.end(); it++)
            delete *it;
    }

    void Update(ostringstream &msg, int severity)
    {
        fMsg.Update(msg, severity);
    }

    bool IsThreadRunning()
    {
        return !fThread.timed_join(boost::posix_time::microseconds(0));
    }

    void SetMaxMemory(unsigned int mb) const
    {
        if (mb*1000000<GetUsedMemory())
        {
            // fMsg.Warn("...");
            return;
        }

        g_maxMem = mb*1000000;
    }

    void Start(const vector<tcp::endpoint> &addr)
    {
        if (IsThreadRunning())
        {
            fMsg.Warn("Start - EventBuilder still running");
            return;
        }

        int cnt = 0;
        for (size_t i=0; i<40; i++)
        {
            if (addr[i]==tcp::endpoint())
            {
                g_port[i].sockDef = -1;
                continue;
            }

            // -1:  if entry shell not be used
            //  0:  event builder will connect but ignore the events
            //  1:  event builder will connect and build events
            g_port[i].sockDef                  = 1;

            g_port[i].sockAddr.sin_family      = AF_INET;
            g_port[i].sockAddr.sin_addr.s_addr = htonl(addr[i].address().to_v4().to_ulong());
            g_port[i].sockAddr.sin_port        = htons(addr[i].port());

            cnt++;
        }

//        g_maxBoards = cnt;
        g_actBoards = cnt;

        g_runStat   = kModeRun;

        fMsg.Message("Starting EventBuilder thread");

        fThread = boost::thread(StartEvtBuild);
    }
    void Abort()
    {
        fMsg.Message("Signal abort to EventBuilder thread...");
        g_runStat = kAbort;
    }

    void Exit()
    {
        fMsg.Message("Signal exit to EventBuilder thread...");
        g_runStat = kExit;
    }

    /*
    void Wait()
    {
        fThread.join();
        fMsg.Message("EventBuilder stopped.");
    }*/

    void Hybernate() const { g_runStat = kHybernate; }
    void Sleep()     const { g_runStat = kSleep;     }
    void FlushMode() const { g_runStat = kModeFlush; }
    void TestMode()  const { g_runStat = kModeTest;  }
    void FlagMode()  const { g_runStat = kModeFlag;  }
    void RunMode()   const { g_runStat = kModeRun;   }

    // FIXME: To be removed
    void SetMode(int mode) const { g_runStat = mode; }

    bool IsConnected(int i) const     { return gi_NumConnect[i]==7; }
    bool IsDisconnected(int i) const  { return gi_NumConnect[i]<=0; }
    int  GetNumConnected(int i) const { return gi_NumConnect[i]; }

    size_t GetUsedMemory() const { return gi_usedMem; }

    virtual int CloseOpenFiles() { CloseRunFile(0, 0); return 0; }


    /*
     struct OpenFileToDim
     {
        int code;
        char fileName[FILENAME_MAX];
     };

     SignalRunOpened(runid, filename);
     // Send num open files
     // Send runid, (more info about the run?), filename via dim

     SignalEvtWritten(runid);
     // Send num events written of newest file

     SignalRunClose(runid);
     // Send new num open files
     // Send empty file-name if no file is open

     */

    // -------------- Mapped event builder callbacks ------------------

    vector<DataFileImp*> fFiles;

    template<class T>
        void Update(DimDescribedService &svc, const T &data) const
    {
        cout << "Update: " << svc.getName() << " (" << sizeof(T) << ")" << endl;
        svc.setData(const_cast<T*>(&data), sizeof(T));
        svc.updateService();
    }

    FileHandle_t runOpen(uint32_t runid, RUN_HEAD *h, size_t)
    {
        // Check if file already exists...
        DataFileImp *file = fFitsFormat ?
            static_cast<DataFileImp*>(new DataFileFits(runid)) :
            static_cast<DataFileImp*>(new DataFileRaw(runid));

        try
        {
            if (!file->OpenFile(runid, h))
                return 0;
        }
        catch (const exception &e)
        {
            return 0;
        }

        cout << "OPEN_FILE #" << runid << " (" << file << ")" <<  endl;
        cout << " Ver= " << h->Version << endl;
        cout << " Typ= " << h->RunType << endl;
        cout << " Nb = " << h->NBoard << endl;
        cout << " Np = " << h->NPix << endl;
        cout << " NTm= " << h->NTm << endl;
        cout << " roi= " << h->Nroi << endl;

        fFiles.push_back(file);

        if (runid>fMaxRun)
        {
            fMaxRun = runid;
            fNumEvts[kCurrent] = 0;

            Update(fDimRuns,         fMaxRun);
            Update(fDimEvents,       fNumEvts);
            Update(fDimCurrentEvent, uint32_t(0));
        }

        Update(fDimFiles, fFiles.size());

//        fDimFiles.setData(fFiles.size());
//        fDimFiles.update();

        return reinterpret_cast<FileHandle_t>(file);
    }

    int runWrite(FileHandle_t handler, EVENT *e, size_t)
    {
        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

        cout << "WRITE_EVENT " << file->GetRunId() << endl;

        cout << " Evt=" << e->EventNum << endl;
        cout << " Typ=" << e->TriggerType << endl;
        cout << " roi=" << e->Roi << endl;
        cout << " trg=" << e->SoftTrig << endl;
        cout << " tim=" << e->PCTime << endl;

        if (!file->Write(e))
            return -1;

        if (file->GetRunId()==fMaxRun)
        {
            Update(fDimCurrentEvent, e->EventNum);
            fNumEvts[kCurrent]++;
        }

        fNumEvts[kTotal]++;
        Update(fDimEvents, fNumEvts);

        // ===> SignalEvtWritten(runid);
        // Send num events written of newest file

        /* close run runId (all all runs if runId=0) */
        /* return: 0=close scheduled / >0 already closed / <0 does not exist */
        //CloseRunFile(file->GetRunId(), time(NULL)+2) ;

        return 0;
    }

    int runClose(FileHandle_t handler, RUN_TAIL *tail, size_t)
    {
        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

         const vector<DataFileImp*>::iterator it = find(fFiles.begin(), fFiles.end(), file);
        if (it==fFiles.end())
        {
            ostringstream str;
            str << "File handler (" << handler << ") requested to close by event builder doesn't exist.";
            fMsg.Fatal(str);
            return -1;
        }

        ostringstream str;
        str << "CLOSE_RUN requested for " << file->GetRunId() << " (" << file << ")" <<endl;
        fMsg.Debug(str);

        fFiles.erase(it);

        Update(fDimFiles, fFiles.size());

        //fDimFiles.setData(fFiles.size());
        //fDimFiles.update();

        const bool rc = file->Close(tail);
        if (!rc)
        {
            // Error message
        }

        delete file;

        // ==> SignalRunClose(runid);
        // Send new num open files
        // Send empty file-name if no file is open

        return rc ? 0 : -1;
    }
};

EventBuilderWrapper *EventBuilderWrapper::This = 0;

// ----------- Event builder callbacks implementation ---------------
extern "C"
{
    FileHandle_t runOpen(uint32_t irun, RUN_HEAD *runhd, size_t len)
    {
        return EventBuilderWrapper::This->runOpen(irun, runhd, len);
    }

    int runWrite(FileHandle_t fileId, EVENT *event, size_t len)
    {
        return EventBuilderWrapper::This->runWrite(fileId, event, len);
    }

    int runClose(FileHandle_t fileId, RUN_TAIL *runth, size_t len)
    {
        return EventBuilderWrapper::This->runClose(fileId, runth, len);
    }

    void factOut(int severity, int err, char *message)
    {
        // FIXME: Make the output to the console stream thread-safe
        ostringstream str;
        str << "EventBuilder(";
        if (err<0)
            str << "---";
        else
            str << err;
        str << "): " << message;
        EventBuilderWrapper::This->Update(str, severity);
    }

    void factStat(int severity, int err, char* message )
    {
        static string last;
        if (message==last)
            return;

        if (err!=-1)
            factOut(severity, err, message);
        else
        {
            ostringstream str("Status: ");
            str << message;
            EventBuilderWrapper::This->Update(str, severity);
        }

        last = message;
    }

    /*
    void message(int severity, const char *msg)
    {
        EventBuilderWrapper::This->Update(msg, severity);
    }*/
}

#endif
