#ifndef FACT_EventBuilderWrapper
#define FACT_EventBuilderWrapper

#include <sstream>

#if BOOST_VERSION < 104400
#if (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 4))
#undef BOOST_HAS_RVALUE_REFS
#endif
#endif
#include <boost/thread.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>


#include <CCfits/CCfits>

#include "EventBuilder.h"

extern "C" {
    extern void StartEvtBuild();
    extern int CloseRunFile(uint32_t runId, uint32_t closeTime, uint32_t maxEvt);
}

namespace ba = boost::asio;
namespace bs = boost::system;

using ba::ip::tcp;

using namespace std;

class DataFileImp : public MessageImp
{
    uint32_t fRunId;

    int Write(const Time &time, const std::string &txt, int qos)
    {
        return fMsg.Write(time, txt, qos);
    }

protected:
    MessageImp &fMsg;
    string fFileName;

public:
    DataFileImp(uint32_t id, MessageImp &imp) : fRunId(id), fMsg(imp) { }
    virtual ~DataFileImp() { }

    virtual bool OpenFile(RUN_HEAD* h) = 0;
    virtual bool WriteEvt(EVENT *) = 0;
    virtual bool Close(RUN_TAIL * = 0) = 0;

    const string &GetFileName() const { return fFileName; }

    uint32_t GetRunId() const { return fRunId; }

    // --------------------------------------------------------------------------
    //
    //! This creates an appropriate file name for a particular run number and type
    //! @param runNumber the run number for which a filename is to be created
    //! @param runType an int describing the kind of run. 0=Data, 1=Pedestal, 2=Calibration, 3=Calibrated data
    //! @param extension a string containing the extension to be appened to the file name
    //
    static string FormFileName(uint32_t runid, string extension)
    {
        ostringstream name;
        name << Time().NightAsInt() << '_' << setfill('0') << setw(3) << runid << '.' << extension;
        return name.str();
    }
};

class DataFileNone : public DataFileImp
{
public:
    DataFileNone(uint32_t id, MessageImp &imp) : DataFileImp(id, imp) { }

    Time fTime;

    bool OpenFile(RUN_HEAD* h)
    {
        fFileName = "/dev/null";

        ostringstream str;
        str << this << " - "
            << "OPEN_FILE #" << GetRunId() << ":"
            << " Ver=" << h->Version
            << " Typ=" << h->RunType
            << " Nb="  << h->NBoard
            << " Np="  << h->NPix
            << " NTm=" << h->NTm
            << " roi=" << h->Nroi;

        Debug(str);

        fTime = Time();

        return true;
    }
    bool WriteEvt(EVENT *e)
    {
        const Time now;
        if (now-fTime<boost::posix_time::seconds(5))
            return true;

        fTime = now;

        ostringstream str;
        str << this << " - EVENT #" << e->EventNum << " / " << e->TriggerNum;
        Debug(str);

        return true;
    }
    bool Close(RUN_TAIL * = 0)
    {
        ostringstream str;
        str << this << " - CLOSE FILE #" << GetRunId();

        Debug(str);

        return true;
    }
};

#include "DrsCalib.h"

class DataFileCalib : public DataFileImp, CalibData
{
    static vector<int32_t> fOffset;
    static vector<int32_t> fGain;
    static vector<int32_t> fTrgOff;

    static vector<float>   fStats;

    static uint32_t fNumOffset;
    static uint32_t fNumGain;
    static uint32_t fNumTrgOff;

    static int fStep;

    DimDescribedService &fDim;

public:
    DataFileCalib(uint32_t id, DimDescribedService &dim, MessageImp &imp) : DataFileImp(id, imp), fDim(dim)
    {
    }

    static void Restart()
    {
        // Default gain:
        // 0.575*[45590]*2.5V / 2^16 = 0.99999 V
        fOffset.assign(1440*1024, 0);
        fGain.assign  (1440*1024, 2);
        fTrgOff.assign(1440*1024, 0);

        fNumOffset = 1;
        fNumGain   = 1;
        fNumTrgOff = 1;

        reinterpret_cast<uint32_t*>(fStats.data())[0] = 0;
        reinterpret_cast<uint32_t*>(fStats.data())[1] = 0;
        reinterpret_cast<uint32_t*>(fStats.data())[2] = 0;

        int i=0;
        while (i<1024*1440*2+3)  // Set mean and RMS to 0
            fStats[i++] = 0;
        while (i<1024*1440*3+3)
            fStats[i++] = 0.5; // Set mean to 0.5
        while (i<1024*1440*6+3)
            fStats[i++] = 0;   // Set everything else to 0

        fStep = 0;
    }

    static void Update(DimDescribedService &dim)
    {
        dim.Update(fStats);
    }

    bool OpenFile(RUN_HEAD* h)
    {
        if (fStep==4)
        {
            fMsg.Warn("DRS Calibration already finished... please restart!");
            return false;
        }

        if (h->NPix != 1440)
        {
            fMsg.Error("Number of pixels in header not 1440.");
            return false;
        }

        if (h->Nroi != 1024)
        {
            fMsg.Error("Region of interest not 1024.");
            return false;
        }

        ostringstream name;
        name << "drs-calib-" << fStep;
        fFileName = name.str();

        Reset();
        InitSize(1440, 1024);

        return true;
    }
    bool WriteEvt(EVENT *e)
    {
        // FIXME: SET StartPix to 0 if StartPix is -1

        if (fStep==0)
        {
            AddRel(e->Adc_Data, e->StartPix);
        }
        if (fStep==1)
        {
            AddRel(e->Adc_Data, e->StartPix, fOffset.data(), fNumOffset);
        }
        if (fStep==2)
        {
            AddAbs(e->Adc_Data, e->StartPix, fOffset.data(), fNumOffset);
        }

        return true;
    }
    bool Close(RUN_TAIL * = 0)
    {
        if (fStep==0)
        {
            fOffset.assign(fSum.begin(), fSum.end());
            fNumOffset = fNumEntries;
            reinterpret_cast<uint32_t*>(fStats.data())[0] = GetRunId();;
            GetSampleStats(fStats.data()+3, 0.5);
        }
        if (fStep==1)
        {
            fGain.assign(fSum.begin(), fSum.end());
            fNumGain = fNumEntries*1000;
            reinterpret_cast<uint32_t*>(fStats.data())[1] = GetRunId();;
            // Scale from Volt to Millivolt
            GetSampleStats(fStats.data()+1024*1440*2+3, 0.5);
        }
        if (fStep==2)
        {
            fTrgOff.assign(fSum.begin(), fSum.end());
            fNumTrgOff = fNumEntries;
            reinterpret_cast<uint32_t*>(fStats.data())[2] = GetRunId();;
            GetSampleStats(fStats.data()+1024*1440*4+3, 0.5);
        }

        fDim.Update(fStats);

        fStep++;

        return true;
    }

    static void Apply(int16_t *val, const int16_t *start, uint32_t roi)
    {
        CalibData::Apply(val, start, roi,
                         fOffset.data(), fNumOffset,
                         fGain.data(),   fNumGain,
                         fTrgOff.data(), fNumTrgOff);
    }

    static void Apply(float *vec, int16_t *val, const int16_t *start, uint32_t roi)
    {
        CalibData::Apply(vec, val, start, roi,
                         fOffset.data(), fNumOffset,
                         fGain.data(),   fNumGain,
                         fTrgOff.data(), fNumTrgOff);
    }
};

int DataFileCalib::fStep = 0;

vector<int32_t> DataFileCalib::fOffset(1440*1024, 0);
vector<int32_t> DataFileCalib::fGain  (1440*1024, 2);
vector<int32_t> DataFileCalib::fTrgOff(1440*1024, 0);
vector<float>   DataFileCalib::fStats (1440*1024*6+3);

uint32_t DataFileCalib::fNumOffset = 1;
uint32_t DataFileCalib::fNumGain   = 1;
uint32_t DataFileCalib::fNumTrgOff = 1;


class DataFileDebug : public DataFileNone
{
public:
    DataFileDebug(uint32_t id, MessageImp &imp) : DataFileNone(id, imp) { }

    bool WriteEvt(EVENT *e)
    {
        cout << "WRITE_EVENT #" << GetRunId() << " (" << e->EventNum << ")" << endl;
        cout << " Typ=" << e->TriggerType << endl;
        cout << " roi=" << e->Roi << endl;
        cout << " trg=" << e->SoftTrig << endl;
        cout << " tim=" << e->PCTime << endl;

        return true;
    }
};

#include "FAD.h"

class DataFileRaw : public DataFileImp
{
    ofstream fOut;

    off_t fPosTail;

    uint32_t fCounter;


    // WRITE uint32_t 0xFAC77e1e  (FACT Tele)
    // ===
    // WRITE uint32_t TYPE(>0)          == 1
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE uint32_t TELESCOPE ID
    // WRITE uint32_t RUNID
    // ===
    // WRITE uint32_t TYPE(>0)          == 2
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE          HEADER
    // ===
    // [ 40 TIMES
    //    WRITE uint32_t TYPE(>0)       == 3
    //    WRITE uint32_t ID(>0)         == 0..39
    //    WRITE uint32_t VERSION(>0)    == 1
    //    WRITE uint32_t LENGTH
    //    -
    //    WRITE          BOARD-HEADER
    // ]
    // ===
    // WRITE uint32_t TYPE(>0)          == 4
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE          FOOTER (empty)
    // ===
    // [ N times
    //    WRITE uint32_t TYPE(>0)       == 10
    //    WRITE uint32_t ID(>0)         == counter
    //    WRITE uint32_t VERSION(>0)    == 1
    //    WRITE uint32_t LENGTH HEADER
    //    -
    //    WRITE          HEADER+DATA
    // ]
    // ===
    // WRITE uint32_t TYPE   ==0
    // WRITE uint32_t VERSION==0
    // WRITE uint32_t LENGTH ==0
    // ===
    // Go back and write footer

public:
    DataFileRaw(uint32_t id, MessageImp &imp) : DataFileImp(id, imp), fPosTail(0) { }
    ~DataFileRaw() { if (fOut.is_open()) Close(); }

    void WriteBlockHeader(uint32_t type, uint32_t ver, uint32_t cnt, uint32_t len)
    {
        const uint32_t val[4] = { type, ver, cnt, len };

        fOut.write(reinterpret_cast<const char*>(val), sizeof(val));
    }

    template<typename T>
        void WriteValue(const T &t)
    {
        fOut.write(reinterpret_cast<const char*>(&t), sizeof(T));
    }

    enum
    {
        kEndOfFile = 0,
        kIdentifier = 1,
        kRunHeader,
        kBoardHeader,
        kRunSummary,
        kEvent,
    };

    bool OpenFile(RUN_HEAD *h)
    {
        const string name = FormFileName(GetRunId(), "bin");
        if (access(name.c_str(), F_OK)==0)
        {
            Error("File '"+name+"' already exists.");
            return false;
        }

        fFileName = name;

        errno = 0;
        fOut.open(name.c_str(), ios_base::out);
        if (!fOut)
        {
            ostringstream str;
            str << "ofstream::open() failed for '" << name << "': " << strerror(errno) << " [errno=" << errno << "]";
            Error(str);

            return false;
        }

        fCounter = 0;

        static uint32_t FACT = 0xFAC77e1e;

        fOut.write(reinterpret_cast<char*>(&FACT), 4);

        WriteBlockHeader(kIdentifier, 1, 0, 8);
        WriteValue(uint32_t(0));
        WriteValue(GetRunId());

        WriteBlockHeader(kRunHeader, 1, 0, sizeof(RUN_HEAD)-sizeof(PEVNT_HEADER*));
        fOut.write(reinterpret_cast<char*>(h), sizeof(RUN_HEAD)-sizeof(PEVNT_HEADER*));

        for (int i=0; i<40; i++)
        {
            WriteBlockHeader(kBoardHeader, 1, i, sizeof(PEVNT_HEADER));
            fOut.write(reinterpret_cast<char*>(h->FADhead+i), sizeof(PEVNT_HEADER));
        }

        // FIXME: Split this
        const vector<char> block(sizeof(uint32_t)+sizeof(RUN_TAIL));
        WriteBlockHeader(kRunSummary, 1, 0, block.size());

        fPosTail = fOut.tellp();
        fOut.write(block.data(), block.size());

        if (!fOut)
        {
            ostringstream str;
            str << "ofstream::write() failed for '" << name << "': " << strerror(errno) << " [errno=" << errno << "]";
            Error(str);

            return false;
        }

        return true;
    }
    bool WriteEvt(EVENT *evt)
    {
        const int sh = sizeof(EVENT)-2 + NPIX*evt->Roi*2;

        WriteBlockHeader(kEvent, 1, fCounter++, sh);
        fOut.write(reinterpret_cast<char*>(evt)+2, sh);
        return true;
    }
    bool Close(RUN_TAIL *tail= 0)
    {
        WriteBlockHeader(kEndOfFile, 0, 0, 0);

        if (tail)
        {
            fOut.seekp(fPosTail);

            WriteValue(uint32_t(1));
            fOut.write(reinterpret_cast<char*>(tail), sizeof(RUN_TAIL));
        }

        if (!fOut)
        {
            ostringstream str;

            str << "ofstream::write() failed for '" << GetFileName() << "': " << strerror(errno) << " [errno=" << errno << "]";
            Error(str);

            return false;
        }

        fOut.close();

        if (!fOut)
        {
            ostringstream str;
            str << "ofstream::close() failed for '" << GetFileName() << "': " << strerror(errno) << " [errno=" << errno << "]";
            Error(str);

            return false;
        }

        return true;
    }
};

#ifdef HAVE_FITS
class DataFileFits : public DataFileImp
{
    CCfits::FITS*  fFile;        /// The pointer to the CCfits FITS file
    CCfits::Table* fTable;       /// The pointer to the CCfits binary table

    uint64_t fNumRows;                ///the number of rows that have been written already to the FITS file.

    Converter *fConv;

public:
    DataFileFits(uint32_t runid, MessageImp &imp) :
        DataFileImp(runid, imp), fFile(0), fNumRows(0), fConv(0)
    {
    }

    // --------------------------------------------------------------------------
    //
    //! Default destructor
    //! The Fits file SHOULD have been closed already, otherwise the informations
    //! related to the RUN_TAIL will NOT be written to the file.
    //
    ~DataFileFits() { Close(); delete fConv; }

    // --------------------------------------------------------------------------
    //
    //! Add a new column to the vectors storing the column data.
    //! @param names the vector of string storing the columns names
    //! @param types the vector of string storing the FITS data format
    //! @param numElems the number of elements in this column
    //! @param type the char describing the FITS data format
    //! @param name the name of the particular column to be added.
    //
    inline void AddColumnEntry(vector<string>& names, vector<string>& types, int numElems, char type, string name)
    {
        names.push_back(name);

        ostringstream str;
        if (numElems != 1)
            str << numElems;
        str << type;
        types.push_back(str.str());
    }

    // --------------------------------------------------------------------------
    //
    //! Writes a single header key entry
    //! @param name the name of the key
    //! @param value its value
    //! @param comment the comment associated to that key
    //
    //FIXME this function is a duplicate from the class Fits. should we try to merge it ?
    template <typename T>
    void WriteKey(const string &name, const T &value, const string &comment)
    {
        try
        {
            fTable->addKey(name, value, comment);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not add header key " << name;
            Error(str);
        }
    }

    template <typename T>
    void WriteKey(const string &name, const int idx, const T &value, const string &comment)
    {
        ostringstream str;
        str << name << idx;

        ostringstream com;
        com << "Board " << setw(2) << idx << ": " << comment;

        WriteKey(str.str(), value, com.str());
    }

    // --------------------------------------------------------------------------
    //
    //! DataFileFits constructor. This is the one that should be used, not the default one (parameter-less)
    //! @param runid This parameter should probably be removed. I first thought it was the run number, but apparently it is not
    //! @param h a pointer to the RUN_HEAD structure that contains the informations relative to this run
    //
    bool OpenFile(RUN_HEAD* h)
        {
        //Form filename, based on runid and run-type
        const string fileName = FormFileName(GetRunId(), "fits");
        if (access(fileName.c_str(), F_OK)==0)
        {
            Error("File '"+fileName+"' already exists.");
            return false;
        }

        fFileName = fileName;

        /*
         out <<
         "SIMPLE  =                    T / file does conform to FITS standard             "
         "BITPIX  =                    8 / number of bits per data pixel                  "
         "NAXIS   =                    0 / number of data axes                            "
         "EXTEND  =                    T / FITS dataset may contain extensions            "
         "COMMENT   FITS (Flexible Image Transport System) format is defined in 'Astronomy"
         "COMMENT   and Astrophysics', volume 376, page 359; bibcode: 2001A&A...376..359H "
         "END                                                                             ";
         for (int i=0; i<29; i++)
         out << "                                                                                "
         */

        //create the FITS object
        try
        {
            fFile = new CCfits::FITS(fileName, CCfits::RWmode::Write);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not open FITS file " << fileName << ": " << e.message();
            Error(str);
            return false;
        }

        vector<string> colNames;
        vector<string> dataTypes;
        AddColumnEntry(colNames, dataTypes, 1,                'J', "EventNum");
        AddColumnEntry(colNames, dataTypes, 1,                'J', "TriggerNum");
        AddColumnEntry(colNames, dataTypes, 1,                'I', "TriggerType");
        AddColumnEntry(colNames, dataTypes, 1,                'J', "NumBoards");
        AddColumnEntry(colNames, dataTypes, 1,                'J', "reserved");
        AddColumnEntry(colNames, dataTypes, 1,                'J', "SoftTrig");
        AddColumnEntry(colNames, dataTypes, 2,                'J', "PCTime");
        AddColumnEntry(colNames, dataTypes, NBOARDS/*h->NBoard*/, 'J', "BoardTime");
        AddColumnEntry(colNames, dataTypes, NPIX   /*h->NPix*/,   'I', "StartCellData");
        AddColumnEntry(colNames, dataTypes, NTMARK /*h->NTm*/,    'I', "StartCellTimeMarker");
        AddColumnEntry(colNames, dataTypes, h->NPix*h->Nroi,      'I', "Data");
        AddColumnEntry(colNames, dataTypes, h->NTm*h->NroiTM,     'I', "TimeMarker");

        // Write length of physical pipeline (1024)

        fConv = new Converter(Converter::ToFormat(dataTypes));

        const size_t sz = (h->NPix*h->Nroi + h->NTm*h->NroiTM)*2;
        if (fConv->GetSize()-sz+4!=sizeof(EVENT))
        {
            ostringstream str;
            str << "The EVENT structure size (" << sizeof(EVENT) << ") doesn't match the described FITS row (";
            str << fConv->GetSize()-sz+4 << ")";
            Error(str);
            return false;
        }

        //actually create the table
        try
        {
            fTable = fFile->addTable("Events", 0, colNames, dataTypes);
        }
        catch (const CCfits::FitsException &e)
        {
            ostringstream str;
            str << "Could not create FITS table 'Events' in file " << fileName << " reason: " << e.message();
            Error(str);
            return false;
        }

        if (fTable->rows() != 0)
        {
            Error("FITS table created on the fly looks non-empty.");
            return false;
        }

        //write header data
        //first the "standard" keys
        WriteKey("EXTREL",   1.0f,   "Release Number");
        WriteKey("TELESCOP", "FACT",    "Telescope that acquired this data");
        WriteKey("ORIGIN",   "ISDC",    "Institution that wrote the file");
        WriteKey("CREATOR",  "fadctrl", "Program that wrote this file (FACT++ Event Builder)");

        WriteKey("PACKAGE",   PACKAGE_NAME, "Package name");
        WriteKey("VERSION",   PACKAGE_VERSION, "Package description");
        WriteKey("COMPILED",  __DATE__" "__TIME__, "Compile time");
        WriteKey("REVISION",  REVISION, "SVN revision");
        //WriteKey("CONTACT",   PACKAGE_BUGREPORT, "Current package maintainer");
        //WriteKey("URL",       PACKAGE_URL, "Current repositiory location");

        WriteKey("BLDVER",   h->Version,  "Builder version");
        WriteKey("RUNID",    GetRunId(),  "Run number");
        WriteKey("RUNTYPE",  h->RunType,  "Type of run");
        WriteKey("NBOARD",   h->NBoard,   "Number of acquisition boards");
        WriteKey("NPIX",     h->NPix,     "Number of pixels");
        WriteKey("NTMARK",   h->NTm,      "Number of Time marks");
        WriteKey("NROI",     h->Nroi,     "Number of slices per pixels");
        WriteKey("NROITM",   h->NroiTM,   "Number of slices per time-marker");

        const Time now;
        WriteKey("TIMESYS",  "UTC",       "Time system");
        WriteKey("DATE",     now.Iso(),   "File creation date");
        WriteKey("NIGHT",    now.NightAsInt(), "Night as int");

        //FIXME should we also put the start and stop time of the received data ?
        //now the events header related variables
        WriteKey("CAMERA",   "MGeomCamFACT", "");
        WriteKey("DAQ",      "DRS4", "");

        WriteKey("ADCCOUNT",  2.0f,       "ADC Counts per milliVolt");

        Info("==> TODO: Write sampling frequency...");

        // Write a single key for:
        // -----------------------
        // Start package flag
        // package length
        // version number
        // status
        // Prescaler

        // Write 40 kays for (?)
        // Phaseshift
        // DAC

        for (int i=0; i<h->NBoard; i++)
        {
            const PEVNT_HEADER &hh = h->FADhead[i];

            // Header values whihc won't change during the run
            WriteKey("ID",    i, hh.board_id,   "Board ID");
            WriteKey("DNA",   i, hh.DNA,        "DNA");
            WriteKey("FWVER", i, hh.version_no, "Firmware Version");
        }


        /*
        //now the boards related keywords
        for (int i=0; i<h->NBoard; i++)
        {
            const PEVNT_HEADER &hh = h->FADhead[i];

            WriteKey("STPKGFG", i, hh.start_package_flag, "Start package flag");
            WriteKey("PKGLEN",  i, hh.package_length, "Package length");
            WriteKey("STATUS",  i, hh.PLLLCK, "");

//            WriteKey("TRIGCRC", i, hh.trigger_crc,      "Trigger CRC");
//            WriteKey("TRIGTYP", i, hh.trigger_type,     "Trigger type");
//            WriteKey("TRIGID",  i, hh.trigger_id,       "Trigger ID");
//            WriteKey("EVTCNTR", i, hh.fad_evt_counter,  "FAD Event Counter");
//            WriteKey("REFCLK",  i, hh.REFCLK_frequency, "Reference Clock Frequency");

            WriteKey("PHASESH", i, hh.adc_clock_phase_shift,          "ADC clock phase shift");
            WriteKey("TRGGEN",  i, hh.number_of_triggers_to_generate, "Number of triggers to generate");
            WriteKey("PRESC",   i, hh.trigger_generator_prescaler,    "Trigger generator prescaler");
            WriteKey("RUNNB",   i, hh.runnumber,                      "Run number");

            WriteKey("TIME",    i, hh.time,      "Time");

//            for (int j=0;j<NTemp;j++)
//            {
//                str.str(""); str2.str("");
//                str << "DRS_T" << i << j;
//                str2 << "DRS temperature #" << i << " " << j;
//                WriteKey(str.str(), h->FADhead[i].drs_temperature[j], str2.str());
//            }
            for (int j=0;j<NDAC;j++)
                WriteKey("DAC", i*NDAC+j, hh.dac[j], "DAC");
        }
*/

        //Last but not least, add header keys that will be updated when closing the file
        WriteFooter(NULL);

        return true;
    }


    int WriteColumns(size_t &start, size_t size, const void *e)
    {
        int status = 0;
        fits_write_tblbytes(fFile->fitsPointer(), fNumRows, start, size,
                            (unsigned char*)e, &status);
        if (status)
        {
            char text[30];//max length of cfitsio error strings (from doc)
            fits_get_errstatus(status, text);

            ostringstream str;
            str << "Writing FITS row " << fNumRows << ": " << text << " (file_write_tblbytes, rc=" << status << ")";
            Error(str);
        }

        start += size;
        return status;
    }

    // --------------------------------------------------------------------------
    //
    //! This writes one event to the file
    //! @param e the pointer to the EVENT
    //
    virtual bool WriteEvt(EVENT *e)
    {
        //FIXME As discussed earlier, we do not swap the bytes yet.
        fTable->makeThisCurrent();

        //insert a new row
        int status(0);
        if (fits_insert_rows(fTable->fitsPointer(), fNumRows, 1, &status))
        {
            char text[30];//max length of cfitsio error strings (from doc)
            fits_get_errstatus(status, text);

            ostringstream str;
            str << "Inserting row " << fNumRows << " into " << fFileName << ": " << text << " (fits_insert_rows, rc=" << status << ")";
            Error(str);

            return false;
        }
        fNumRows++;

        // FIXME: Get NPIX and NTMARK from header
        const size_t sz = sizeof(EVENT) + sizeof(e->StartPix)*e->Roi+sizeof(e->StartTM)*e->RoiTM;

        const vector<char> data = fConv->ToFits(reinterpret_cast<char*>(e)+4, sz-4);

        // column size pointer
        size_t col = 1;
        if (!WriteColumns(col, data.size(), data.data()))
            return true;

        //TODO output an error
        return false;

        /*
        //write the data, chunk by chunk
        //FIXME hard-coded size corresponds to current variables of the event, in bytes.
        //FIXME no padding was taken into account. Because smallest member is 2 bytes, I don't think that this should be a problem.
        const long sizeInBytesOfEventBeforePointers = 16;

        long col = 1;
        if (FitsWriteTblBytes(col, sizeInBytesOfEventBeforePointers, e))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NBOARDS*2, e->BoardTime))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*2, e->StartPix))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NTMARK*2, e->StartTM))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*fRoi*2, e->Adc_Data))
        {
            //TODO output an error
            return false;
        }
        return true;*/
    }

    void WriteFooter(RUN_TAIL *rt)
    {
        //write final header keys
        fTable->makeThisCurrent();

        WriteKey("NBEVTOK",  rt ? rt->nEventsOk  : uint32_t(0),
                 "How many events were written");

        WriteKey("NBEVTREJ", rt ? rt->nEventsRej : uint32_t(0),
                 "How many events were rejected by SW-trig");

        WriteKey("NBEVTBAD", rt ? rt->nEventsBad : uint32_t(0),
                 "How many events were rejected by Error");

        //FIXME shouldn't we convert start and stop time to MjD first ?
        //FIXME shouldn't we also add an MjD reference ?

        WriteKey("TSTART",   rt ? rt->PCtime0    : uint32_t(0),
                 "Time when first event received");

        WriteKey("TSTOP",    rt ? rt->PCtimeX    : uint32_t(0),
                 "Time when last event received");
    }

    // --------------------------------------------------------------------------
    //
    //! Closes the file, and before this it write the TAIL data
    //! @param rt the pointer to the RUN_TAIL data structure
    //
    virtual bool Close(RUN_TAIL *rt = 0)
    {
        if (!fFile)
            return false;

        WriteFooter(rt);

        delete fFile;
        fFile = NULL;

        return true;
    }

};
#else
#define DataFileFits DataFileRaw
#endif

#include "DimWriteStatistics.h"

class EventBuilderWrapper
{
public:
    // FIXME
    static EventBuilderWrapper *This;

    MessageImp &fMsg;

private:
    boost::thread fThread;

    enum CommandStates_t // g_runStat
    {
        kAbort      = -2,   // quit as soon as possible ('abort')
        kExit       = -1,   // stop reading, quit when buffered events done ('exit')
        kInitialize =  0,   // 'initialize' (e.g. dim not yet started)
        kHybernate  =  1,   // do nothing for long time ('hybernate') [wakeup within ~1sec]
        kSleep      =  2,   // do nothing ('sleep')                   [wakeup within ~10msec]
        kModeFlush  = 10,   // read data from camera, but skip them ('flush')
        kModeTest   = 20,   // read data and process them, but do not write to disk ('test')
        kModeFlag   = 30,   // read data, process and write all to disk ('flag')
        kModeRun    = 40,   // read data, process and write selected to disk ('run')
    };

    enum
    {
        kCurrent   = 0,
        kTotal     = 1,
        kEventId   = 2,
        kTriggerId = 3,
    };

    enum FileFormat_t
    {
        kNone = 0,
        kDebug,
        kFits,
        kRaw,
        kCalib
    };

    FileFormat_t fFileFormat;


    uint32_t fMaxRun;
    uint32_t fLastOpened;
    uint32_t fLastClosed;
    uint32_t fNumEvts[4];

    DimWriteStatistics  fDimWriteStats;
    DimDescribedService fDimRuns;
    DimDescribedService fDimEvents;
    DimDescribedService fDimEventData;
    DimDescribedService fDimFwVersion;
    DimDescribedService fDimRunNumber;
    DimDescribedService fDimStatus;
    DimDescribedService fDimDNA;
    DimDescribedService fDimTemperature;
    DimDescribedService fDimPrescaler;
    DimDescribedService fDimRefClock;
    DimDescribedService fDimRoi;
    DimDescribedService fDimDac;
    DimDescribedService fDimDrsCalibration;
    DimDescribedService fDimStatistics1;
    DimDescribedService fDimStatistics2;

    bool fDebugStream;
    bool fDebugRead;
    bool fDebugLog;

    uint32_t fRunNumber;

    void InitRunNumber()
    {
        // FIXME: Add a check that we are not too close to noon!
        //const int night = Time().NightAsInt();

        fRunNumber = 1000;

        while (--fRunNumber>0)
        {
            const string name = DataFileImp::FormFileName(fRunNumber, "");

            if (access((name+"bin").c_str(), F_OK) == 0)
                break;
            if (access((name+"fits").c_str(), F_OK) == 0)
                break;
        }

        fRunNumber++;

        ostringstream str;
        str << "Starting with run number " << fRunNumber;
        fMsg.Message(str);

        fMsg.Info(" ==> TODO: Run-number detection doesn't work when noon passes!");
        fMsg.Info(" ==> TODO: Crosscheck with database!");
    }

public:
    EventBuilderWrapper(MessageImp &imp) : fMsg(imp),
        fFileFormat(kNone), fMaxRun(0), fLastOpened(0), fLastClosed(0),
        fDimWriteStats  ("FAD_CONTROL", imp),
        fDimRuns        ("FAD_CONTROL/RUNS",               "I:5;C", ""),
        fDimEvents      ("FAD_CONTROL/EVENTS",             "I:4", ""),
        fDimEventData   ("FAD_CONTROL/EVENT_DATA",         "S:1;I:1;S:1;I:1;I:2;I:40;S:1440;S:160;F", ""),
        fDimFwVersion   ("FAD_CONTROL/FIRMWARE_VERSION",   "F:42", ""),
        fDimRunNumber   ("FAD_CONTROL/RUN_NUMBER",         "I:42", ""),
        fDimStatus      ("FAD_CONTROL/STATUS",             "S:42", ""),
        fDimDNA         ("FAD_CONTROL/DNA",                "X:40", ""),
        fDimTemperature ("FAD_CONTROL/TEMPERATURE",        "F:82", ""),
        fDimPrescaler   ("FAD_CONTROL/PRESCALER",          "S:42", ""),
        fDimRefClock    ("FAD_CONTROL/REFERENCE_CLOCK",    "I:42", ""),
        fDimRoi         ("FAD_CONTROL/REGION_OF_INTEREST", "S:2",  ""),
        fDimDac         ("FAD_CONTROL/DAC",                "S:336", ""),
        fDimDrsCalibration("FAD_CONTROL/DRS_CALIBRATION",  "I:3;F:1474560;F:1474560;F:1474560;F:1474560;F:1474560;F:1474560", ""),
        fDimStatistics1 ("FAD_CONTROL/STATISTICS1",        "I:3;I:5;X:4;I:3;I:3;I:40;I:1;I:2;C:40;I:40;I:40;X:40", ""),
        fDimStatistics2 ("FAD_CONTROL/STATISTICS2",        "I:1;I:280;X:40;I:40;I:4;I:4;I:2;I:2;I:3;C:40",  ""),
        fDebugStream(false), fDebugRead(false), fDebugLog(false)
    {
        if (This)
            throw logic_error("EventBuilderWrapper cannot be instantiated twice.");

        This = this;

        memset(fNumEvts, 0, sizeof(fNumEvts));

        fDimEvents.Update(fNumEvts);

        for (size_t i=0; i<40; i++)
            ConnectSlot(i, tcp::endpoint());

        InitRunNumber();
    }
    virtual ~EventBuilderWrapper()
    {
        Abort();

        // FIXME: Used timed_join and abort afterwards
        //        What's the maximum time the eb need to abort?
        fThread.join();
        //ffMsg.Info("EventBuilder stopped.");

        for (vector<DataFileImp*>::iterator it=fFiles.begin(); it!=fFiles.end(); it++)
            delete *it;
    }

    struct RunDescription
    {
        uint32_t maxtime;
        uint32_t maxevt;

        FAD::Configuration reference;
    };

    map<uint32_t, RunDescription> fExpectedRuns;

    uint32_t StartNewRun(int64_t maxtime, int64_t maxevt, const FAD::Configuration &ref)
    {
        if (maxtime<=0 || maxtime>24*60*60)
            maxtime = 24*60*60;
        if (maxevt<=0 || maxevt>INT32_MAX)
            maxevt  = INT32_MAX;

        const RunDescription descr =
        {
            uint32_t(maxtime),
            uint32_t(maxevt),
            ref
        };

        // FIMXE: Maybe reset an event counter so that the mcp can count events?

        fExpectedRuns[fRunNumber] = descr;
        return fRunNumber++;
    }

    bool IsThreadRunning()
    {
        return !fThread.timed_join(boost::posix_time::microseconds(0));
    }

    void SetMaxMemory(unsigned int mb) const
    {
        /*
        if (mb*1000000<GetUsedMemory())
        {
            // ffMsg.Warn("...");
            return;
        }*/

        g_maxMem = size_t(mb)*1000000;
    }

    void StartThread(const vector<tcp::endpoint> &addr)
    {
        if (IsThreadRunning())
        {
            fMsg.Warn("Start - EventBuilder still running");
            return;
        }

        fLastMessage.clear();

        for (size_t i=0; i<40; i++)
            ConnectSlot(i, addr[i]);

        g_runStat = kModeRun;

        fMsg.Message("Starting EventBuilder thread");

        fThread = boost::thread(StartEvtBuild);
    }
    void ConnectSlot(unsigned int i, const tcp::endpoint &addr)
    {
        if (i>39)
            return;

        if (addr==tcp::endpoint())
        {
            DisconnectSlot(i);
            return;
        }

        g_port[i].sockAddr.sin_family      = AF_INET;
        g_port[i].sockAddr.sin_addr.s_addr = htonl(addr.address().to_v4().to_ulong());
        g_port[i].sockAddr.sin_port        = htons(addr.port());
        // In this order
        g_port[i].sockDef = 1;
    }
    void DisconnectSlot(unsigned int i)
    {
        if (i>39)
            return;

        g_port[i].sockDef = 0;
        // In this order
        g_port[i].sockAddr.sin_family      = AF_INET;
        g_port[i].sockAddr.sin_addr.s_addr = 0;
        g_port[i].sockAddr.sin_port        = 0;
    }
    void IgnoreSlot(unsigned int i)
    {
        if (i>39)
            return;
        if (g_port[i].sockAddr.sin_port==0)
            return;

        g_port[i].sockDef = -1;
    }


    void Abort()
    {
        fMsg.Message("Signal abort to EventBuilder thread...");
        g_runStat = kAbort;
    }

    void ResetThread(bool soft)
    {
        /*
         if (g_reset > 0)

            * suspend reading
            * reset = g_reset;
            * g_reset=0

            * reset% 10
                == 0  leave event Buffers as they are
                == 1  let all buffers drain (write (incomplete) events)
                >  1  flush all buffers (do not write buffered events)

            * (reset/10)%10
                > 0   close all sockets and destroy them (also free the
                      allocated read-buffers)
                      recreate before resuming operation
                      [ this is more than just close/open that can be
                        triggered by e.g. close/open the base-socket ]

            * (reset/100)%10
                > 0   close all open run-files

            * (reset/1000)
                      sleep so many seconds before resuming operation
                      (does not (yet) take into account time left when waiting
                      for buffers getting empty ...)

            * resume_reading

         */
        fMsg.Message("Signal reset to EventBuilder thread...");
        g_reset = soft ? 101 : 102;
    }

    void Exit()
    {
        fMsg.Message("Signal exit to EventBuilder thread...");
        g_runStat = kExit;
    }

    /*
    void Wait()
    {
        fThread.join();
        ffMsg.Message("EventBuilder stopped.");
    }*/

    void Hybernate() const { g_runStat = kHybernate; }
    void Sleep()     const { g_runStat = kSleep;     }
    void FlushMode() const { g_runStat = kModeFlush; }
    void TestMode()  const { g_runStat = kModeTest;  }
    void FlagMode()  const { g_runStat = kModeFlag;  }
    void RunMode()   const { g_runStat = kModeRun;   }

    // FIXME: To be removed
    void SetMode(int mode) const { g_runStat = mode; }

    bool IsConnected(int i) const     { return gi_NumConnect[i]==7; }
    bool IsConnecting(int i) const    { return !IsConnected(i) && !IsDisconnected(i); }
    bool IsDisconnected(int i) const  { return gi_NumConnect[i]<=0 && g_port[i].sockDef==0; }
    int  GetNumConnected(int i) const { return gi_NumConnect[i]; }

    void SetIgnore(int i, bool b) const { if (g_port[i].sockDef!=0) g_port[i].sockDef=b?-1:1; }
    bool IsIgnored(int i) const { return g_port[i].sockDef==-1; }

    void SetOutputFormat(FileFormat_t f)
    {
        fFileFormat = f;
        if (fFileFormat==kCalib)
        {
            DataFileCalib::Restart();
            DataFileCalib::Update(fDimDrsCalibration);
            fMsg.Message("Resetted DRS calibration.");
        }
    }

    void SetDebugLog(bool b) { fDebugLog = b; }

    void SetDebugStream(bool b)
    {
        fDebugStream = b;
        if (b)
            return;

        for (int i=0; i<40; i++)
        {
            if (!fDumpStream[i].is_open())
                continue;

            fDumpStream[i].close();

            ostringstream name;
            name << "socket_dump-" << setfill('0') << setw(2) << i << ".bin";
            fMsg.Message("Closed file '"+name.str()+"'");
        }
    }

    void SetDebugRead(bool b)
    {
        fDebugRead = b;
        if (b || !fDumpRead.is_open())
            return;

        fDumpRead.close();
        fMsg.Message("Closed file 'socket_events.txt'");
    }

//    size_t GetUsedMemory() const { return gi_usedMem; }

    virtual int CloseOpenFiles() { CloseRunFile(0, 0, 0); return 0; }


    /*
     struct OpenFileToDim
     {
        int code;
        char fileName[FILENAME_MAX];
     };

     SignalRunOpened(runid, filename);
     // Send num open files
     // Send runid, (more info about the run?), filename via dim

     SignalEvtWritten(runid);
     // Send num events written of newest file

     SignalRunClose(runid);
     // Send new num open files
     // Send empty file-name if no file is open

     */

    // -------------- Mapped event builder callbacks ------------------

    void UpdateRuns(const string &fname="")
    {
        uint32_t values[5] =
        {
            static_cast<uint32_t>(fFiles.size()),
            0xffffffff,
            0,
            fLastOpened,
            fLastClosed
        };

        for (vector<DataFileImp*>::const_iterator it=fFiles.begin();
             it!=fFiles.end(); it++)
        {
            const DataFileImp *file = *it;

            if (file->GetRunId()<values[1])
                values[1] = file->GetRunId();

            if (file->GetRunId()>values[2])
                values[2] = file->GetRunId();
        }

        fMaxRun = values[2];

        vector<char> data(sizeof(values)+fname.size()+1);
        memcpy(data.data(), values, sizeof(values));
        strcpy(data.data()+sizeof(values), fname.c_str());

        fDimRuns.Update(data);
    }

    vector<DataFileImp*> fFiles;

    FileHandle_t runOpen(uint32_t runid, RUN_HEAD *h, size_t)
    {
        fMsg.Info(" ==> TODO: Update run configuration in database!");

        // Check if file already exists...
        DataFileImp *file = 0;
        switch (fFileFormat)
        {
        case kNone:  file = new DataFileNone(runid,  fMsg); break;
        case kDebug: file = new DataFileDebug(runid, fMsg); break;
        case kFits:  file = new DataFileFits(runid,  fMsg); break;
	case kRaw:   file = new DataFileRaw(runid,   fMsg); break;
	case kCalib: file = new DataFileCalib(runid, fDimDrsCalibration, fMsg); break;
        }

        try
        {
            if (!file->OpenFile(h))
                return 0;
        }
        catch (const exception &e)
        {
            fMsg.Error("Exception trying to open file: "+string(e.what()));
            return 0;
        }

        fFiles.push_back(file);

        ostringstream str;
        str << "Opened: " << file->GetFileName() << " (" << file->GetRunId() << ")";
        fMsg.Info(str);

        fDimWriteStats.FileOpened(file->GetFileName());

        fLastOpened = runid;
        UpdateRuns(file->GetFileName());

        fNumEvts[kEventId] = 0;
        fNumEvts[kTriggerId] = 0;

        fNumEvts[kCurrent] = 0;
        fDimEvents.Update(fNumEvts);
        // fDimCurrentEvent.Update(uint32_t(0));

        return reinterpret_cast<FileHandle_t>(file);
    }

    int runWrite(FileHandle_t handler, EVENT *e, size_t)
    {
        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

        if (!file->WriteEvt(e))
            return -1;

        if (file->GetRunId()==fMaxRun)
        {
            fNumEvts[kCurrent]++;
            fNumEvts[kEventId]   = e->EventNum;
            fNumEvts[kTriggerId] = e->TriggerNum;
        }

        fNumEvts[kTotal]++;

        static Time oldt(boost::date_time::neg_infin);
        Time newt;
        if (newt>oldt+boost::posix_time::seconds(1))
        {
            fDimEvents.Update(fNumEvts);
            oldt = newt;
        }


        // ===> SignalEvtWritten(runid);
        // Send num events written of newest file

        /* close run runId (all all runs if runId=0) */
        /* return: 0=close scheduled / >0 already closed / <0 does not exist */
        //CloseRunFile(file->GetRunId(), time(NULL)+2) ;

        return 0;
    }

    int runClose(FileHandle_t handler, RUN_TAIL *tail, size_t)
    {
        fMsg.Info(" ==> TODO: Update run configuration in database!");

        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

        const vector<DataFileImp*>::iterator it = find(fFiles.begin(), fFiles.end(), file);
        if (it==fFiles.end())
        {
            ostringstream str;
            str << "File handler (" << handler << ") requested to close by event builder doesn't exist.";
            fMsg.Fatal(str);
            return -1;
        }

        fFiles.erase(it);

        fLastClosed = file->GetRunId();
        UpdateRuns();

        fDimEvents.Update(fNumEvts);

        const bool rc = file->Close(tail);
        if (!rc)
        {
            // Error message
        }

        ostringstream str;
        str << "Closed: " << file->GetFileName() << " (" << file->GetRunId() << ")";
        fMsg.Info(str);

        delete file;

        // ==> SignalRunClose(runid);
        // Send new num open files
        // Send empty file-name if no file is open

        return rc ? 0 : -1;
    }

    ofstream fDumpStream[40];

    void debugStream(int isock, void *buf, int len)
    {
        if (!fDebugStream)
            return;

        const int slot = isock/7;
        if (slot<0 || slot>39)
            return;

        if (!fDumpStream[slot].is_open())
        {
            ostringstream name;
            name << "socket_dump-" << setfill('0') << setw(2) << slot << ".bin";

            fDumpStream[slot].open(name.str().c_str(), ios::app);
            if (!fDumpStream[slot])
            {
                ostringstream str;
                str << "Open file '" << name << "': " << strerror(errno) << " (errno=" << errno << ")";
                fMsg.Error(str);

                return;
            }

            fMsg.Message("Opened file '"+name.str()+"' for writing.");
        }

        fDumpStream[slot].write(reinterpret_cast<const char*>(buf), len);
    }

    ofstream fDumpRead; // Stream to possibly dump docket events

    void debugRead(int isock, int ibyte, uint32_t event, uint32_t ftmevt, uint32_t runno, int state, uint32_t tsec, uint32_t tusec)
    {
        //   isock = socketID (0-279)
        //   ibyte = #bytes gelesen
        //   event = eventId (oder 0 wenn noch nicht bekannt)
        //   state : 1=finished reading data
        //           0=reading data
        //          -1=start reading data (header)
        //          -2=start reading data,
        //             eventId not known yet (too little data)
        //   tsec, tusec = time when reading seconds, microseconds
        //
        if (!fDebugRead || ibyte==0)
	    return;

        if (!fDumpRead.is_open())
        {
            fDumpRead.open("socket_events.txt", ios::app);
            if (!fDumpRead)
            {
                ostringstream str;
                str << "Open file 'socket_events.txt': " << strerror(errno) << " (errno=" << errno << ")";
                fMsg.Error(str);

                return;
            }

            fMsg.Message("Opened file 'socket_events.txt' for writing.");

            fDumpRead << "# START: " << Time().GetAsStr() << endl;
            fDumpRead << "# state time_sec time_usec socket slot runno event_id trigger_id bytes_received" << endl;
        }

        fDumpRead
            << setw(2) << state   << " "
            << setw(8) << tsec    << " "
            << setw(9) << tusec   << " "
            << setw(3) << isock   << " "
            << setw(2) << isock/7 << " "
            << runno  << " "
            << event  << " "
            << ftmevt << " "
            << ibyte << endl;
    }

    array<uint16_t,2> fVecRoi;

    int eventCheck(PEVNT_HEADER *fadhd, EVENT *event)
    {
        /*
         fadhd[i] ist ein array mit den 40 fad-headers
         (falls ein board nicht gelesen wurde, ist start_package_flag =0 )

         event  ist die Struktur, die auch die write routine erhaelt;
         darin sind im header die 'soll-werte' fuer z.B. eventID
         als auch die ADC-Werte (falls Du die brauchst)

         Wenn die routine einen negativen Wert liefert, wird das event
         geloescht (nicht an die write-routine weitergeleitet [mind. im Prinzip]
         */

	const array<uint16_t,2> roi = {{ event->Roi, event->RoiTM }};

	if (roi!=fVecRoi)
	{
	    Update(fDimRoi, roi);
	    fVecRoi = roi;
	}

        const FAD::EventHeader *beg = reinterpret_cast<FAD::EventHeader*>(fadhd);
        const FAD::EventHeader *end = reinterpret_cast<FAD::EventHeader*>(fadhd)+40;

        for (const FAD::EventHeader *ptr=beg; ptr!=end; ptr++)
        {
            // Event incomplete
            //if (ptr->fStartDelimiter==0)
            //    return -1;

                // Either one of
                //   * fStatus
                //   * fRunNumber
                //   * fEventCounter
                //   * fAdcClockPhaseShift
                //   * fTriggerGeneratorPrescaler
                //   * fDac
	    // inconsistent

	    // FIXME: Produce some output, only once per run or
            //        minute

            /*
            if (*ptr != *beg)
                return -1;

            if (ptr->fTriggerType != beg->fTriggerType)
                return -1;
            if (ptr->fTriggerId   != beg->fTriggerId)
                return -1;
            if (ptr->fVersion     != beg->fVersion)
	    return -1;
            */
        }

        // check REFCLK_frequency
        // check consistency with command configuration
        // how to log errors?
        // need gotNewRun/closedRun to know it is finished

        static Time oldt(boost::date_time::neg_infin);
        Time newt;

        // FIXME: Only send events if the have newer run-numbers
        if (newt<oldt+boost::posix_time::seconds(1))
            return 0;

        oldt = newt;

	const size_t sz = sizeof(EVENT)+event->Roi*2*1440;

        vector<char> data(sz+event->Roi*2*1440);
        memcpy(data.data(), event, sizeof(EVENT));

        DataFileCalib::Apply(reinterpret_cast<float*>(data.data()+sizeof(EVENT)),
                             event->Adc_Data, event->StartPix, event->Roi);

        fDimEventData.Update(data);

	return 0;
    }

    bool IsRunStarted() const
    {
        return fExpectedRuns.find(fRunNumber-1)==fExpectedRuns.end();
    }

    void gotNewRun(int runnr, PEVNT_HEADER */*headers*/)
    {
        // This function is called even when writing is switched off
        const map<uint32_t,RunDescription>::iterator it = fExpectedRuns.find(runnr);
        if (it==fExpectedRuns.end())
        {
            ostringstream str;
            str << "gotNewRun - Run " << runnr << " wasn't expected." << endl;
            return;
        }

        CloseRunFile(runnr, time(NULL)+it->second.maxtime, it->second.maxevt);
        // return: 0=close scheduled / >0 already closed / <0 does not exist

        fExpectedRuns.erase(it);
    }

    map<boost::thread::id, string> fLastMessage;

    void factOut(int severity, int err, const char *message)
    {
        if (!fDebugLog && severity==99)
            return;

        ostringstream str;
        //str << boost::this_thread::get_id() << " ";
        str << "EventBuilder(";
        if (err<0)
            str << "---";
        else
            str << err;
        str << "): " << message;

        string &old = fLastMessage[boost::this_thread::get_id()];

        if (str.str()==old)
            return;
        old = str.str();

        fMsg.Update(str, severity);
    }
/*
    void factStat(int64_t *stat, int len)
    {
        if (len!=7)
        {
            fMsg.Warn("factStat received unknown number of values.");
            return;
        }

        vector<int64_t> data(1, g_maxMem);
        data.insert(data.end(), stat, stat+len);

        static vector<int64_t> last(8);
        if (data==last)
            return;
        last = data;

        fDimStatistics.Update(data);

        //   len ist die Laenge des arrays.
        //   array[4] enthaelt wieviele bytes im Buffer aktuell belegt sind; daran
        //   kannst Du pruefen, ob die 100MB voll sind ....

        ostringstream str;
        str
            << "Wait=" << stat[0] << " "
            << "Skip=" << stat[1] << " "
            << "Del="  << stat[2] << " "
            << "Tot="  << stat[3] << " "
            << "Mem="  << stat[4] << "/" << g_maxMem << " "
            << "Read=" << stat[5] << " "
            << "Conn=" << stat[6];

        fMsg.Info(str);
    }
    */

    void factStat(const EVT_STAT &stat)
    {
        fDimStatistics2.Update(stat);
        /*
         //some info about what happened since start of program (or last 'reset')
         uint32_t reset ;             //#if increased, reset all counters
         uint32_t numRead[MAX_SOCK] ; //how often succesfull read from N sockets per loop

         uint64_t gotByte[NBOARDS] ;  //#Bytes read per Board
         uint32_t gotErr[NBOARDS] ;   //#Communication Errors per Board
         uint32_t evtGet;             //#new Start of Events read
         uint32_t evtTot;             //#complete Events read
         uint32_t evtErr;             //#Events with Errors
         uint32_t evtSkp;             //#Events incomplete (timeout)

         uint32_t procTot;            //#Events processed
         uint32_t procErr;            //#Events showed problem in processing
         uint32_t procTrg;            //#Events accepted by SW trigger
         uint32_t procSkp;            //#Events rejected by SW trigger

         uint32_t feedTot;            //#Events used for feedBack system
         uint32_t feedErr;            //#Events rejected by feedBack

         uint32_t wrtTot;             //#Events written to disk
         uint32_t wrtErr;             //#Events with write-error

         uint32_t runOpen;            //#Runs opened
         uint32_t runClose;           //#Runs closed
         uint32_t runErr;             //#Runs with open/close errors


         //info about current connection status
         uint8_t  numConn[NBOARDS] ;  //#Sockets succesfully open per board
         */
    }

    void factStat(const GUI_STAT &stat)
    {
        fDimStatistics1.Update(stat);
        /*
         //info about status of the main threads
         int32_t  readStat ;          //read thread
         int32_t  procStat ;          //processing thread(s)
         int32_t  writStat ;          //write thread

         //info about some rates
         int32_t  deltaT ;            //time in milli-seconds for rates
         int32_t  readEvt ;           //#events read
         int32_t  procEvt ;           //#events processed
         int32_t  writEvt ;           //#events written
         int32_t  skipEvt ;           //#events skipped

         //some info about current state of event buffer (snapspot)
         int32_t evtBuf;             //#Events currently waiting in Buffer
         uint64_t totMem;             //#Bytes available in Buffer
         uint64_t usdMem;             //#Bytes currently used
         uint64_t maxMem;             //max #Bytes used during past Second
         */
    }


    array<FAD::EventHeader, 40> fVecHeader;

    template<typename T, class S>
    array<T, 42> Compare(const S *vec, const T *t)
    {
        const int offset = reinterpret_cast<const char *>(t) - reinterpret_cast<const char *>(vec);

        const T *min = NULL;
        const T *val = NULL;
        const T *max = NULL;

        array<T, 42> arr;

        bool rc = true;
        for (int i=0; i<40; i++)
        {
            const char *base = reinterpret_cast<const char*>(vec+i);
            const T *ref = reinterpret_cast<const T*>(base+offset);

            arr[i] = *ref;

            if (gi_NumConnect[i]!=7)
            {
                arr[i] = 0;
                continue;
            }

            if (!val)
            {
                min = ref;
                val = ref;
                max = ref;
            }

            if (*ref<*min)
                min = ref;

            if (*ref>*max)
                max = ref;

            if (*val!=*ref)
                rc = false;
        }

        arr[40] = val ? *min : 1;
        arr[41] = val ? *max : 0;

        return arr;
    }

    template<typename T>
    array<T, 42> CompareBits(const FAD::EventHeader *h, const T *t)
    {
        const int offset = reinterpret_cast<const char *>(t) - reinterpret_cast<const char *>(h);

        T val = 0;
        T rc  = 0;

        array<T, 42> vec;

        bool first = true;

        for (int i=0; i<40; i++)
        {
            const char *base = reinterpret_cast<const char*>(&fVecHeader[i]);
            const T *ref = reinterpret_cast<const T*>(base+offset);

            vec[i+2] = *ref;

            if (gi_NumConnect[i]!=7)
            {
                vec[i+2] = 0;
                continue;
            }

            if (first)
            {
                first = false;
                val = *ref;
                rc = 0;
            }

            rc |= val^*ref;
        }

        vec[0] = rc;
        vec[1] = val;

        return vec;
    }

    template<typename T, size_t N>
    void Update(DimDescribedService &svc, const array<T, N> &data, int n=N)
    {
//        svc.setQuality(vec[40]<=vec[41]);
        svc.setData(const_cast<T*>(data.data()), sizeof(T)*n);
        svc.updateService();
    }

    template<typename T>
        void Print(const char *name, const pair<bool,array<T, 43>> &data)
    {
        cout << name << "|" << data.first << "|" << data.second[1] << "|" << data.second[0] << "<x<" << data.second[1] << ":";
        for (int i=0; i<40;i++)
            cout << " " << data.second[i+3];
        cout << endl;
    }

    vector<uint> fNumConnected;

    void debugHead(int /*socket*/, const FAD::EventHeader &h)
    {
        const uint16_t id = h.Id();
        if (id>39) 
            return;

        if (fNumConnected.size()!=40)
	    fNumConnected.resize(40);

	const vector<uint> con(gi_NumConnect, gi_NumConnect+40);

	const bool changed = con!=fNumConnected || !IsThreadRunning();

        fNumConnected = con;

	const FAD::EventHeader old = fVecHeader[id];
        fVecHeader[id] = h;

        if (old.fVersion != h.fVersion || changed)
        {
            const array<uint16_t,42> ver = Compare(&fVecHeader[0], &fVecHeader[0].fVersion);

            array<float,42> data;
            for (int i=0; i<42; i++)
            {
                ostringstream str;
                str << (ver[i]>>8) << '.' << (ver[i]&0xff);
                data[i] = stof(str.str());
            }
            Update(fDimFwVersion, data);
        }

        if (old.fRunNumber != h.fRunNumber || changed)
        {
            const array<uint32_t,42> run = Compare(&fVecHeader[0], &fVecHeader[0].fRunNumber);
            fDimRunNumber.Update(run);
        }

        if (old.fTriggerGeneratorPrescaler != h.fTriggerGeneratorPrescaler || changed)
        {
            const array<uint16_t,42> pre = Compare(&fVecHeader[0], &fVecHeader[0].fTriggerGeneratorPrescaler);
            fDimPrescaler.Update(pre);
        }

        if (old.fDNA != h.fDNA || changed)
        {
            const array<uint64_t,42> dna = Compare(&fVecHeader[0], &fVecHeader[0].fDNA);
            Update(fDimDNA, dna, 40);
        }

        if (old.fStatus != h.fStatus || changed)
        {
            const array<uint16_t,42> sts = CompareBits(&fVecHeader[0], &fVecHeader[0].fStatus);
            Update(fDimStatus, sts);
        }

        if (memcmp(old.fDac, h.fDac, sizeof(h.fDac)) || changed)
        {
            array<uint16_t, FAD::kNumDac*42> dacs;

            for (int i=0; i<FAD::kNumDac; i++)
            {
                const array<uint16_t, 42> dac = Compare(&fVecHeader[0], &fVecHeader[0].fDac[i]);
                memcpy(&dacs[i*42], &dac[0], sizeof(uint16_t)*42);
            }

            Update(fDimDac, dacs);
        }

        // -----------

        static Time oldt(boost::date_time::neg_infin);
        Time newt;

        if (newt>oldt+boost::posix_time::seconds(1))
        {
            oldt = newt;

            // --- RefClock

            const array<uint32_t,42> clk = Compare(&fVecHeader[0], &fVecHeader[0].fFreqRefClock);
            Update(fDimRefClock, clk);

            // --- Temperatures

            const array<int16_t,42> tmp[4] =
            {
                Compare(&fVecHeader[0], &fVecHeader[0].fTempDrs[0]),    // 0-39:val, 40:min, 41:max
                Compare(&fVecHeader[0], &fVecHeader[0].fTempDrs[1]),    // 0-39:val, 40:min, 41:max
                Compare(&fVecHeader[0], &fVecHeader[0].fTempDrs[2]),    // 0-39:val, 40:min, 41:max
                Compare(&fVecHeader[0], &fVecHeader[0].fTempDrs[3])     // 0-39:val, 40:min, 41:max
            };

            vector<int16_t> data;
            data.reserve(82);
            data.push_back(tmp[0][40]);                                 // min: 0
            data.insert(data.end(), tmp[0].data(), tmp[0].data()+40);   // val: 1-40
            data.push_back(tmp[0][41]);                                 // max: 41
            data.insert(data.end(), tmp[0].data(), tmp[0].data()+40);   // val: 42-81

            for (int j=1; j<=3; j++)
            {
                const array<int16_t,42> &ref = tmp[j];

                // Gloabl min
                if (ref[40]<data[0])           // 40=min
                    data[0] = ref[40];

                // Global max
                if (ref[41]>data[41])          // 41=max
                    data[41] = ref[41];

                for (int i=0; i<40; i++)
                {
                    // min per board
                    if (ref[i]<data[i+1])      // data:  1-40
                        data[i+1] = ref[i];    // ref:   0-39

                    // max per board
                    if (ref[i]>data[i+42])     // data: 42-81
                        data[i+42] = ref[i];   // ref:   0-39
                }


           }

            vector<float> deg(82);              //  0: global min,  1-40: min
            for (int i=0; i<82; i++)            // 41: global max, 42-81: max
                deg[i] = data[i]/16.;
            fDimTemperature.Update(deg);
        }

        /*
         uint16_t fTriggerType;
         uint32_t fTriggerId;
         uint32_t fEventCounter;
         uint16_t fAdcClockPhaseShift;
         uint16_t fNumTriggersToGenerate;
         uint16_t fTriggerGeneratorPrescaler;
         uint32_t fTimeStamp;
         int16_t  fTempDrs[kNumTemp];   // In units of 1/16 deg(?)
         uint16_t fDac[kNumDac];
         */
    }
};

EventBuilderWrapper *EventBuilderWrapper::This = 0;

// ----------- Event builder callbacks implementation ---------------
extern "C"
{
    FileHandle_t runOpen(uint32_t irun, RUN_HEAD *runhd, size_t len)
    {
        return EventBuilderWrapper::This->runOpen(irun, runhd, len);
    }

    int runWrite(FileHandle_t fileId, EVENT *event, size_t len)
    {
        return EventBuilderWrapper::This->runWrite(fileId, event, len);
    }

    int runClose(FileHandle_t fileId, RUN_TAIL *runth, size_t len)
    {
        return EventBuilderWrapper::This->runClose(fileId, runth, len);
    }

    void factOut(int severity, int err, const char *message)
    {
        EventBuilderWrapper::This->factOut(severity, err, message);
    }

    void factStat(GUI_STAT stat)
    {
        EventBuilderWrapper::This->factStat(stat);
    }

    void factStatNew(EVT_STAT stat)
    {
        EventBuilderWrapper::This->factStat(stat);
    }

    void debugHead(int socket, int/*board*/, void *buf)
    {
        const uint16_t *ptr = reinterpret_cast<uint16_t*>(buf);

        EventBuilderWrapper::This->debugHead(socket, FAD::EventHeader(ptr));
    }

    void debugStream(int isock, void *buf, int len)
    {
        return EventBuilderWrapper::This->debugStream(isock, buf, len);
    }

    void debugRead(int isock, int ibyte, int32_t event, int32_t ftmevt, int32_t runno, int state, uint32_t tsec, uint32_t tusec)
    {
        EventBuilderWrapper::This->debugRead(isock, ibyte, event, ftmevt, runno, state, tsec, tusec);
    }

    int eventCheck(PEVNT_HEADER *fadhd, EVENT *event)
    {
        return EventBuilderWrapper::This->eventCheck(fadhd, event);
    }

    void gotNewRun( int runnr, PEVNT_HEADER *headers )
    {
        return EventBuilderWrapper::This->gotNewRun(runnr, headers);
    }
}

#endif
