#include "Dim.h"
#include "Event.h"
#include "Shell.h"
#include "StateMachineDim.h"
#include "Connection.h"
#include "Configuration.h"
#include "Console.h"
#include "Converter.h"
#include "DimServiceInfoList.h"

#include "tools.h"

#include "LocalControl.h"

#include "HeadersFTM.h"
#include "HeadersFAD.h"


namespace ba    = boost::asio;
namespace bs    = boost::system;
namespace dummy = ba::placeholders;

using namespace std;

// ------------------------------------------------------------------------

#include "DimDescriptionService.h"

// ------------------------------------------------------------------------

class StateMachineMCP : public StateMachineDim, public DimInfoHandler
{
    /*
    int Wrap(boost::function<void()> f)
    {
        f();
        return T::GetCurrentState();
    }

    boost::function<int(const EventImp &)> Wrapper(boost::function<void()> func)
    {
        return bind(&StateMachineMCP::Wrap, this, func);
    }*/

private:
    enum states_t
    {
        kStateDimNetworkNA = 1,
        kStateDisconnected,
        kStateConnecting,
        kStateConnected,
        kStateIdle,
        kStateReadyForDataTaking,
        kStateConfiguring1,
        kStateConfiguring2,
        kStateConfiguring3,
        kStateConfigured,
        kStateTriggerOn,
        kStateTakingData,
//        kStateRunInProgress,
    };

    DimServiceInfoList fNetwork;

    pair<Time, int> fStatusDim;
    pair<Time, int> fStatusFTM;
    pair<Time, int> fStatusFAD;
    pair<Time, int> fStatusLog;
    pair<Time, int> fStatusRC;

    DimStampedInfo fDim;
    DimStampedInfo fFTM;
    DimStampedInfo fFAD;
    DimStampedInfo fLog;
    DimStampedInfo fRC;

    DimDescribedService fService;

    pair<Time, int> GetNewState(DimStampedInfo &info) const
    {
        const bool disconnected = info.getSize()==0;

        // Make sure getTimestamp is called _before_ getTimestampMillisecs
        const int tsec = info.getTimestamp();
        const int tms  = info.getTimestampMillisecs();

        return make_pair(Time(tsec, tms*1000),
                         disconnected ? -2 : info.getQuality());
    }

    void infoHandler()
    {
        DimInfo *curr = getInfo(); // get current DimInfo address
        if (!curr)
            return;

        if (curr==&fFTM)
        {
            fStatusFTM = GetNewState(fFTM);
            return;
        }

        if (curr==&fFAD)
        {
            fStatusFAD = GetNewState(fFAD);
            return;
        }

        if (curr==&fLog)
        {
            fStatusLog = GetNewState(fLog);
            return;
        }

        if (curr==&fRC)
        {
            fStatusRC = GetNewState(fRC);
            return;
        }

        if (curr==&fDim)
        {
            fStatusDim = GetNewState(fDim);
            fStatusDim.second = curr->getSize()==4 ? curr->getInt() : 0;
            return;
        }
    }

    bool CheckEventSize(size_t has, const char *name, size_t size)
    {
        if (has==size)
            return true;

        ostringstream msg;
        msg << name << " - Received event has " << has << " bytes, but expected " << size << ".";
        Fatal(msg);
        return false;
    }

    int SetVerbosity(const EventImp &)
    {
        /*
        if (!CheckEventSize(evt.GetSize(), "SetVerbosity", 1))
            return T::kSM_FatalError;

        fFSC.SetVerbose(evt.GetBool());

        */

        return GetCurrentState();
    }


    void PrintState(const pair<Time,int> &state, const char *server)
    {
        const State rc = fNetwork.GetState(server, state.second);

        Out() << state.first.GetAsStr("%H:%M:%S.%f").substr(0, 12) << " - ";
        Out() << kBold << server << ": ";
        Out() << rc.name << "[" << rc.index << "]";
        Out() << kReset << " - " << kBlue << rc.comment << endl;
    }

    int Print()
    {
        Out() << fStatusDim.first.GetAsStr("%H:%M:%S.%f").substr(0, 12) << " - ";
        Out() << kBold << "DIM_DNS: ";
        if (fStatusDim.second==0)
            Out() << "Offline" << endl;
        else
            Out() << "V" << fStatusDim.second/100 << 'r' << fStatusDim.second%100 << endl;

        PrintState(fStatusFTM, "FTM_CONTROL");
        PrintState(fStatusFAD, "FAD_CONTROL");
        PrintState(fStatusLog, "DATA_LOGGER");
        PrintState(fStatusRC,  "RATE_CONTROL");

        return GetCurrentState();
    }

    int GetReady()
    {
        return GetCurrentState();
    }

    int StopRun(const EventImp &)
    {
	if (fStatusFTM.second==FTM::kTriggerOn)
	{
            Message("Stopping FTM");
	    Dim::SendCommand("FTM_CONTROL/STOP_TRIGGER");
	}

        // FIXME: Do step 2 only when FTM is stopped
        if (fStatusFAD.second==FAD::kConnected)
        {
            //Dim::SendCommand("FAD_CONTROL/ENABLE_TRIGGER_LINE",      bool(false));
	    Message("Stopping FAD");
            Dim::SendCommand("FAD_CONTROL/ENABLE_CONTINOUS_TRIGGER", bool(false));
        }

        return GetCurrentState();
    }

    int Reset(const EventImp &)
    {
        fRunType = "";
	Message("Reseting configuration states of FAD and FTM");
	Dim::SendCommand("FTM_CONTROL/RESET_CONFIGURE");
	Dim::SendCommand("FAD_CONTROL/RESET_CONFIGURE");
        return kStateIdle;
        /*
        // FIMXE: Handle error states!
        if (fStatusLog.second>=20)//kSM_NightlyOpen
            Dim::SendCommand("DATA_LOGGER/STOP");

        if (fStatusLog.second==0)
            Dim::SendCommand("DATA_LOGGER/WAIT_FOR_RUN_NUMBER");

        if (fStatusFAD.second==FAD::kConnected)
        {
            Dim::SendCommand("FAD_CONTROL/ENABLE_TRIGGER_LINE", bool(false));
            Dim::SendCommand("FAD_CONTROL/ENABLE_CONTINOUS_TRIGGER", bool(false));
        }

        if (fStatusFTM.second==FTM::kTakingData)
            Dim::SendCommand("FTM_CONTROL/STOP");

        return GetCurrentState(); */
    }

    int64_t fMaxTime;
    int64_t fNumEvents;
    string  fRunType;

    int StartRun(const EventImp &evt)
    {
        if (fStatusFTM.second==-2)
        {
            Error("No connection to ftmcontrol (see PRINT).");
            return GetCurrentState();
        }
        if (fStatusFAD.second==-2)
        {
            Warn("No connection to fadcontrol (see PRINT).");
            return GetCurrentState();
        }
        if (fStatusLog.second==-2)
        {
            Warn("No connection to datalogger (see PRINT).");
            return GetCurrentState();
        }
        if (fStatusRC.second==-2)
        {
            Warn("No connection to ratecontrol (see PRINT).");
            return GetCurrentState();
        }

        fMaxTime   = evt.Get<int64_t>();
        fNumEvents = evt.Get<int64_t>(8);
        fRunType   = evt.Ptr<char>(16);

        ostringstream str;
        str << "Starting configuration '" << fRunType << "' for new run";
        if (fNumEvents>0 || fMaxTime>0)
            str << " [";
        if (fNumEvents>0)
            str << fNumEvents << " events";
        if (fNumEvents>0 && fMaxTime>0)
            str << " / ";
        if (fMaxTime>0)
            str << fMaxTime << "s";
        if (fNumEvents>0 || fMaxTime>0)
            str << "]";
        Message(str);

        Update(kStateConfiguring1);

        return kStateConfiguring1;
    }

    struct Value
    {
        uint64_t time;
        uint64_t nevts;
        char type[];
    };

    Value *GetBuffer()
    {
        const size_t len = sizeof(Value)+fRunType.length()+1;

        char *buf = new char[len];

        Value *val = reinterpret_cast<Value*>(buf);

        val->time  = fMaxTime;
        val->nevts = fNumEvents;

        strcpy(val->type, fRunType.c_str());

        return val;
    }

    void Update(int newstate)
    {
        Value *buf = GetBuffer();
        fService.setQuality(newstate);
        fService.setData(buf, sizeof(Value)+fRunType.length()+1);
        fService.Update();
        delete buf;
    }

    void ConfigureFAD()
    {
        Value *buf = GetBuffer();

	Message("Configuring FAD");
        Dim::SendCommand("FAD_CONTROL/CONFIGURE", buf, sizeof(Value)+fRunType.length()+1);

        delete buf;
    }

    int Execute()
    {
        // Dispatch (execute) at most one handler from the queue. In contrary
        // to run_one(), it doesn't wait until a handler is available
        // which can be dispatched, so poll_one() might return with 0
        // handlers dispatched. The handlers are always dispatched/executed
        // synchronously, i.e. within the call to poll_one()
        //poll_one();

        if (fStatusDim.second==0)
            return kStateDimNetworkNA;

        if (fStatusFTM.second >= FTM::kConnected &&
            fStatusFAD.second >= FAD::kConnected &&
            fStatusLog.second >= kSM_Ready)
        {
            if (GetCurrentState()==kStateConfiguring1)
            {
		if (fStatusLog.second<30/*kSM_WaitForRun*/)
		{
		    Message("Starting datalogger");
                    Dim::SendCommand("DATA_LOGGER/START_RUN_LOGGING");
		}
		Message("Configuring Trigger (FTM)");
                Dim::SendCommand("FTM_CONTROL/CONFIGURE", fRunType);
                Update(kStateConfiguring2);
                return kStateConfiguring2;
            }

            if (GetCurrentState()==kStateConfiguring2)
            {
                // FIMXE: Reset in case of error
                if ((/*fStatusFTM.second != FTM::kConfiguring2 &&*/
                     fStatusFTM.second != FTM::kConfigured) ||
                    fStatusLog.second<30 || fStatusLog.second>0xff)
                    return GetCurrentState();

                // FIMXE: This is to make sure that the rate control
                // has received the correct trigger setup already...
                //usleep(1000000);

                Message("Starting Rate Control");
                Dim::SendCommand("RATE_CONTROL/CALIBRATE");

                ConfigureFAD();
                Update(kStateConfiguring3);
                return kStateConfiguring3;
            }

            if (GetCurrentState()==kStateConfiguring3)
            {
                if (fStatusFTM.second != FTM::kConfigured ||
                    fStatusFAD.second != FAD::kConfigured ||
                    fStatusRC.second  < 6)
                    return GetCurrentState();

                Message("Starting Trigger (FTM)");
                Dim::SendCommand("FTM_CONTROL/START_TRIGGER");
                Update(kStateConfigured);
                return kStateConfigured;
            }

            if (GetCurrentState()==kStateConfigured)
            {
                if (fStatusFTM.second != FTM::kTriggerOn)
                    return GetCurrentState();

                Update(kStateTriggerOn);

                return kStateTriggerOn;
            }

            if (GetCurrentState()==kStateTriggerOn)
            {
                if (fStatusFAD.second != FAD::kWritingData)
                    return GetCurrentState();

                Update(kStateTakingData);

                return kStateTakingData;
            }

            if (GetCurrentState()==kStateTakingData)
            {
                if (fStatusFTM.second==FTM::kTriggerOn &&
                    fStatusFAD.second==FAD::kWritingData)
                    return kStateTakingData;

                Update(kStateIdle);
            }

            return kStateIdle;
        }

        /*
        if (fStatusFTM.second >= FTM::kConnected &&
            fStatusFAD.second >= FAD::kConnected &&
            fStatusLog.second >= kSM_Ready)
            return kStateIdle;
         */
        if (fStatusFTM.second >-2 &&
            fStatusFAD.second >-2 &&
            fStatusLog.second >-2 &&
            fStatusRC.second >-2)
            return kStateConnected;

        if (fStatusFTM.second >-2 ||
            fStatusFAD.second >-2 ||
            fStatusLog.second >-2 ||
            fStatusRC.second >-2)
            return kStateConnecting;

        return kStateDisconnected;
    }

public:
    StateMachineMCP(ostream &out=cout) : StateMachineDim(out, "MCP"),
        fStatusDim(make_pair(Time(), -2)),
        fStatusFTM(make_pair(Time(), -2)),
        fStatusFAD(make_pair(Time(), -2)),
        fStatusLog(make_pair(Time(), -2)),
        fDim("DIS_DNS/VERSION_NUMBER", (void*)NULL, 0, this),
        fFTM("FTM_CONTROL/STATE",      (void*)NULL, 0, this),
        fFAD("FAD_CONTROL/STATE",      (void*)NULL, 0, this),
        fLog("DATA_LOGGER/STATE",      (void*)NULL, 0, this),
        fRC("RATE_CONTROL/STATE",      (void*)NULL, 0, this),
        fService("MCP/CONFIGURATION", "X:1;X:1;C", "Run configuration information"
                 "|MaxTime[s]:Maximum time before the run gets stopped"
                 "|MaxEvents[num]:Maximum number of events before the run gets stopped"
                 "|Name[text]:Name of the chosen configuration")
    {
        // ba::io_service::work is a kind of keep_alive for the loop.
        // It prevents the io_service to go to stopped state, which
        // would prevent any consecutive calls to run()
        // or poll() to do nothing. reset() could also revoke to the
        // previous state but this might introduce some overhead of
        // deletion and creation of threads and more.

        // State names
        AddStateName(kStateDimNetworkNA, "DimNetworkNotAvailable",
                     "DIM dns server not available.");

        AddStateName(kStateDisconnected, "Disconnected",
                     "Neither ftmctrl, fadctrl, datalogger nor rate control online.");

        AddStateName(kStateConnecting, "Connecting",
                     "Either ftmctrl, fadctrl, datalogger or rate control not online.");

        AddStateName(kStateConnected, "Connected",
                     "All needed subsystems online.");

        AddStateName(kStateIdle, "Idle",
                     ".");

        AddStateName(kStateReadyForDataTaking, "ReadyForDataTaking",
                     ".");

        AddStateName(kStateConfiguring1, "Configuring1",
                     ".");

        AddStateName(kStateConfiguring2, "Configuring2",
                     "Waiting for FTM and Datalogger to get ready");

        AddStateName(kStateConfiguring3, "Configuring3",
                     "Waiting for FADs and reate control to get ready");

        AddStateName(kStateConfigured, "Configured",
                     "Everything is configured, trigger will be switched on now");

        AddStateName(kStateTriggerOn, "TriggerOn",
                     "The trigger is switched on, waiting for FAD to receive data");

        AddStateName(kStateTakingData, "TakingData",
                     "The trigger is switched on, FADs are sending data");


        AddEvent("START", "X:2;C")//, kStateIdle)
            (bind(&StateMachineMCP::StartRun, this, placeholders::_1))
            ("");

        AddEvent("STOP")
            (bind(&StateMachineMCP::StopRun, this, placeholders::_1))
            ("");

        AddEvent("RESET", kStateConfiguring1, kStateConfiguring2, kStateConfiguring3, kStateConfigured)
            (bind(&StateMachineMCP::Reset, this, placeholders::_1))
            ("");

        // Verbosity commands
        AddEvent("SET_VERBOSE", "B:1")
            (bind(&StateMachineMCP::SetVerbosity, this, placeholders::_1))
            ("set verbosity state"
             "|verbosity[bool]:disable or enable verbosity for received data (yes/no), except dynamic data");

        AddEvent("PRINT")
            (bind(&StateMachineMCP::Print, this))
            ("");
    }

    int EvalOptions(Configuration &)
    {
        //SetEndpoint(conf.Get<string>("addr"));

        //fFSC.SetVerbose(!conf.Get<bool>("quiet"));

        return -1;
    }
};

// ------------------------------------------------------------------------

#include "Main.h"

template<class T>
int RunShell(Configuration &conf)
{
    return Main::execute<T, StateMachineMCP>(conf);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "The ftmctrl controls the FSC (FACT Slow Control) board.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: fscctrl [-c type] [OPTIONS]\n"
        "  or:  fscctrl [OPTIONS]\n";
    cout << endl;
}

void PrintHelp()
{
    Main::PrintHelp<StateMachineMCP>();

    /* Additional help text which is printed after the configuration
     options goes here */

    /*
     cout << "bla bla bla" << endl << endl;
     cout << endl;
     cout << "Environment:" << endl;
     cout << "environment" << endl;
     cout << endl;
     cout << "Examples:" << endl;
     cout << "test exam" << endl;
     cout << endl;
     cout << "Files:" << endl;
     cout << "files" << endl;
     cout << endl;
     */
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    Main::SetupConfiguration(conf);

    if (!conf.DoParse(argc, argv, PrintHelp))
        return -1;

    //try
    {
        // No console access at all
        if (!conf.Has("console"))
        {
//            if (conf.Get<bool>("no-dim"))
//                return RunShell<LocalStream, StateMachine, ConnectionFSC>(conf);
//            else
                return RunShell<LocalStream>(conf);
        }
        // Cosole access w/ and w/o Dim
/*        if (conf.Get<bool>("no-dim"))
        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell, StateMachine, ConnectionFSC>(conf);
            else
                return RunShell<LocalConsole, StateMachine, ConnectionFSC>(conf);
        }
        else
*/        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell>(conf);
            else
                return RunShell<LocalConsole>(conf);
        }
    }
    /*catch (std::exception& e)
    {
        cerr << "Exception: " << e.what() << endl;
        return -1;
    }*/

    return 0;
}
