#include "InterpreterV8.h"

#ifdef HAVE_V8

#include <fstream>
#include <sstream>
#include <iomanip>

#include <boost/tokenizer.hpp>

#ifdef HAVE_NOVA
#include <libnova/solar.h>
#include <libnova/lunar.h>
#include <libnova/transform.h>
#endif

#ifdef HAVE_SQL
#include "Database.h"
#endif

#include <v8.h>

#include "dim.h"
#include "tools.h"
#include "Readline.h"
#include "externals/izstream.h"

#include "WindowLog.h"

using namespace std;
using namespace v8;

v8::Handle<v8::FunctionTemplate> InterpreterV8::fTemplateLocal;
v8::Handle<v8::FunctionTemplate> InterpreterV8::fTemplateSky;
v8::Handle<v8::FunctionTemplate> InterpreterV8::fTemplateEvent;
//v8::Handle<v8::FunctionTemplate> InterpreterV8::fTemplateDatabase;


// ==========================================================================
//                           Some documentation
// ==========================================================================
//
// Threads:
// --------
// In most cases Js* and other calls to native C++ code could be wrapped
// with an Unlocker to allow possible other JavaScipt 'threads' to run
// during that time. However, all of these calls should take much less than
// the preemption time of 10ms, so it would just be a waste of tim.
//
// Termination:
// ------------
// Each thread running V8 code needs to be signalled individually for
// termination. Therefor a list of V8 thread ids is created.
//
// If termination has already be signalled, no thread should start running
// anymore (thy could, e.g., wait for their locking). So after locking
// it has to be checked if the thread was terminated already. Note
// that all calls to Terminate() must be locked to ensure that fThreadId
// is correct when it is checked.
//
// The current thread id must be added to fThreadIds _before_ any
// function is called after Locking and before execution is given
// back to JavaScript, e.g. in script->Run(). So until the thread
// is added to the list Terminate will not be executed. If Terminate
// is then executed, it is ensured that the current thread is
// already in the list. If terminate has been called before
// the Locking, the check for the validiy of fThreadId ensures that
// nothing is executed.
//
// Empty handles:
// --------------
// If exceution is terminated, V8 calls might return with empty handles,
// e.g. Date::New(). Therefore, the returned handles of these calls have to
// be checked in all placed to avoid that V8 will core dump.
//
// HandleScope:
// ------------
// A handle scope is a garbage collector and collects all handles created
// until it goes out of scope. Handles which are not needed anymore are
// then deleted. To return a handle from a HandleScope you need to use
// Close(). E.g., String::AsciiValue does not create a new handle and
// hence does not need a HandleScope. Any ::New will need a handle scope.
// Forgetting the HandleScope could in principle fill your memory,
// but everything is properly deleted by the global HandleScope at
// script termination.
//

// ==========================================================================
//                            Simple interface
// ==========================================================================

Handle<Value> InterpreterV8::FuncExit(const Arguments &)
{
    V8::TerminateExecution(fThreadId);

    // we have to throw an excption to make sure that the
    // calling thread does not go on executing until it
    // has realized that it should terminate
    return ThrowException(Null());
}

Handle<Value> InterpreterV8::FuncSleep(const Arguments& args)
{
    if (args.Length()==0)
    {
        // Theoretically, the CPU usage can be reduced by maybe a factor
        // of four using a larger value, but this also means that the
        // JavaScript is locked for a longer time.
        const Unlocker unlock;
        usleep(1000);
        return Undefined();
    }

    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 1 must be an uint32."));

    // Using a Javascript function has the advantage that it is fully
    // interruptable without the need of C++ code
    const string code =
        "(function(){"
        "var t=new Date();"
        "while ((new Date()-t)<"+to_string(args[0]->Int32Value())+") v8.sleep();"
        "})();";

    return ExecuteInternal(code);
}

void InterpreterV8::Thread(int &id, Persistent<Function> func, uint32_t ms)
{
    const Locker lock;

    if (fThreadId<0)
    {
        id = -1;
        return;
    }

    id = V8::GetCurrentThreadId();
    fThreadIds.insert(id);

    const HandleScope handle_scope;

    func->CreationContext()->Enter();

    TryCatch exception;

    const bool rc = ms==0 || !ExecuteInternal("v8.sleep("+to_string(ms)+");").IsEmpty();
    if (rc)
        func->Call(func, 0, NULL);

    func.Dispose();
    fThreadIds.erase(id);

    if (!HandleException(exception, "thread"))
        V8::TerminateExecution(fThreadId);
}

Handle<Value> InterpreterV8::FuncThread(const Arguments& args)
{
    if (!args.IsConstructCall())
        return ThrowException(String::New("Thread must be called as constructor."));

    if (args.Length()!=2)
        return ThrowException(String::New("Number of arguments must be two."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 0 not an uint32."));

    if (!args[1]->IsFunction())
        return ThrowException(String::New("Argument 1 not a function."));

    //if (!args.IsConstructCall())
    //    return Constructor(args);

    const HandleScope handle_scope;

    Handle<Function> handle = Handle<Function>::Cast(args[1]);

    Persistent<Function> func = Persistent<Function>::New(handle);

    const uint32_t ms = args[0]->Uint32Value();

    int id=-2;
    fThreads.push_back(thread(bind(&InterpreterV8::Thread, this, ref(id), func, ms)));
    {
        // Allow the thread to lock, so we can get the thread id.
        const Unlocker unlock;
        while (id==-2)
            usleep(1);
    }

    Handle<Object> self = args.This();

    self->Set(String::New("id"), Integer::NewFromUnsigned(id), ReadOnly);
    self->Set(String::New("kill"), FunctionTemplate::New(WrapKill)->GetFunction(), ReadOnly);

    return Undefined();
}

Handle<Value> InterpreterV8::FuncKill(const Arguments& args)
{
    const uint32_t id = args.This()->Get(String::New("id"))->Uint32Value();

    V8::TerminateExecution(id);

    return Boolean::New(fThreadIds.erase(id));
}

Handle<Value> InterpreterV8::FuncSend(const Arguments& args)
{
    if (args.Length()==0)
        return ThrowException(String::New("Number of arguments must be at least 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    const String::AsciiValue str(args[0]);

    string command = *str;

    if (command.length()==0)
        return ThrowException(String::New("Server name empty."));

    if (args.Length()==0)
    {
        if (command.find_first_of('/')==string::npos)
            command += "/";
    }

    // Escape all string arguments. All others can be kept as they are.
    for (int i=1; i<args.Length(); i++)
    {
        string arg = *String::AsciiValue(args[i]);

        // Escape string
        if (args[i]->IsString())
        {
            boost::replace_all(arg, "\\", "\\\\");
            boost::replace_all(arg, "'", "\\'");
            boost::replace_all(arg, "\"", "\\\"");
        }

        command += " "+arg;
    }

    try
    {
        return Boolean::New(JsSend(command));
    }
    catch (const runtime_error &e)
    {
        return ThrowException(String::New(e.what()));
    }
}

// ==========================================================================
//                               State control
// ==========================================================================

Handle<Value> InterpreterV8::FuncWait(const Arguments& args)
{
    if (args.Length()!=2 && args.Length()!=3)
        return ThrowException(String::New("Number of arguments must be 2 or 3."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 not a string."));

    if (!args[1]->IsInt32() && !args[1]->IsString())
        return ThrowException(String::New("Argument 2 not an int32 and not a string."));

    if (args.Length()==3 && !args[2]->IsInt32())
        return ThrowException(String::New("Argument 3 not an int32."));

    // Using a Javascript function has the advantage that it is fully
    // interruptable without the need of C++ code

    const string index   = args[1]->IsInt32() ? "s.index" : "s.name";
    const bool   timeout = args.Length()==3;
    const string arg0    = *String::AsciiValue(args[0]);
    const string state   = args[1]->IsString() ? *String::AsciiValue(args[1]) : "";
    const string arg1    = args[1]->IsString() ? ("\""+state+"\"") : to_string(args[1]->Int32Value());

    if (arg0.find_first_of("\"'")!=string::npos)
        return ThrowException(String::New("Server name must not contain quotation marks."));

    if (args[1]->IsString())
        if (state.find_first_of("\"'")!=string::npos)
            return ThrowException(String::New("State name must not contain quotation marks."));

    string code =  "(function(name,state,ms)"
                   "{";
    if (timeout)
        code +=       "var t = new Date();";
    code +=           "while (1)"
                      "{"
                         "var s = dim.state(name);"
                         "if(!s)throw new Error('Waitig for state "+arg1+" of server "+arg0+" failed.');"
                         "if(state=="+index+")return true;";
    if (timeout)
        code +=          "if((new Date()-t)>Math.abs(ms))break;";

    code +=              "v8.sleep();"
                      "}";
    if (timeout)
        code +=    "if(ms>0)throw new Error('Waiting for state "+arg1+" of server "+arg0+" timed out.');";
    code +=        "return false;"
                   "})('"+arg0+"',"+arg1;
    if (timeout)
        code +=    "," + to_string(args[2]->Int32Value());
    code +=        ");";

    return ExecuteInternal(code);
}

Handle<Value> InterpreterV8::FuncState(const Arguments& args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    // Return state.name/state.index

    const String::AsciiValue str(args[0]);

    const State rc = JsState(*str);
    if (rc.index<=-256)
        return Undefined();

    HandleScope handle_scope;

    Handle<Object> obj = Object::New();

    obj->Set(String::New("server"), String::New(*str),            ReadOnly);
    obj->Set(String::New("index"),  Integer::New(rc.index),       ReadOnly);
    obj->Set(String::New("name"),   String::New(rc.name.c_str()), ReadOnly);

    const Local<Value> date = Date::New(rc.time.JavaDate());
    if (rc.index>-256 && !date.IsEmpty())
        obj->Set(String::New("time"),  date);

    return handle_scope.Close(obj);
}

Handle<Value> InterpreterV8::FuncNewState(const Arguments& args)
{
    if (args.Length()<1 || args.Length()>3)
        return ThrowException(String::New("Number of arguments must be 1, 2 or 3."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 1 must be an uint32."));
    if (args.Length()>1 && !args[1]->IsString())
        return ThrowException(String::New("Argument 2 must be a string."));
    if (args.Length()>2 && !args[2]->IsString())
        return ThrowException(String::New("Argument 3 must be a string."));

    const uint32_t index   = args[0]->Int32Value();
    const string   name    = *String::AsciiValue(args[1]);
    const string   comment = *String::AsciiValue(args[2]);

    if (index<10 || index>255)
        return ThrowException(String::New("State must be in the range [10, 255]."));

    if (name.empty())
        return ThrowException(String::New("State name must not be empty."));

    if (name.find_first_of(':')!=string::npos || name.find_first_of('=')!=string::npos)
        return ThrowException(String::New("State name must not contain : or =."));

    struct Find : State
    {
        Find(int idx, const string &n) : State(idx, n) { }
        bool operator()(const pair<int, string> &p) { return index==p.first || name==p.second; }
    };

    if (find_if(fStates.begin(), fStates.end(), Find(index, name))!=fStates.end())
    {
        const string what =
            "State index ["+to_string(index)+"] or name ["+name+"] already defined.";

        return ThrowException(String::New(what.c_str()));
    }

    return Boolean::New(JsNewState(index, name, comment));
}

Handle<Value> InterpreterV8::FuncSetState(const Arguments& args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsUint32() && !args[0]->IsString())
        return ThrowException(String::New("Argument must be an unint32 or a  string."));

    int index = -2;
    if (args[0]->IsUint32())
    {
        index = args[0]->Int32Value();
    }
    else
    {
        const string name = *String::AsciiValue(args[0]);
        index = JsGetState(name);
        if (index==-2)
            return ThrowException(String::New(("State '"+name+"' not found.").c_str()));
    }

    if (index<10 || index>255)
        return ThrowException(String::New("State must be in the range [10, 255]."));

    return Boolean::New(JsSetState(index));
}

Handle<Value> InterpreterV8::FuncGetState(const Arguments& args)
{
    if (args.Length()>0)
        return ThrowException(String::New("getState must not take arguments."));

    const State state = JsGetCurrentState();

    HandleScope handle_scope;

    Handle<Object> rc = Object::New();
    if (rc.IsEmpty())
        return Undefined();

    rc->Set(String::New("index"), Integer::New(state.index), ReadOnly);
    rc->Set(String::New("name"), String::New(state.name.c_str()), ReadOnly);
    rc->Set(String::New("description"), String::New(state.comment.c_str()), ReadOnly);

    return handle_scope.Close(rc);
}

Handle<Value> InterpreterV8::FuncGetStates(const Arguments& args)
{
    if (args.Length()>1)
        return ThrowException(String::New("getStates must not take more than one arguments."));

    if (args.Length()==1 && !args[0]->IsString())
        return ThrowException(String::New("Argument must be a string."));

    const string server = args.Length()==1 ? *String::AsciiValue(args[0]) : "DIM_CONTROL";

    const vector<State> states = JsGetStates(server);

    HandleScope handle_scope;

    Handle<Object> list = Object::New();
    if (list.IsEmpty())
        return Undefined();

    for (auto it=states.begin(); it!=states.end(); it++)
    {
        Handle<Value> entry = StringObject::New(String::New(it->name.c_str()));
        if (entry.IsEmpty())
            return Undefined();

        StringObject::Cast(*entry)->Set(String::New("description"), String::New(it->comment.c_str()), ReadOnly);
        list->Set(Integer::New(it->index), entry, ReadOnly);
    }

    return handle_scope.Close(list);
}

Handle<Value> InterpreterV8::FuncGetDescription(const Arguments& args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("getDescription must take exactly one argument."));

    if (args.Length()==1 && !args[0]->IsString())
        return ThrowException(String::New("Argument must be a string."));

    const string service = *String::AsciiValue(args[0]);

    const vector<Description> descriptions = JsGetDescription(service);
    const set<Service> services = JsGetServices();

    auto is=services.begin();
    for (; is!=services.end(); is++)
        if (is->name==service)
            break;

    HandleScope handle_scope;

    Handle<Array> arr = Array::New();
    if (arr.IsEmpty())
        return Undefined();

    auto it=descriptions.begin();
    arr->Set(String::New("name"), String::New(it->name.c_str()), ReadOnly);
    if (!it->comment.empty())
        arr->Set(String::New("description"), String::New(it->comment.c_str()), ReadOnly);
    if (is!=services.end())
    {
        arr->Set(String::New("server"), String::New(is->server.c_str()), ReadOnly);
        arr->Set(String::New("service"), String::New(is->service.c_str()), ReadOnly);
        arr->Set(String::New("isCommand"), Boolean::New(is->iscmd), ReadOnly);
        if (!is->format.empty())
            arr->Set(String::New("format"), String::New(is->format.c_str()), ReadOnly);
    }

    uint32_t i=0;
    for (it++; it!=descriptions.end(); it++)
    {
        Handle<Object> obj = Object::New();
        if (obj.IsEmpty())
            return Undefined();

        obj->Set(String::New("name"), String::New(it->name.c_str()), ReadOnly);
        if (!it->comment.empty())
            obj->Set(String::New("description"), String::New(it->comment.c_str()), ReadOnly);
        if (!it->unit.empty())
            obj->Set(String::New("unit"), String::New(it->unit.c_str()), ReadOnly);

        arr->Set(i++, obj);
    }

    return handle_scope.Close(arr);
}

Handle<Value> InterpreterV8::FuncGetServices(const Arguments& args)
{
    if (args.Length()>2)
        return ThrowException(String::New("getServices must not take more than two argument."));

    if (args.Length()>=1 && !args[0]->IsString())
        return ThrowException(String::New("First argument must be a string."));

    if (args.Length()==2 && !args[1]->IsBoolean())
        return ThrowException(String::New("Second argument must be a boolean."));

    string arg0 = args.Length() ? *String::AsciiValue(args[0]) : "";
    if (arg0=="*")
        arg0=="";

    const set<Service> services = JsGetServices();

    HandleScope handle_scope;

    Handle<Array> arr = Array::New();
    if (arr.IsEmpty())
        return Undefined();

    uint32_t i=0;
    for (auto is=services.begin(); is!=services.end(); is++)
    {
        if (!arg0.empty() && is->name.find(arg0)!=0)
            continue;

        if (args.Length()==2 && args[1]->BooleanValue()!=is->iscmd)
            continue;

        Handle<Object> obj = Object::New();
        if (obj.IsEmpty())
            return Undefined();

        obj->Set(String::New("name"), String::New(is->name.c_str()), ReadOnly);
        obj->Set(String::New("server"), String::New(is->server.c_str()), ReadOnly);
        obj->Set(String::New("service"), String::New(is->service.c_str()), ReadOnly);
        obj->Set(String::New("isCommand"), Boolean::New(is->iscmd), ReadOnly);
        if (!is->format.empty())
            obj->Set(String::New("format"), String::New(is->format.c_str()), ReadOnly);

        arr->Set(i++, obj);
    }

    return handle_scope.Close(arr);
}

// ==========================================================================
//                             Internal functions
// ==========================================================================


// The callback that is invoked by v8 whenever the JavaScript 'print'
// function is called.  Prints its arguments on stdout separated by
// spaces and ending with a newline.
Handle<Value> InterpreterV8::FuncLog(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const String::AsciiValue str(args[i]);
        if (*str)
            JsPrint(*str);
    }
    return Undefined();
}

Handle<Value> InterpreterV8::FuncAlarm(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const String::AsciiValue str(args[i]);
        if (*str)
            JsAlarm(*str);
    }

    if (args.Length()==0)
        JsAlarm();

    return Undefined();
}

Handle<Value> InterpreterV8::FuncOut(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const String::AsciiValue str(args[i]);
        if (*str)
            JsOut(*str);
    }
    return Undefined();
}

// The callback that is invoked by v8 whenever the JavaScript 'load'
// function is called.  Loads, compiles and executes its argument
// JavaScript file.
Handle<Value> InterpreterV8::FuncInclude(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const String::AsciiValue file(args[i]);
        if (*file == NULL)
            return ThrowException(String::New("File name missing"));

        if (!ExecuteFile(*file))
        {
            V8::TerminateExecution(fThreadId);
            return ThrowException(Null());
        }
    }

    return Boolean::New(true);
}

Handle<Value> InterpreterV8::FuncFile(const Arguments& args)
{
    if (args.Length()!=1 && args.Length()!=2)
        return ThrowException(String::New("Number of arguments must be one or two."));

    const String::AsciiValue file(args[0]);
    if (*file == NULL)
        return ThrowException(String::New("File name missing"));

    if (args.Length()==2 && !args[1]->IsString())
        return ThrowException(String::New("Second argument must be a string."));

    const string delim = args.Length()==2 ? *String::AsciiValue(args[1]) : "";

    if (args.Length()==2 && delim.size()!=1)
        return ThrowException(String::New("Second argument must be a string of length 1."));

    HandleScope handle_scope;

    izstream fin(*file);
    if (!fin)
        return ThrowException(String::New(errno!=0?strerror(errno):"Insufficient memory for decompression"));

    if (args.Length()==1)
    {
        string buffer;
        if (!getline(fin, buffer, '\0'))
            return ThrowException(String::New(strerror(errno)));

        Handle<Value> str = StringObject::New(String::New(buffer.c_str()));
        StringObject::Cast(*str)->Set(String::New("name"), String::New(*file));
        return handle_scope.Close(str);
    }

    Handle<Array> arr = Array::New();
    if (arr.IsEmpty())
        return Undefined();

    int i=0;
    string buffer;
    while (getline(fin, buffer, delim[0]))
        arr->Set(i++, String::New(buffer.c_str()));

    if ((fin.fail() && !fin.eof()) || fin.bad())
        return ThrowException(String::New(strerror(errno)));

    arr->Set(String::New("name"),  String::New(*file));
    arr->Set(String::New("delim"), String::New(delim.c_str(), 1));

    return handle_scope.Close(arr);
}

// ==========================================================================
//                                 Database
// ==========================================================================

Handle<Value> InterpreterV8::FuncDbClose(const Arguments &args)
{
    void *ptr = External::Unwrap(args.This()->GetInternalField(0));
    if (!ptr)
        return Boolean::New(false);

#ifdef HAVE_SQL
    Database *db = reinterpret_cast<Database*>(ptr);
    auto it = find(fDatabases.begin(), fDatabases.end(), db);
    fDatabases.erase(it);
    delete db;
#endif

    HandleScope handle_scope;

    args.This()->SetInternalField(0, External::New(0));

    return handle_scope.Close(Boolean::New(true));
}

Handle<Value> InterpreterV8::FuncDbQuery(const Arguments &args)
{
    if (args.Length()==0)
        return ThrowException(String::New("Arguments expected."));

    void *ptr = External::Unwrap(args.This()->GetInternalField(0));
    if (!ptr)
        return Undefined();

    string query;
    for (int i=0; i<args.Length(); i++)
        query += string(" ") + *String::AsciiValue(args[i]);
    query.erase(0, 1);

#ifdef HAVE_SQL
    try
    {
        HandleScope handle_scope;

        Database *db = reinterpret_cast<Database*>(ptr);

        const mysqlpp::StoreQueryResult res = db->query(query).store();

        Handle<Array> ret = Array::New();
        if (ret.IsEmpty())
            return Undefined();

        ret->Set(String::New("table"), String::New(res.table()),   ReadOnly);
        ret->Set(String::New("query"), String::New(query.c_str()), ReadOnly);

        Handle<Array> cols = Array::New();
        if (cols.IsEmpty())
            return Undefined();

        int irow=0;
        for (vector<mysqlpp::Row>::const_iterator it=res.begin(); it<res.end(); it++)
        {
            Handle<Object> row = Object::New();
            if (row.IsEmpty())
                return Undefined();

            const mysqlpp::FieldNames *list = it->field_list().list;

            for (size_t i=0; i<it->size(); i++)
            {
                const Handle<Value> name = String::New((*list)[i].c_str());
                if (irow==0)
                    cols->Set(i, name);

                if ((*it)[i].is_null())
                {
                    row->Set(name, Undefined(), ReadOnly);
                    continue;
                }

                const string sql_type = (*it)[i].type().sql_name();

                const bool uns = sql_type.find("UNSIGNED")==string::npos;

                if (sql_type.find("BIGINT")!=string::npos)
                {
                    if (uns)
                    {
                        const uint64_t val = (uint64_t)(*it)[i];
                        if (val>UINT32_MAX)
                            row->Set(name, Number::New(val), ReadOnly);
                        else
                            row->Set(name, Integer::NewFromUnsigned(val), ReadOnly);
                    }
                    else
                    {
                        const int64_t val = (int64_t)(*it)[i];
                        if (val<INT32_MIN || val>INT32_MAX)
                            row->Set(name, Number::New(val), ReadOnly);
                        else
                            row->Set(name, Integer::NewFromUnsigned(val), ReadOnly);
                    }
                    continue;
                }

                // 32 bit
                if (sql_type.find("INT")!=string::npos)
                {
                    if (uns)
                        row->Set(name, Integer::NewFromUnsigned((uint32_t)(*it)[i]), ReadOnly);
                    else
                        row->Set(name, Integer::New((int32_t)(*it)[i]), ReadOnly);
                    continue;
                }

                if (sql_type.find("BOOL")!=string::npos )
                {
                    row->Set(name, Boolean::New((bool)(*it)[i]), ReadOnly);
                    continue;
                }

                if (sql_type.find("FLOAT")!=string::npos)
                {
                    ostringstream val;
                    val << setprecision(7) << (float)(*it)[i];
                    row->Set(name, Number::New(stod(val.str())), ReadOnly);
                    continue;

                }
                if (sql_type.find("DOUBLE")!=string::npos)
                {
                    row->Set(name, Number::New((double)(*it)[i]), ReadOnly);
                    continue;
                }

                if (sql_type.find("CHAR")!=string::npos ||
                    sql_type.find("TEXT")!=string::npos)
                {
                    row->Set(name, String::New((const char*)(*it)[i]), ReadOnly);
                    continue;
                }

                time_t date = 0;
                if (sql_type.find("TIMESTAMP")!=string::npos)
                    date = mysqlpp::Time((*it)[i]);

                if (sql_type.find("DATETIME")!=string::npos)
                    date = mysqlpp::DateTime((*it)[i]);

                if (sql_type.find(" DATE ")!=string::npos)
                    date = mysqlpp::Date((*it)[i]);

                if (date>0)
                {
                    // It is important to catch the exception thrown
                    // by Date::New in case of thread termination!
                    const Local<Value> val = Date::New(date*1000);
                    if (val.IsEmpty())
                        return Undefined();

                    row->Set(name, val, ReadOnly);
                }
            }

            ret->Set(irow++, row);
        }

        if (irow>0)
            ret->Set(String::New("cols"), cols, ReadOnly);

        return handle_scope.Close(ret);
    }
    catch (const exception &e)
    {
        return ThrowException(String::New(e.what()));
    }
#endif
}

Handle<Value> InterpreterV8::FuncDatabase(const Arguments &args)
{
    if (!args.IsConstructCall())
        return ThrowException(String::New("Database must be called as constructor."));

    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 not a string."));

#ifdef HAVE_SQL
    try
    {
        HandleScope handle_scope;

        //if (!args.IsConstructCall())
        //    return Constructor(fTemplateDatabase, args);

        Database *db = new Database(*String::AsciiValue(args[0]));
        fDatabases.push_back(db);

        Handle<Object> self = args.This();
        self->Set(String::New("user"),     String::New(db->user.c_str()), ReadOnly);
        self->Set(String::New("server"),   String::New(db->server.c_str()), ReadOnly);
        self->Set(String::New("database"), String::New(db->db.c_str()), ReadOnly);
        self->Set(String::New("port"),     db->port==0?Undefined():Integer::NewFromUnsigned(db->port), ReadOnly);
        self->Set(String::New("query"),    FunctionTemplate::New(WrapDbQuery)->GetFunction(), ReadOnly);
        self->Set(String::New("close"),    FunctionTemplate::New(WrapDbClose)->GetFunction(),   ReadOnly);
        self->SetInternalField(0, External::New(db));

        return handle_scope.Close(self);
    }
    catch (const exception &e)
    {
        return ThrowException(String::New(e.what()));
    }
#endif
}

// ==========================================================================
//                                 Services
// ==========================================================================

Handle<Value> InterpreterV8::Convert(char type, const char* &ptr)
{
    // Dim values are always unsigned per (FACT++) definition
    switch (type)
    {
    case 'F':
        {
            // Remove the "imprecision" effect coming from casting a float to
            // a double and then showing it with double precision
            ostringstream val;
            val << setprecision(7) << *reinterpret_cast<const float*>(ptr);
            ptr += 4;
            return Number::New(stod(val.str()));
        }
    case 'D':  { Handle<Value> v=Number::New(*reinterpret_cast<const double*>(ptr)); ptr+=8; return v; }
    case 'I':
    case 'L':  { Handle<Value> v=Integer::NewFromUnsigned(*reinterpret_cast<const uint32_t*>(ptr)); ptr += 4; return v; }
    case 'X':
        {
            const uint64_t val = *reinterpret_cast<const uint64_t*>(ptr);
            ptr += 8;
            if (val>UINT32_MAX)
                return Number::New(val);
            return Integer::NewFromUnsigned(val);
        }
    case 'S':  { Handle<Value> v=Integer::NewFromUnsigned(*reinterpret_cast<const uint16_t*>(ptr)); ptr += 2; return v; }
    case 'C':  { Handle<Value> v=Integer::NewFromUnsigned((uint16_t)*reinterpret_cast<const uint8_t*>(ptr));  ptr += 1; return v; }
    }
    return Undefined();
}

Handle<Value> InterpreterV8::FuncClose(const Arguments &args)
{
    HandleScope handle_scope;

    //const void *ptr = Local<External>::Cast(args.Holder()->GetInternalField(0))->Value();

    const String::AsciiValue str(args.This()->Get(String::New("name")));

    const auto it = fReverseMap.find(*str);
    if (it!=fReverseMap.end())
    {
        it->second.Dispose();
        fReverseMap.erase(it);
    }

    args.This()->Set(String::New("isOpen"), Boolean::New(false), ReadOnly);

    return handle_scope.Close(Boolean::New(JsUnsubscribe(*str)));
}

Handle<Value> InterpreterV8::ConvertEvent(const EventImp *evt, uint64_t counter, const char *str)
{
    const vector<Description> vec = JsDescription(str);

    Handle<Object> ret = fTemplateEvent->GetFunction()->NewInstance();//Object::New();
    if (ret.IsEmpty())
        return Undefined();

    const Local<Value> date = Date::New(evt->GetJavaDate());
    if (date.IsEmpty())
        return Undefined();

    ret->Set(String::New("name"),    String::New(str),              ReadOnly);
    ret->Set(String::New("format"),  String::New(evt->GetFormat().c_str()), ReadOnly);
    ret->Set(String::New("qos"),     Integer::New(evt->GetQoS()),   ReadOnly);
    ret->Set(String::New("size"),    Integer::New(evt->GetSize()),  ReadOnly);
    ret->Set(String::New("counter"), Integer::New(counter),         ReadOnly);
    if (evt->GetJavaDate()>0)
        ret->Set(String::New("time"), date, ReadOnly);

    // If names are available data will also be provided as an
    // object. If an empty event was received, but names are available,
    // the object will be empty. Otherwise 'obj' will be undefined.
    // obj===undefined:               no data received
    // obj!==undefined, length==0:    names for event available
    // obj!==undefined, obj.length>0: names available, data received
    Handle<Object> named = Object::New();
    if (vec.size()>0)
        ret->Set(String::New("obj"), named, ReadOnly);

    // If no event was received (usually a disconnection event in
    // the context of FACT++), no data is returned
    if (evt->IsEmpty())
        return ret;

    // If valid data was received, but the size was zero, then
    // null is returned as data
    // data===undefined: no data received
    // data===null:      event received, but no data
    // data.length>0:    event received, contains data
    if (evt->GetSize()==0 || evt->GetFormat().empty())
    {
        ret->Set(String::New("data"), Null(), ReadOnly);
        return ret;
    }

    typedef boost::char_separator<char> separator;
    const boost::tokenizer<separator> tokenizer(evt->GetFormat(), separator(";:"));

    const vector<string> tok(tokenizer.begin(), tokenizer.end());

    Handle<Object> arr = tok.size()>1 ? Array::New() : ret;
    if (arr.IsEmpty())
        return Undefined();

    const char *ptr = evt->GetText();
    const char *end = evt->GetText()+evt->GetSize();

    try
    {
        size_t pos = 1;
        for (auto it=tok.begin(); it<tok.end() && ptr<end; it++, pos++)
        {
            char type = (*it)[0];
            it++;

            string name = pos<vec.size() ? vec[pos].name : "";
            if (tok.size()==1)
                name = "data";

            // Get element size
            uint32_t sz = 1;
            switch (type)
            {
            case 'X':
            case 'D': sz = 8; break;
            case 'F':
            case 'I':
            case 'L': sz = 4; break;
            case 'S': sz = 2; break;
            case 'C': sz = 1; break;
            }

            // Check if no number is attached if the size of the
            // received data is consistent with the format string
            if (it==tok.end() && (end-ptr)%sz>0)
                return Exception::Error(String::New(("Number of received bytes ["+to_string(evt->GetSize())+"] does not match format ["+evt->GetFormat()+"]").c_str()));

            // Check if format has a number attached.
            // If no number is attached calculate number of elements
            const uint32_t cnt = it==tok.end() ? (end-ptr)/sz : stoi(it->c_str());

            // is_str: Array of type C but unknown size (String)
            // is_one: Array of known size, but size is 1 (I:1)
            const bool is_str = type=='C' && it==tok.end();
            const bool is_one = cnt==1    && it!=tok.end();

            Handle<Value> v;

            if (is_str)
                v = String::New(ptr);
            if (is_one)
                v = Convert(type, ptr);

            // Array of known (I:5) or unknown size (I), but no string
            if (!is_str && !is_one)
            {
                Handle<Object> a = Array::New(cnt);
                if (a.IsEmpty())
                    return Undefined();

                for (uint32_t i=0; i<cnt; i++)
                    a->Set(i, Convert(type, ptr));

                v = a;
            }

            if (tok.size()>1)
                arr->Set(pos-1, v);
            else
                ret->Set(String::New("data"), v, ReadOnly);

            if (!name.empty())
            {
                const Handle<String> n = String::New(name.c_str());
                named->Set(n, v);
            }
        }

        if (tok.size()>1)
            ret->Set(String::New("data"), arr, ReadOnly);

        return ret;
    }
    catch (...)
    {
        return Exception::Error(String::New(("Format string conversion '"+evt->GetFormat()+"' failed.").c_str()));
    }
}
/*
Handle<Value> InterpreterV8::FuncGetData(const Arguments &args)
{
    HandleScope handle_scope;

    const String::AsciiValue str(args.Holder()->Get(String::New("name")));

    const pair<uint64_t, EventImp *> p = JsGetEvent(*str);

    const EventImp *evt = p.second;
    if (!evt)
        return Undefined();

    //if (counter==cnt)
    //    return info.Holder();//Holder()->Get(String::New("data"));

    Handle<Value> ret = ConvertEvent(evt, p.first, *str);
    return ret->IsNativeError() ? ThrowException(ret) : handle_scope.Close(ret);
}
*/
Handle<Value> InterpreterV8::FuncGetData(const Arguments &args)
{
    if (args.Length()>2)
        return ThrowException(String::New("Number of arguments must not be greater than 2."));

    if (args.Length()>=1 && !args[0]->IsInt32() && !args[0]->IsNull())
        return ThrowException(String::New("Argument 1 not an uint32."));

    if (args.Length()==2 && !args[1]->IsBoolean())
        return ThrowException(String::New("Argument 2 not a boolean."));

    // Using a Javascript function has the advantage that it is fully
    // interruptable without the need of C++ code
    const bool    null    = args.Length()>=1 && args[0]->IsNull();
    const int32_t timeout = args.Length()>=1 ? args[0]->Int32Value() : 0;
    const bool    named   = args.Length()<2 || args[1]->BooleanValue();

    HandleScope handle_scope;

    const Handle<Script> sleep = Script::Compile(String::New("v8.sleep();"), String::New("internal"));
    if (sleep.IsEmpty())
        return Undefined();

    //const Handle<String> data   = String::New("data");
    const Handle<String> object = String::New("obj");

    const String::AsciiValue name(args.Holder()->Get(String::New("name")));

    TryCatch exception;

    Time t;
    while (!exception.HasCaught())
    {
        const pair<uint64_t, EventImp *> p = JsGetEvent(*name);

        const EventImp *evt = p.second;
        if (evt)
        {
            const Handle<Value> val = ConvertEvent(evt, p.first, *name);
            if (val->IsNativeError())
                return ThrowException(val);

            // Protect against the return of an exception
            if (!val.IsEmpty() && val->IsObject())
            {
                if (!named)
                    return handle_scope.Close(val);

                const Handle<Object> event = val->ToObject();
                const Handle<Value>  obj   = event->Get(object);

                if (!obj.IsEmpty() && obj->IsObject())
                {
                    // Has names and data was received?
                    if (obj->ToObject()->GetOwnPropertyNames()->Length()>0)
                        return handle_scope.Close(val);
                }
            }
        }

        if (args.Length()==0)
            break;

        if (!null && Time()-t>=boost::posix_time::milliseconds(abs(timeout)))
            break;

        // We cannot sleep directly because we have to give control back to
        // JavaScript ever now and then. This also allows us to catch
        // exceptions, either from the preemption or ConvertEvent
        sleep->Run();
    }

    if (exception.HasCaught())
        return exception.ReThrow();

    if (timeout<0)
        return Undefined();

    const string str = "Waiting for a valid event of "+string(*name)+" timed out.";
    return ThrowException(String::New(str.c_str()));
}


// This is a callback from the RemoteControl piping event handling
// to the java script ---> in test phase!
void InterpreterV8::JsHandleEvent(const EventImp &evt, uint64_t cnt, const string &service)
{
    const Locker locker;

    if (fThreadId<0)
        return;

    const auto it = fReverseMap.find(service);
    if (it==fReverseMap.end())
        return;

    const HandleScope handle_scope;

    Handle<Object> obj = it->second;

    obj->CreationContext()->Enter();

    const Handle<String> onchange = String::New("onchange");
    if (!obj->Has(onchange))
        return;

    const Handle<Value> val = obj->Get(onchange);
    if (!val->IsFunction())
        return;

    // -------------------------------------------------------------------

    TryCatch exception;

    const int id = V8::GetCurrentThreadId();
    fThreadIds.insert(id);

    Handle<Value> ret = ConvertEvent(&evt, cnt, service.c_str());
    if (ret->IsObject())
        Handle<Function>::Cast(val)->Call(obj, 1, &ret);

    fThreadIds.erase(id);

    if (!HandleException(exception, "Service.onchange"))
        V8::TerminateExecution(fThreadId);

    if (ret->IsNativeError())
    {
        JsException(service+".onchange callback - "+*String::AsciiValue(ret));
        V8::TerminateExecution(fThreadId);
    }
}

Handle<Value> InterpreterV8::OnChangeSet(Local<String> prop, Local<Value> value, const AccessorInfo &)
{
    // Returns the value if the setter intercepts the request. Otherwise, returns an empty handle.
    const string server = *String::AsciiValue(prop);
    auto it = fStateCallbacks.find(server);

    if (it!=fStateCallbacks.end())
    {
        it->second.Dispose();
        fStateCallbacks.erase(it);
    }

    if (value->IsFunction())
        fStateCallbacks[server] = Persistent<Object>::New(value->ToObject());

    return Handle<Value>();
}


void InterpreterV8::JsHandleState(const std::string &server, const State &state)
{
    const Locker locker;

    if (fThreadId<0)
        return;

    auto it = fStateCallbacks.find(server);
    if (it==fStateCallbacks.end())
    {
        it = fStateCallbacks.find("*");
        if (it==fStateCallbacks.end())
            return;
    }

    const HandleScope handle_scope;

    it->second->CreationContext()->Enter();

    // -------------------------------------------------------------------

    Handle<ObjectTemplate> obj = ObjectTemplate::New();
    obj->Set(String::New("server"),  String::New(server.c_str()), ReadOnly);

    if (state.index>-256)
    {
        obj->Set(String::New("index"),   Integer::New(state.index),          ReadOnly);
        obj->Set(String::New("name"),    String::New(state.name.c_str()),    ReadOnly);
        obj->Set(String::New("comment"), String::New(state.comment.c_str()), ReadOnly);
        const Local<Value> date = Date::New(state.time.JavaDate());
        if (!date.IsEmpty())
            obj->Set(String::New("time"), date);
    }

    // -------------------------------------------------------------------

    TryCatch exception;

    const int id = V8::GetCurrentThreadId();
    fThreadIds.insert(id);

    Handle<Value> args[] = { obj->NewInstance() };
    Handle<Function> fun = Handle<Function>(Function::Cast(*it->second));
    fun->Call(fun, 1, args);

    fThreadIds.erase(id);

    if (!HandleException(exception, "dim.onchange"))
        V8::TerminateExecution(fThreadId);
}

/*
void Cleanup( Persistent<Value> object, void *parameter )
{
    cout << "======================> RemoveMyObj()" << endl;
}*/

Handle<Value> InterpreterV8::FuncSubscription(const Arguments &args)
{
    if (args.Length()!=1 && args.Length()!=2)
        return ThrowException(String::New("Number of arguments must be one or two."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    if (args.Length()==2 && !args[1]->IsFunction())
        return ThrowException(String::New("Argument 2 must be a function."));

    const String::AsciiValue str(args[0]);

    if (!args.IsConstructCall())
    {
        const auto it = fReverseMap.find(*str);
        if (it!=fReverseMap.end())
            return it->second;

        return Undefined();
    }

    const HandleScope handle_scope;

    Handle<Object> self = args.This();
    self->Set(String::New("get"),    FunctionTemplate::New(WrapGetData)->GetFunction(),  ReadOnly);
    self->Set(String::New("close"),  FunctionTemplate::New(WrapClose)->GetFunction(),    ReadOnly);
    self->Set(String::New("name"),   String::New(*str), ReadOnly);
    self->Set(String::New("isOpen"), Boolean::New(true));

    if (args.Length()==2)
        self->Set(String::New("onchange"), args[1]);

    fReverseMap[*str] = Persistent<Object>::New(self);

    void *ptr = JsSubscribe(*str);
    if (ptr==0)
        return ThrowException(String::New(("Subscription to '"+string(*str)+"' already exists.").c_str()));

    self->SetInternalField(0, External::New(ptr));

    return Undefined();

    // Persistent<Object> p = Persistent<Object>::New(obj->NewInstance());
    // obj.MakeWeak((void*)1, Cleanup);
    // return obj;
}

// ==========================================================================
//                            Astrometry
// ==========================================================================
#ifdef HAVE_NOVA

double InterpreterV8::GetDataMember(const Arguments &args, const char *name)
{
    return args.This()->Get(String::New(name))->NumberValue();
}

Handle<Value> InterpreterV8::CalcDist(const Arguments &args, const bool local)
{
    if (args.Length()!=2)
        return ThrowException(String::New("dist must not be called with two arguments."));

    if (!args[0]->IsObject() || !args[1]->IsObject())
        return ThrowException(String::New("at least one argument not an object."));

    HandleScope handle_scope;

    Handle<Object> obj[2] =
    {
        Handle<Object>::Cast(args[0]),
        Handle<Object>::Cast(args[1])
    };

    const Handle<String> s_theta = String::New(local?"zd":"dec"); // was: zd
    const Handle<String> s_phi   = String::New(local?"az":"ra");  // was: az

    const double conv_t = M_PI/180;
    const double conv_p = local ? -M_PI/180 : M_PI/12;
    const double offset = local ? 0 : M_PI;

    const double theta0 = offset - obj[0]->Get(s_theta)->NumberValue() * conv_t;
    const double phi0   =          obj[0]->Get(s_phi  )->NumberValue() * conv_p;
    const double theta1 = offset - obj[1]->Get(s_theta)->NumberValue() * conv_t;
    const double phi1   =          obj[1]->Get(s_phi  )->NumberValue() * conv_p;

    if (!finite(theta0) || !finite(theta1) || !finite(phi0) || !finite(phi1))
        return ThrowException(String::New("some values not valid or not finite."));

    /*
    const double x0 = sin(zd0) * cos(az0);   // az0 -= az0
    const double y0 = sin(zd0) * sin(az0);   // az0 -= az0
    const double z0 = cos(zd0);

    const double x1 = sin(zd1) * cos(az1);   // az1 -= az0
    const double y1 = sin(zd1) * sin(az1);   // az1 -= az0
    const double z1 = cos(zd1);

    const double res = acos(x0*x1 + y0*y1 + z0*z1) * 180/M_PI;
    */

    // cos(az1-az0) = cos(az1)*cos(az0) + sin(az1)*sin(az0)

    const double x = sin(theta0) * sin(theta1) * cos(phi1-phi0);
    const double y = cos(theta0) * cos(theta1);

    const double res = acos(x + y) * 180/M_PI;

    return handle_scope.Close(Number::New(res));
}

Handle<Value> InterpreterV8::LocalDist(const Arguments &args)
{
    return CalcDist(args, true);
}

Handle<Value> InterpreterV8::SkyDist(const Arguments &args)
{
    return CalcDist(args, false);
}

Handle<Value> InterpreterV8::MoonDisk(const Arguments &args)
{
    if (args.Length()>1)
        return ThrowException(String::New("disk must not be called with more than one argument."));

    const uint64_t v = uint64_t(args[0]->NumberValue());
    const Time utc = args.Length()==0 ? Time() : Time(v/1000, v%1000);

    return Number::New(ln_get_lunar_disk(utc.JD()));
}

Handle<Value> InterpreterV8::LocalToSky(const Arguments &args)
{
    if (args.Length()>1)
        return ThrowException(String::New("toSky must not be called with more than one argument."));

    if (args.Length()==1 && !args[0]->IsDate())
        return ThrowException(String::New("Argument must be a Date"));

    ln_hrz_posn hrz;
    hrz.alt = 90-GetDataMember(args, "zd");
    hrz.az  =    GetDataMember(args, "az");

    if (!finite(hrz.alt) || !finite(hrz.az))
        return ThrowException(String::New("zd and az must be finite."));

    HandleScope handle_scope;

    const Local<Value> date =
        args.Length()==0 ? Date::New(Time().JavaDate()) : args[0];
    if (date.IsEmpty())
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_lnlat_posn obs;
    obs.lng = -(17.+53./60+26.525/3600);
    obs.lat =   28.+45./60+42.462/3600;

    ln_equ_posn equ;
    ln_get_equ_from_hrz(&hrz, &obs, utc.JD(), &equ);

    // -----------------------------

    Handle<Value> arg[] = { Number::New(equ.ra/15), Number::New(equ.dec), date };
    return handle_scope.Close(fTemplateSky->GetFunction()->NewInstance(3, arg));
}

Handle<Value> InterpreterV8::SkyToLocal(const Arguments &args)
{
    if (args.Length()>1)
        return ThrowException(String::New("toLocal must not be called with more than one argument."));

    if (args.Length()==1 && !args[0]->IsDate())
        return ThrowException(String::New("Argument must be a Date"));

    ln_equ_posn equ;
    equ.ra  = GetDataMember(args, "ra")*15;
    equ.dec = GetDataMember(args, "dec");

    if (!finite(equ.ra) || !finite(equ.dec))
        return ThrowException(String::New("Ra and dec must be finite."));

    HandleScope handle_scope;

    const Local<Value> date =
        args.Length()==0 ? Date::New(Time().JavaDate()) : args[0];
    if (date.IsEmpty())
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_lnlat_posn obs;
    obs.lng = -(17.+53./60+26.525/3600);
    obs.lat =   28.+45./60+42.462/3600;

    ln_hrz_posn hrz;
    ln_get_hrz_from_equ(&equ, &obs, utc.JD(), &hrz);

    Handle<Value> arg[] = { Number::New(90-hrz.alt), Number::New(hrz.az), date };
    return handle_scope.Close(fTemplateLocal->GetFunction()->NewInstance(3, arg));
}

Handle<Value> InterpreterV8::MoonToLocal(const Arguments &args)
{
    if (args.Length()>0)
        return ThrowException(String::New("toLocal must not be called with arguments."));

    ln_equ_posn equ;
    equ.ra  = GetDataMember(args, "ra")*15;
    equ.dec = GetDataMember(args, "dec");

    if (!finite(equ.ra) || !finite(equ.dec))
        return ThrowException(String::New("ra and dec must be finite."));

    HandleScope handle_scope;

    const Local<Value> date = args.This()->Get(String::New("time"));
    if (date.IsEmpty() || date->IsUndefined() )
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_lnlat_posn obs;
    obs.lng = -(17.+53./60+26.525/3600);
    obs.lat =   28.+45./60+42.462/3600;

    ln_hrz_posn hrz;
    ln_get_hrz_from_equ(&equ, &obs, utc.JD(), &hrz);

    Handle<Value> arg[] = { Number::New(90-hrz.alt), Number::New(hrz.az), date };
    return handle_scope.Close(fTemplateLocal->GetFunction()->NewInstance(3, arg));
}

Handle<Value> InterpreterV8::ConstructorMoon(const Arguments &args)
{
    if (args.Length()>1)
        return ThrowException(String::New("Moon constructor must not be called with more than one argument."));

    if (args.Length()==1 && !args[0]->IsDate())
        return ThrowException(String::New("Argument must be a Date"));

    HandleScope handle_scope;

    const Local<Value> date =
        args.Length()==0 ? Date::New(Time().JavaDate()) : args[0];
    if (date.IsEmpty())
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_equ_posn equ;
    ln_get_lunar_equ_coords_prec(utc.JD(), &equ, 0.01);

    // ----------------------------

    if (!args.IsConstructCall())
        return Constructor(args);

    Handle<Function> function =
        FunctionTemplate::New(MoonToLocal)->GetFunction();
    if (function.IsEmpty())
        return Undefined();

    Handle<Object> self = args.This();
    self->Set(String::New("ra"),      Number::New(equ.ra/15),  ReadOnly);
    self->Set(String::New("dec"),     Number::New(equ.dec),    ReadOnly);
    self->Set(String::New("toLocal"), function,                ReadOnly);
    self->Set(String::New("time"),    date,                    ReadOnly);

    return handle_scope.Close(self);
}

Handle<Value> InterpreterV8::ConstructorSky(const Arguments &args)
{
    if (args.Length()<2 || args.Length()>3)
        return ThrowException(String::New("Sky constructor takes two or three arguments."));

    if (args.Length()==3 && !args[2]->IsDate())
        return ThrowException(String::New("Third argument must be a Date."));

    const double ra  = args[0]->NumberValue();
    const double dec = args[1]->NumberValue();

    if (!finite(ra) || !finite(dec))
        return ThrowException(String::New("Both arguments to Sky must be valid numbers."));

    // ----------------------------

    HandleScope handle_scope;

    if (!args.IsConstructCall())
        return Constructor(args);

    Handle<Function> function =
        FunctionTemplate::New(SkyToLocal)->GetFunction();
    if (function.IsEmpty())
        return Undefined();

    Handle<Object> self = args.This();
    self->Set(String::New("ra"),      Number::New(ra),  ReadOnly);
    self->Set(String::New("dec"),     Number::New(dec), ReadOnly);
    self->Set(String::New("toLocal"), function,         ReadOnly);
    if (args.Length()==3)
        self->Set(String::New("time"), args[2], ReadOnly);

    return handle_scope.Close(self);
}

Handle<Value> InterpreterV8::ConstructorLocal(const Arguments &args)
{
    if (args.Length()<2 || args.Length()>3)
        return ThrowException(String::New("Local constructor takes two or three arguments."));

    if (args.Length()==3 && !args[2]->IsDate())
        return ThrowException(String::New("Third argument must be a Date."));

    const double zd = args[0]->NumberValue();
    const double az = args[1]->NumberValue();

    if (!finite(zd) || !finite(az))
        return ThrowException(String::New("Both arguments to Local must be valid numbers."));

    // --------------------

    HandleScope handle_scope;

    if (!args.IsConstructCall())
        return Constructor(args);

    Handle<Function> function =
        FunctionTemplate::New(LocalToSky)->GetFunction();
    if (function.IsEmpty())
        return Undefined();

    Handle<Object> self = args.This();
    self->Set(String::New("zd"),    Number::New(zd), ReadOnly);
    self->Set(String::New("az"),    Number::New(az), ReadOnly);
    self->Set(String::New("toSky"), function,        ReadOnly);
    if (args.Length()==3)
        self->Set(String::New("time"), args[2], ReadOnly);

    return handle_scope.Close(self);
}

Handle<Object> InterpreterV8::ConstructRiseSet(const Handle<Value> time, const ln_rst_time &rst, const bool &rc)
{
    Handle<Object> obj = Object::New();
    obj->Set(String::New("time"), time, ReadOnly);

    const uint64_t v = uint64_t(time->NumberValue());
    const double jd = Time(v/1000, v%1000).JD();

    const bool isUp = rc>0 ||
        (rst.rise<rst.set && (jd>rst.rise && jd<rst.set)) ||
        (rst.rise>rst.set && (jd<rst.set  || jd>rst.rise));

    obj->Set(String::New("isUp"), Boolean::New(rc>=0 && isUp), ReadOnly);

    if (rc!=0)
        return obj;

    Handle<Value> rise  = Date::New(Time(rst.rise).JavaDate());
    Handle<Value> set   = Date::New(Time(rst.set).JavaDate());
    Handle<Value> trans = Date::New(Time(rst.transit).JavaDate());
    if (rise.IsEmpty() || set.IsEmpty() || trans.IsEmpty())
        return Handle<Object>();

    obj->Set(String::New("rise"), rise, ReadOnly);
    obj->Set(String::New("set"), set, ReadOnly);
    obj->Set(String::New("transit"), trans, ReadOnly);

    return obj;
}

Handle<Value> InterpreterV8::SunHorizon(const Arguments &args)
{
    if (args.Length()>2)
        return ThrowException(String::New("Sun.horizon must not be called with one or two arguments."));

    if (args.Length()==2 && !args[1]->IsDate())
        return ThrowException(String::New("Second argument must be a Date"));

    HandleScope handle_scope;

    double hrz = NAN;
    if (args.Length()==0 || args[0]->IsNull())
        hrz = LN_SOLAR_STANDART_HORIZON;
    if (args.Length()>0 && args[0]->IsNumber())
        hrz = args[0]->NumberValue();
    if (args.Length()>0 && args[0]->IsString())
    {
        string arg(Tools::Trim(*String::AsciiValue(args[0])));
        transform(arg.begin(), arg.end(), arg.begin(), ::tolower);

        if (arg==string("horizon").substr(0, arg.length()))
            hrz = LN_SOLAR_STANDART_HORIZON;
        if (arg==string("civil").substr(0, arg.length()))
            hrz = LN_SOLAR_CIVIL_HORIZON;
        if (arg==string("nautical").substr(0, arg.length()))
            hrz = LN_SOLAR_NAUTIC_HORIZON;
        if (arg==string("fact").substr(0, arg.length()))
            hrz = -15;
        if (arg==string("astronomical").substr(0, arg.length()))
            hrz = LN_SOLAR_ASTRONOMICAL_HORIZON;
    }

    if (!finite(hrz))
        return ThrowException(String::New("Second argument did not yield a valid number."));

    const Local<Value> date =
        args.Length()<2 ? Date::New(Time().JavaDate()) : args[1];
    if (date.IsEmpty())
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_lnlat_posn obs;
    obs.lng = -(17.+53./60+26.525/3600);
    obs.lat =   28.+45./60+42.462/3600;

    // get Julian day from local time
    const double JD = utc.JD();

    ln_rst_time sun;
    const int rc = ln_get_solar_rst_horizon(JD-0.5, &obs, hrz, &sun);
    Handle<Object> rst = ConstructRiseSet(date, sun, rc);
    rst->Set(String::New("horizon"), Number::New(hrz));
    return handle_scope.Close(rst);
};

Handle<Value> InterpreterV8::MoonHorizon(const Arguments &args)
{
    if (args.Length()>1)
        return ThrowException(String::New("Moon.horizon must not be called with one argument."));

    if (args.Length()==1 && !args[0]->IsDate())
        return ThrowException(String::New("Argument must be a Date"));

    HandleScope handle_scope;

    const Local<Value> date =
        args.Length()==0 ? Date::New(Time().JavaDate()) : args[0];
    if (date.IsEmpty())
        return Undefined();

    const uint64_t v = uint64_t(date->NumberValue());
    const Time utc(v/1000, v%1000);

    ln_lnlat_posn obs;
    obs.lng = -(17.+53./60+26.525/3600);
    obs.lat =   28.+45./60+42.462/3600;

    // get Julian day from local time
    const double JD = utc.JD();

    ln_rst_time moon;
    const int rc = ln_get_lunar_rst(JD-0.5, &obs, &moon);
    Handle<Object> rst = ConstructRiseSet(date, moon, rc);
    return handle_scope.Close(rst);
};
#endif

// ==========================================================================
//                            Process control
// ==========================================================================

bool InterpreterV8::HandleException(TryCatch& try_catch, const char *where)
{
    if (!try_catch.HasCaught() || !try_catch.CanContinue())
        return true;

    const HandleScope handle_scope;

    Handle<Value> except = try_catch.Exception();
    if (except.IsEmpty() || except->IsNull())
        return true;

    const String::AsciiValue exception(except);

    const Handle<Message> message = try_catch.Message();
    if (message.IsEmpty())
        return false;

    ostringstream out;

    if (!message->GetScriptResourceName()->IsUndefined())
    {
        // Print (filename):(line number): (message).
        const String::AsciiValue filename(message->GetScriptResourceName());
        if (filename.length()>0)
        {
            out << *filename;
            if (message->GetLineNumber()>0)
                out << ": l." << message->GetLineNumber();
            if (*exception)
                out << ": ";
        }
    }

    if (*exception)
        out << *exception;

    out << " [" << where << "]";

    JsException(out.str());

    // Print line of source code.
    const String::AsciiValue sourceline(message->GetSourceLine());
    if (*sourceline)
        JsException(*sourceline);

    // Print wavy underline (GetUnderline is deprecated).
    const int start = message->GetStartColumn();
    const int end   = message->GetEndColumn();

    out.str("");
    if (start>0)
        out << setfill(' ') << setw(start) << ' ';
    out << setfill('^') << setw(end-start) << '^';

    JsException(out.str());

    const String::AsciiValue stack_trace(try_catch.StackTrace());
    if (stack_trace.length()<=0)
        return false;

    if (!*stack_trace)
        return false;

    const string trace(*stack_trace);

    typedef boost::char_separator<char> separator;
    const boost::tokenizer<separator> tokenizer(trace, separator("\n"));

    // maybe skip: "    at internal:"

    auto it = tokenizer.begin();
    JsException("");
    while (it!=tokenizer.end())
        JsException(*it++);

    return false;
}

Handle<Value> InterpreterV8::ExecuteInternal(const string &code)
{
    // Try/catch and re-throw hides our internal code from
    // the displayed exception showing the origin and shows
    // the user function instead.
    TryCatch exception;

    const Handle<Value> result = ExecuteCode(code);
    if (exception.HasCaught())
        exception.ReThrow();

    return result;
}

Handle<Value> InterpreterV8::ExecuteCode(const string &code, const string &file, bool main)
{
    HandleScope handle_scope;

    const Handle<String> source = String::New(code.c_str(), code.size());
    const Handle<String> origin = String::New(file.c_str());
    if (source.IsEmpty())
        return Handle<Value>();

    const Handle<Script> script = Script::Compile(source, origin);
    if (script.IsEmpty())
        return Handle<Value>();

    if (main)
        JsSetState(3);

    TryCatch exception;

    const Handle<Value> result = script->Run();

    if (exception.HasCaught())
    {
        if (file=="internal")
            return exception.ReThrow();

        HandleException(exception, "code");
        return Handle<Value>();
    }

    // If all went well and the result wasn't undefined then print
    // the returned value.
    if (!result.IsEmpty() && !result->IsUndefined() && file!="internal")
        JsResult(*String::AsciiValue(result));

    return handle_scope.Close(result);
}

bool InterpreterV8::ExecuteFile(const string &name, bool main)
{
    ifstream fin(name.c_str());
    if (!fin)
    {
        JsException("Error - Could not open file '"+name+"'");
        return false;
    }

    string buffer;
    if (!getline(fin, buffer, '\0'))
        return true;

    if (fin.fail())
    {
        JsException("Error - reading file.");
        return false;
    }

    return !ExecuteCode(buffer, name, main).IsEmpty();
}

bool InterpreterV8::ExecuteConsole()
{
    JsSetState(3);

    WindowLog lout;
    lout << "\n " << kUnderline << " JavaScript interpreter " << kReset << " (enter '.q' to quit)\n" << endl;

    Readline::StaticPushHistory("java.his");

    string command;

    while (1)
    {
        const string buffer = Tools::Trim(Readline::StaticPrompt(command.empty() ? "JS> " : " \\> "));
        if (buffer==".q")
            break;

        // buffer empty, do nothing
        if (buffer.empty())
            continue;

        // Compose command
        if (!command.empty())
            command += ' ';
        command += buffer;

        // If line ends with a backslash, allow addition of next line
        auto back = command.rbegin();
        if (*back=='\\')
        {
            *back = ' ';
            command = Tools::Trim(command);
            continue;
        }

        // Catch exceptions during code compilation
        TryCatch exception;

        // Execute code which was entered
        bool rc = ExecuteCode(command, "", false).IsEmpty();
        if (exception.HasCaught())
        {
            HandleException(exception, "compile");
            rc = true;
        }

        // In case of an exception add an empty line to the output
        if (rc)
            lout << endl;

        // command has been executed, collect new command
        command = "";
    }

    lout << endl;

    Readline::StaticPopHistory("java.his");

    return true;
}

// ==========================================================================
//                                  CORE
// ==========================================================================

InterpreterV8::InterpreterV8() : fThreadId(-1)
{
    const string ver(V8::GetVersion());

    typedef boost::char_separator<char> separator;
    const boost::tokenizer<separator> tokenizer(ver, separator("."));

    const vector<string> tok(tokenizer.begin(), tokenizer.end());

    const int major = tok.size()>0 ? stol(tok[0]) : -1;
    const int minor = tok.size()>1 ? stol(tok[1]) : -1;
    const int build = tok.size()>2 ? stol(tok[2]) : -1;

    if (major>3 || (major==3 && minor>9) || (major==3 && minor==9 && build>10))
    {
        const string argv = "--use_strict";
        V8::SetFlagsFromString(argv.c_str(), argv.size());
    }

    This = this;
}

Handle<Value> InterpreterV8::Constructor(/*Handle<FunctionTemplate> T,*/ const Arguments &args)
{
    Handle<Value> argv[args.Length()];

    for (int i=0; i<args.Length(); i++)
        argv[i] = args[i];

    return args.Callee()->NewInstance(args.Length(), argv);
}


void InterpreterV8::AddFormatToGlobal()// const
{
    const string code =
        "String.form = function(str, arr)"
        "{"
            "var i = -1;"
            "function callback(exp, p0, p1, p2, p3, p4/*, pos, str*/)"
            "{"
                "if (exp=='%%')"
                    "return '%';"
                ""
                "if (arr[++i]===undefined)"
                    "return undefined;"
                ""
                "var exp  = p2 ? parseInt(p2.substr(1)) : undefined;"
                "var base = p3 ? parseInt(p3.substr(1)) : undefined;"
                ""
                "var val;"
                "switch (p4)"
                "{"
                "case 's': val = arr[i]; break;"
                "case 'c': val = arr[i][0]; break;"
                "case 'f': val = parseFloat(arr[i]).toFixed(exp); break;"
                "case 'p': val = parseFloat(arr[i]).toPrecision(exp); break;"
                "case 'e': val = parseFloat(arr[i]).toExponential(exp); break;"
                "case 'x': val = parseInt(arr[i]).toString(base?base:16); break;"
                "case 'd': val = parseFloat(parseInt(arr[i], base?base:10).toPrecision(exp)).toFixed(0); break;"
                //"default:\n"
                //"    throw new SyntaxError('Conversion specifier '+p4+' unknown.');\n"
                "}"
                ""
                "val = typeof(val)=='object' ? JSON.stringify(val) : val.toString(base);"
                ""
                "var sz = parseInt(p1); /* padding size */"
                "var ch = p1 && p1[0]=='0' ? '0' : ' '; /* isnull? */"
                "while (val.length<sz)"
                    "val = p0 !== undefined ? val+ch : ch+val; /* isminus? */"
                ""
                "return val;"
            "}"
            ""
            "var regex = /%(-)?(0?[0-9]+)?([.][0-9]+)?([#][0-9]+)?([scfpexd])/g;"
            "return str.replace(regex, callback);"
        "}"
        "\n"
        "String.prototype.$ = function()"
        "{"
            "return String.form(this, Array.prototype.slice.call(arguments));"
        "}"/*
        "\n"
        "var format = function()"
        "{"
            "return dim.format(arguments[0], Array.prototype.slice.call(arguments,1));"
        "}"*/;

    // ExcuteInternal does not work properly here...
    // If suring compilation an exception is thrown, it will not work
    Handle<Script> script = Script::New(String::New(code.c_str()), String::New("internal"));
    if (!script.IsEmpty())
        script->Run();
}

bool InterpreterV8::JsRun(const string &filename, const map<string, string> &map)
{
    const Locker locker;
    fThreadId = V8::GetCurrentThreadId();

    JsPrint(string("JavaScript Engine V8 ")+V8::GetVersion());

    JsLoad(filename);

    const HandleScope handle_scope;

    // Create a template for the global object.
    Handle<ObjectTemplate> dim = ObjectTemplate::New();
    dim->Set(String::New("log"),       FunctionTemplate::New(WrapLog),       ReadOnly);
    dim->Set(String::New("alarm"),     FunctionTemplate::New(WrapAlarm),     ReadOnly);
    dim->Set(String::New("wait"),      FunctionTemplate::New(WrapWait),      ReadOnly);
    dim->Set(String::New("send"),      FunctionTemplate::New(WrapSend),      ReadOnly);
    dim->Set(String::New("state"),     FunctionTemplate::New(WrapState),     ReadOnly);
    dim->Set(String::New("version"),   Integer::New(DIM_VERSION_NUMBER),     ReadOnly);
    dim->Set(String::New("getStates"), FunctionTemplate::New(WrapGetStates), ReadOnly);
    dim->Set(String::New("getDescription"), FunctionTemplate::New(WrapGetDescription), ReadOnly);
    dim->Set(String::New("getServices"), FunctionTemplate::New(WrapGetServices), ReadOnly);

    Handle<ObjectTemplate> dimctrl = ObjectTemplate::New();
    dimctrl->Set(String::New("defineState"), FunctionTemplate::New(WrapNewState),  ReadOnly);
    dimctrl->Set(String::New("setState"),    FunctionTemplate::New(WrapSetState),  ReadOnly);
    dimctrl->Set(String::New("getState"),    FunctionTemplate::New(WrapGetState),  ReadOnly);

    Handle<ObjectTemplate> v8 = ObjectTemplate::New();
    v8->Set(String::New("sleep"),   FunctionTemplate::New(WrapSleep), ReadOnly);
    v8->Set(String::New("version"), String::New(V8::GetVersion()),    ReadOnly);

    Handle<ObjectTemplate> console = ObjectTemplate::New();
    console->Set(String::New("out"), FunctionTemplate::New(WrapOut), ReadOnly);

    Handle<ObjectTemplate> onchange = ObjectTemplate::New();
    onchange->SetNamedPropertyHandler(OnChangeGet, WrapOnChangeSet);
    dim->Set(String::New("onchange"), onchange);

    Handle<ObjectTemplate> global = ObjectTemplate::New();
    global->Set(String::New("v8"),      v8,      ReadOnly);
    global->Set(String::New("dim"),     dim,     ReadOnly);
    global->Set(String::New("dimctrl"), dimctrl, ReadOnly);
    global->Set(String::New("console"), console, ReadOnly);
    global->Set(String::New("include"), FunctionTemplate::New(WrapInclude),                ReadOnly);
    global->Set(String::New("exit"),    FunctionTemplate::New(WrapExit),                   ReadOnly);

    Handle<FunctionTemplate> sub = FunctionTemplate::New(WrapSubscription);
    sub->SetClassName(String::New("Subscription"));
    sub->InstanceTemplate()->SetInternalFieldCount(1);
    global->Set(String::New("Subscription"), sub, ReadOnly);

    Handle<FunctionTemplate> db = FunctionTemplate::New(WrapDatabase);
    db->SetClassName(String::New("Database"));
    db->InstanceTemplate()->SetInternalFieldCount(1);
    global->Set(String::New("Database"), db, ReadOnly);

    Handle<FunctionTemplate> thread = FunctionTemplate::New(WrapThread);
    thread->SetClassName(String::New("Thread"));
    global->Set(String::New("Thread"), thread, ReadOnly);

    Handle<FunctionTemplate> file = FunctionTemplate::New(WrapFile);
    file->SetClassName(String::New("File"));
    global->Set(String::New("File"), file, ReadOnly);

    Handle<FunctionTemplate> evt = FunctionTemplate::New();
    evt->SetClassName(String::New("Event"));
    global->Set(String::New("Event"), evt, ReadOnly);

    fTemplateEvent = evt;

#ifdef HAVE_NOVA
    Handle<FunctionTemplate> sky = FunctionTemplate::New(ConstructorSky);
    sky->SetClassName(String::New("Sky"));
    sky->Set(String::New("dist"),  FunctionTemplate::New(SkyDist), ReadOnly);
    global->Set(String::New("Sky"), sky, ReadOnly);

    Handle<FunctionTemplate> loc = FunctionTemplate::New(ConstructorLocal);
    loc->SetClassName(String::New("Local"));
    loc->Set(String::New("dist"),  FunctionTemplate::New(LocalDist), ReadOnly);
    global->Set(String::New("Local"), loc, ReadOnly);

    Handle<FunctionTemplate> moon = FunctionTemplate::New(ConstructorMoon);
    moon->SetClassName(String::New("Moon"));
    moon->Set(String::New("disk"), FunctionTemplate::New(MoonDisk), ReadOnly);
    moon->Set(String::New("horizon"), FunctionTemplate::New(MoonHorizon), ReadOnly);
    global->Set(String::New("Moon"), moon, ReadOnly);

    Handle<FunctionTemplate> sun = FunctionTemplate::New();
    sun->SetClassName(String::New("Sun"));
    sun->Set(String::New("horizon"), FunctionTemplate::New(SunHorizon), ReadOnly);
    global->Set(String::New("Sun"), sun, ReadOnly);

    fTemplateLocal = loc;
    fTemplateSky   = sky;
#endif

    // Persistent
    Persistent<Context> context = Context::New(NULL, global);
    if (context.IsEmpty())
    {
        JsException("Creation of global context failed...");
        JsEnd(filename);
        return false;
    }

    // Switch off eval(). It is not possible to track it's exceptions.
    context->AllowCodeGenerationFromStrings(false);

    Context::Scope scope(context);

    Handle<Array> args = Array::New(map.size());
    for (auto it=map.begin(); it!=map.end(); it++)
        args->Set(String::New(it->first.c_str()), String::New(it->second.c_str()));
    context->Global()->Set(String::New("$"),   args, ReadOnly);
    context->Global()->Set(String::New("arg"), args, ReadOnly);

    //V8::ResumeProfiler();

    TryCatch exception;

    AddFormatToGlobal();

    bool rc = true;
    if (!exception.HasCaught())
    {
        JsStart(filename);

        Locker::StartPreemption(10);

        rc &= filename.empty() ? ExecuteConsole() : ExecuteFile(filename, true);

        Locker::StopPreemption();

        // Stop all other threads
        for (auto it=fThreadIds.begin(); it!=fThreadIds.end(); it++)
            V8::TerminateExecution(*it);
        fThreadIds.clear();
    }

    // Handle an exception
    rc &= HandleException(exception, "main");

    // IsProfilerPaused()
    // V8::PauseProfiler();

    // -----
    // This is how an exit handler could look like, but there is no way to interrupt it
    // -----
    // Handle<Object> obj = Handle<Object>::Cast(context->Global()->Get(String::New("dim")));
    // if (!obj.IsEmpty())
    // {
    //     Handle<Value> onexit = obj->Get(String::New("onexit"));
    //     if (!onexit->IsUndefined())
    //         Handle<Function>::Cast(onexit)->NewInstance(0, NULL); // argc, argv
    //     // Handle<Object> result = Handle<Function>::Cast(onexit)->NewInstance(0, NULL); // argc, argv
    // }

    //context->Exit();

    // The threads are started already and wait to get the lock
    // So we have to unlock (manual preemtion) so that they get
    // the signal to terminate.
    {
        const Unlocker unlock;

        for (auto it=fThreads.begin(); it!=fThreads.end(); it++)
            it->join();
        fThreads.clear();
    }

    // Now we can dispose all persistent handles from state callbacks
    for (auto it=fStateCallbacks.begin(); it!=fStateCallbacks.end(); it++)
        it->second.Dispose();
    fStateCallbacks.clear();

    // Now we can dispose all persistent handles from reverse maps
    for (auto it=fReverseMap.begin(); it!=fReverseMap.end(); it++)
        it->second.Dispose();
    fReverseMap.clear();

#ifdef HAVE_SQL
    // ...and close all database  handles
    for (auto it=fDatabases.begin(); it!=fDatabases.end(); it++)
        delete *it;
    fDatabases.clear();
#endif

    fStates.clear();

    context.Dispose();

    JsEnd(filename);

    return true;
}

void InterpreterV8::JsStop()
{
    Locker locker;
    V8::TerminateExecution(This->fThreadId);
}

#endif

InterpreterV8 *InterpreterV8::This = 0;
