#ifndef FACT_InterpreterV8
#define FACT_InterpreterV8

#include <map>
#include <set>
#include <list>
#include <string>
#include <thread>

#ifdef HAVE_V8
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wshadow"
#include <v8.h>
#pragma GCC diagnostic pop
#endif

#include "State.h"
#include "Service.h"
#include "Description.h"
#include "EventImp.h"

class Database;
namespace v8
{
    class Platform;
};

#ifdef HAVE_NOVA
struct ln_rst_time;
#endif

class InterpreterV8
{
    template<class T, class S>
       using PersistentMap = std::map<T, v8::UniquePersistent<S>>;

    static InterpreterV8 *This;
    static std::unique_ptr<v8::Platform> fPlatform;

    // The main thread id, needed to be able to terminate
    // the thread forcefully from 'the outside'
    v8::Isolate *fMainThread;

    // A loookup table which allows to indentify
    // the JavaScript object corrsponding to the
    // service name (for checking of an .onchange
    // subscription exists for that object)
    PersistentMap<std::string, v8::Object> fReverseMap;

    // Lookup table for the callbacks in cases of state changes
    PersistentMap<std::string, v8::Object> fStateCallbacks;

    // List of all threads
    std::vector<std::thread> fThreads;

    // List of all states already set
    std::vector<std::pair<int, std::string>> fStates;

    // Interrupt handler
    v8::UniquePersistent<v8::Object> fInterruptCallback;

    static v8::Handle<v8::FunctionTemplate> fTemplateLocal;
    static v8::Handle<v8::FunctionTemplate> fTemplateSky;
    static v8::Handle<v8::FunctionTemplate> fTemplateEvent;
    static v8::Handle<v8::FunctionTemplate> fTemplateDescription;

#ifdef HAVE_SQL
    std::list<Database*> fDatabases;
#endif

    std::string fIncludePath;

#ifdef HAVE_V8
    bool HandleException(v8::TryCatch &try_catch, const char *where);
    void ExecuteConsole();
    v8::Handle<v8::Value> ExecuteCode(const std::string &code, const std::string &file="internal");
    v8::Handle<v8::Value> ExecuteInternal(const std::string &code);

    void Thread(v8::UniquePersistent<v8::Object> &_this, v8::UniquePersistent<v8::Object> &func, uint32_t ms);

    std::vector<std::string> ValueToArray(const v8::Handle<v8::Value> &val, bool only=true);

    typedef v8::FunctionCallbackInfo<v8::Value> FactArguments;

    static void TerminateExecution(v8::Isolate*);

    void FuncWait(const FactArguments& args);
    void FuncSend(const FactArguments& args);
    void FuncSleep(const FactArguments& args);
    void FuncTimeout(const FactArguments& args);
    void FuncThread(const FactArguments& args);
    void FuncKill(const FactArguments& args);
    void FuncLog(const FactArguments& args);
    void FuncAlarm(const FactArguments& args);
    void FuncOut(const FactArguments& args);
    void FuncWarn(const FactArguments& args);
    void FuncFile(const FactArguments& args);
    void FuncSendMail(const FactArguments& args);
    void FuncSendCurl(const FactArguments& args);
    void FuncInclude(const FactArguments& args);
    void FuncExit(const FactArguments& args);
    void FuncState(const FactArguments& args);
    void FuncSetState(const FactArguments& args);
    void FuncGetState(const FactArguments& args);
    void FuncGetStates(const FactArguments& args);
    void FuncGetDescription(const FactArguments& args);
    void FuncGetServices(const FactArguments& args);
    void FuncNewState(const FactArguments& args);
    void FuncSetInterrupt(const FactArguments& args);
    void FuncTriggerInterrupt(const FactArguments& args);
    //v8::Handle<v8::Value> FuncOpen(const FactArguments& args);
    void FuncSubscription(const FactArguments& args);
    void FuncGetData(const FactArguments &args);
    void FuncClose(const FactArguments &args);
    void FuncQuery(const FactArguments &args);
    void FuncDatabase(const FactArguments &args);
    void FuncDbQuery(const FactArguments &args);
    void FuncDbClose(const FactArguments &args);
    void OnChangeSet(v8::Local<v8::Name>, v8::Local<v8::Value>, const v8::PropertyCallbackInfo<v8::Value> &);

    static v8::Handle<v8::Value> Constructor(const FactArguments &args);

    static void ConstructorMail(const FactArguments &args);
    static void ConstructorCurl(const FactArguments &args);

#ifdef HAVE_NOVA
    static double GetDataMember(const FactArguments &args, const char *name);

    static void CalcDist(const FactArguments &args, const bool);

    static void LocalToString(const FactArguments &args);
    static void SkyToString(const FactArguments &args);
    static void MoonToString(const FactArguments &args);
    static void LocalDist(const FactArguments &args);
    static void SkyDist(const FactArguments &args);
    static void MoonDisk(const FactArguments &args);
    static void LocalToSky(const FactArguments &args);
    static void SkyToLocal(const FactArguments &args);
    static void MoonToLocal(const FactArguments &args);
    static void ConstructorMoon(const FactArguments &args);
    static void ConstructorSky(const FactArguments &args);
    static void ConstructorLocal(const FactArguments &args);
    static void MoonHorizon(const FactArguments &args);
    static void SunHorizon(const FactArguments &args);
#endif

    static void WrapInclude(const FactArguments &args)  { if (This) This->FuncInclude(args);  }
    static void WrapFile(const FactArguments &args)     { if (This) This->FuncFile(args);     }
    static void WrapSendMail(const FactArguments &args) { if (This) This->FuncSendMail(args); }
    static void WrapSendCurl(const FactArguments &args) { if (This) This->FuncSendCurl(args); }
    static void WrapLog(const FactArguments &args)      { if (This) This->FuncLog(args);      }
    static void WrapAlarm(const FactArguments &args)    { if (This) This->FuncAlarm(args);    }
    static void WrapOut(const FactArguments &args)      { if (This) This->FuncOut(args);      }
    static void WrapWarn(const FactArguments &args)     { if (This) This->FuncWarn(args);     }
    static void WrapWait(const FactArguments &args)     { if (This) This->FuncWait(args);     }
    static void WrapSend(const FactArguments &args)     { if (This) This->FuncSend(args);     }
    static void WrapSleep(const FactArguments &args)    { if (This) This->FuncSleep(args);    }
    static void WrapTimeout(const FactArguments &args)  { if (This) This->FuncTimeout(args);  }
    static void WrapThread(const FactArguments &args)   { if (This) This->FuncThread(args);   }
    static void WrapKill(const FactArguments &args)     { if (This) This->FuncKill(args);     }
    static void WrapExit(const FactArguments &args)     { if (This) This->FuncExit(args);     }
    static void WrapState(const FactArguments &args)    { if (This) This->FuncState(args);    }
    static void WrapNewState(const FactArguments &args) { if (This) This->FuncNewState(args); }
    static void WrapSetState(const FactArguments &args) { if (This) This->FuncSetState(args); }
    static void WrapGetState(const FactArguments &args) { if (This) This->FuncGetState(args); }
    static void WrapGetStates(const FactArguments &args){ if (This) This->FuncGetStates(args);}
    static void WrapGetDescription(const FactArguments &args){ if (This) This->FuncGetDescription(args); }
    static void WrapGetServices(const FactArguments &args){ if (This) This->FuncGetServices(args); }
    static void WrapSetInterrupt(const FactArguments &args){ if (This)This->FuncSetInterrupt(args); }
    static void WrapTriggerInterrupt(const FactArguments &args){ if (This) This->FuncTriggerInterrupt(args); }
    //static v8::Handle<v8::Value> WrapOpen(const FactArguments &args)     { if (This) return This->FuncOpen(args);     else return Undefined(v8::Isolate::GetCurrent()); }
    static void WrapSubscription(const FactArguments &args){ if (This) This->FuncSubscription(args); }
    static void WrapGetData(const FactArguments &args)  { if (This) This->FuncGetData(args);  }
    static void WrapClose(const FactArguments &args)    { if (This) This->FuncClose(args); }
    static void WrapQuery(const FactArguments &args)    { if (This) This->FuncQuery(args); }
    static void WrapDatabase(const FactArguments &args) { if (This) This->FuncDatabase(args); }
    static void WrapDbQuery(const FactArguments &args)  { if (This) This->FuncDbQuery(args); }
    static void WrapDbClose(const FactArguments &args)  { if (This) This->FuncDbClose(args); }
    static void WrapOnChangeSet(v8::Local<v8::Name> name, v8::Local<v8::Value> value, const v8::PropertyCallbackInfo<v8::Value> &info)
    {
        if (This) This->OnChangeSet(name, value, info);
    }

    static void OnChangeGet(v8::Local<v8::Name> /*property*/, const v8::PropertyCallbackInfo<v8::Value> &/*info*/)
    {
        //return Undefined(v8::Isolate::GetCurrent());
    }

    static v8::Handle<v8::Value> Convert(char type, const char* &ptr);
    v8::Handle<v8::Value> ConvertEvent(const EventImp *evt, uint64_t, const char *str);
#endif

    v8::Handle<v8::Value> HandleInterruptImp(std::string, uint64_t);

public:
    InterpreterV8();
    virtual ~InterpreterV8()
    {
        This = 0;

#ifdef HAVE_V8
        //v8::Locker locker;
        v8::V8::Dispose();
#endif
    }

    std::vector<std::string> JsGetCommandList(const char *, int) const;

    virtual void  JsLoad(const std::string & = "");
    virtual void  JsStart(const std::string &) { }
    virtual void  JsEnd(const std::string & = "");
    virtual void  JsPrint(const std::string & = "") { }
    virtual void  JsAlarm(const std::string & = "") { }
    virtual void  JsOut(const std::string &) { }
    virtual void  JsWarn(const std::string &) { }
    virtual void  JsResult(const std::string &) { }
    virtual void  JsException(const std::string &) { }
    virtual bool  JsSend(const std::string &) { return true; }
    //virtual void  JsSleep(uint32_t) { }
    //virtual int   JsWait(const std::string &, int32_t, uint32_t) { return -1; };
    virtual State JsState(const std::string &) { return State(); };
    virtual void *JsSubscribe(const std::string &) { return 0; };
    virtual bool  JsUnsubscribe(const std::string &) { return false; };

    virtual bool  JsNewState(int, const std::string&, const std::string&) { return false; }
    virtual bool  JsSetState(int) { return false; }
    virtual bool  JsHasState(int) const { return false; }
    virtual bool  JsHasState(const std::string &) const { return false; }
    virtual int   JsGetState(const std::string &) const { return -2; }
    virtual State JsGetCurrentState() const { return State(); }
    virtual std::vector<State> JsGetStates(const std::string &) { return std::vector<State>(); }
    virtual std::set<Service> JsGetServices() { return std::set<Service>(); }
    virtual std::vector<Description> JsGetDescription(const std::string &) { return std::vector<Description>(); }

    virtual std::vector<Description> JsDescription(const std::string &) { return std::vector<Description>(); };
    virtual std::pair<uint64_t, EventImp *> JsGetEvent(const std::string &) { return std::make_pair(0, (EventImp*)0); };

    int JsHandleInterrupt(const EventImp &);
    void JsHandleEvent(const EventImp &, uint64_t, const std::string &);
    void JsHandleState(const std::string &, const State &);

    void AddFormatToGlobal();

    bool JsRun(const std::string &, const std::map<std::string,std::string> & = std::map<std::string,std::string>());
    static void JsStop();
};

#ifndef HAVE_V8
inline bool InterpreterV8::JsRun(const std::string &, const std::map<std::string,std::string> &) { return false; }
inline void InterpreterV8::JsStop() { }
#endif

#endif
