#ifndef FACT_EventImp
#define FACT_EventImp

#include <string>
#include <vector>

#include <boost/function.hpp>

#include "Time.h"

class EventImp
{
    int              fTargetState;   /// Target state of an event
    std::vector<int> fAllowedStates; /// List of states in which this event is allowed

    /// http://www.boost.org/doc/libs/1_45_0/libs/bind/bind.html
    boost::function<int(const EventImp &)> fFunction;

public:
    /// Constructor. Stores the target state given.
    EventImp(int target=-1) : fTargetState(target) { }
    /// Copy constructor
    EventImp(const EventImp &cmd);

    // Function handling
    void AssignFunction(boost::function<int(const EventImp &)> func) { fFunction = func; }
    bool HasFunc() const { return !fFunction.empty(); }
    int ExecFunc() const { return HasFunc() ? fFunction(*this) : -1; }

    // Print contents
    virtual void Print(std::ostream &out) const;
    virtual void Print() const;

    // Handling of the states
    void AddAllowedState(int state);
    void AddAllowedStates(const char *states);

    bool IsStateAllowed(int state) const;

    int  GetTargetState() const { return fTargetState; }

    // virtual function to return the data as stored in the derived classes
    virtual std::string GetName() const   { return ""; }
    virtual std::string GetFormat() const { return ""; }

    virtual const void *GetData() const { return 0; }
    virtual int         GetSize() const { return 0; }

    virtual Time GetTime() const { return Time::None; }

    // Getter for all the data contained (name, format, data and time)
    short          GetShort()  const { return *reinterpret_cast<const short*>(GetData()); }
    unsigned short GetUShort() const { return *reinterpret_cast<const unsigned short*>(GetData()); }
    int            GetInt() const    { return *reinterpret_cast<const int*>(GetData()); }
    unsigned int   GetUInt() const   { return *reinterpret_cast<const unsigned int*>(GetData()); }
    float          GetFloat() const  { return *reinterpret_cast<const float*>(GetData()); }
    double         GetDouble() const { return *reinterpret_cast<const double*>(GetData()); }
    const char    *GetText() const   { return  reinterpret_cast<const char*>(GetData()); }

};

#endif
