#ifndef FACT_Configuration
#define FACT_Configuration

#include <boost/program_options.hpp>
/*
\attention
\callgraph
\callergraph
\category
\dot
\remark
\see
\throws
\todo
\warning

\e  italic
\b  bold
\c  typewriter

\code
\endcode

\f$
\f[ \f]
*/

namespace po = boost::program_options;

class Configuration
{
private:
    /// Convienience enum to access the fOption* data memebers more verbosely.
    enum
    {
        kHidden  = 0, ///< Index for hidden options (not shown in PrintParsed)
        kVisible = 1  ///< Index for options visible in PrintParsed
    };

    const std::string fName; /// argv[0]

    po::options_description fOptionsCommandline[2]; /// Description of the command-line options
    po::options_description fOptionsConfigfile[2];  /// Description of the options in the configuration file
    po::options_description fOptionsDatabase[2];    /// Description of options from the database
    po::options_description fOptionsEnvironment[2]; /// Description of options from the environment

    po::positional_options_description fArgumentPositions; /// Description of positional command-line options (arguments)

    std::vector<std::string> fUnknownCommandline;   /// Storage container for unrecognized commandline options
    std::vector<std::string> fUnknownConfigfile;    /// Storage container for unrecognized options from configuration files
    std::vector<std::string> fUnknownEnvironment;   /// Storage container for unrecognized options from the environment
    std::vector<std::string> fUnknownDatabase;      /// Storage container for unrecognized options retrieved from the database

    std::string fPriorityFile;  /// File name of the priority configuration file (overwrites option from the databse)
    std::string fDefaultFile;   /// File name of the default configuration file (usually {program}.rc)
    std::string fDatabase;      /// URL for database connection (see Configuration::parse_database)

    po::variables_map fVariables;  /// Variables as compiled by the Parse-function, which will be passed to the program

    /// A default mapper for environment variables skipping all of them
    static const std::string NameMapper(const std::string &) { return ""; }

    /// Pointer to the mapper function for environment variables
    const std::string (*fNameMapper)(const std::string&);

    /// Helper function which return the max of the two arguments in the first argument
    static void Max(int &val, const int &comp)
    {
        if (comp>val)
            val=comp;
    }

    /// Print all options from a list of already parsed options
    void PrintParsed(const po::parsed_options &parsed) const;
    /// Print a list of all unkown options within the given vector
    void PrintUnknown(std::vector<std::string> &vec, int steps=1);

public:
    Configuration(const std::string &prgname="");

    /// Retrieve data from a database and return them as options
    static po::basic_parsed_options<char>
        parse_database(const std::string &database, const po::options_description& desc, bool allow_unregistered=false);

    // Setup
    void AddOptionsCommandline(const po::options_description &cl, bool visible=true);
    void AddOptionsConfigfile(const po::options_description &cf, bool visible=true);
    void AddOptionsEnvironment(const po::options_description &env, bool visible=true);
    void AddOptionsDatabase(const po::options_description &db, bool visible=true);

    void SetArgumentPositions(const po::positional_options_description &desc);

    void SetNameMapper(const std::string (*mapper)(const std::string&));

    // Output
    void PrintOptions();
    void PrintUnknown();

    // Process command line arguments
    const po::variables_map &Parse(int argc, char **argv);

    bool HasHelp() const
    {
        return Has("help") || Has("help-config") || Has("help-env") || Has("help-database");
    }

    bool HasPrint() const
    {
        return Has("print-all") || Has("print") || Has("print-default") ||
            Has("print-database") || Has("print-config") ||
            Has("print-environment") || Has("print-unkown") ||
            Has("print-options");
    }

    // Simplified access to the parsed options
    template<class T>
       T Get(const char *var) const { return Has(var) ? fVariables[var].as<T>() : T(); }
    bool Has(const char *var) const { return fVariables.count(var)>0; }

    const std::string &GetName() const { return fName; }
};

template<class T>
inline po::typed_value<T> *var(T *ptr=0)
{ return po::value<T>(ptr); }

template<class T>
inline po::typed_value<T> *var(const T &val, T *ptr=0)
{ return po::value<T>(ptr)->default_value(val); }

template<class T>
inline po::typed_value<std::vector<T>> *vars()
{ return po::value<std::vector<T>>(); }

inline po::typed_value<bool> *po_switch()
{ return po::bool_switch(); }

#endif
