#include "RemoteControl.h"

StateClient::StateClient(const std::string &name, MessageImp &imp) :
    MessageDimRX(name, imp), fState(-2),
    fInfoState((name + "/STATE").c_str(), const_cast<char*>(""), this)
{
}

void StateClient::infoHandler()
{
    DimInfo *curr = getInfo(); // get current DimInfo address
    if (!curr)
        return;

    if (curr==&fInfoState)
    {
        const char *txt = fInfoState.getString();

        fState     = strlen(txt)==0 ? -2 : fInfoState.getQuality();
        fStateTime = Time(fInfoState.getTimestamp(),
                          fInfoState.getTimestampMillisecs());
        fStateMsg  = txt;

        if (fState!=-2)
        {
            stringstream msg;
            msg << fInfoState.getName() << ": Changed state to " << fState << " '" << txt << "' received.";
            fMsg.Write(fStateTime, msg.str().c_str(), MessageImp::kInfo);
        }
        return;
    }

    MessageDimRX::infoHandler();
}

// ==========================================================================

void RemoteControlImp::infoHandler()
{
    const vector<string> list = fServiceList.GetServerList();

    // Remover StateClients which are not available anymore
    for (ClientList::iterator l=fClientList.begin(); l!=fClientList.end(); l++)
    {
        const string &server = l->first;

        // HasServer doesn't work here, I don't understand why.
        if (find(list.begin(), list.end(), server)!=list.end())
            continue;

        delete l->second;
        fClientList.erase(l);
    }


    // Add StateClients which are new
    for (vector<string>::const_iterator l=list.begin(); l!=list.end(); l++)
    {
        if (*l=="DIS_DNS")
            continue;

        const ClientList::const_iterator p = fClientList.find(*l);
        if (p!=fClientList.end())
            continue;

        fClientList[*l] = new StateClient(*l, *this);
    }
}

bool RemoteControlImp::ProcessCommand(const std::string &str)
{
    if (fCurrentServer.empty())
    {
        if(fServiceList.HasServer(str))
        {
            fCurrentServer = str;
            return true;
        }
        lout << kRed << "Unkown server '" << str << "'" << endl;
        return false;
    }

    if (!fCurrentServer.empty() && str=="..")
    {
        fCurrentServer = "";
        return false;
    }

    return !fServiceList.SendDimCommand(lout, fCurrentServer, str);
}

// ==========================================================================
#include <boost/regex.hpp>

#include "tools.h"

string RemoteConsole::GetUpdatePrompt() const
{
    // If we are continously flushing the buffer omit the buffer size
    // If we are buffering show the buffer size
    const string beg = "\n" + GetLinePrompt();

    // If we have not cd'ed to a server show only the line start
    if (fCurrentServer.empty())
        return beg + "> ";

    // Check if we have cd'ed to a valid server
    const ClientList::const_iterator l = fClientList.find(fCurrentServer);
    if (l==fClientList.end())
        return beg + "> ";//Form("\n[%d] \033[34m\033[1m%s\033[0m> ", GetLine(), fCurrentServer.c_str());

    // The server is valid, check for the state from the last status message
    const string msg = l->second->GetMsg();

    static const boost::regex expr("(.+)\\[[0-9]+\\].*");

    // The server
    const string serv = Form("\033[34m\033[1m%s\033[0m", fCurrentServer.c_str());

    boost::smatch what;
    const int rc = boost::regex_match(msg, what, expr, boost::match_extra);

    // If no match found or something wrong found just output the server
    if (!rc || what.size()!=2)
        return beg + " " + serv + "> ";//Form("[%d-%s] ", GetLine(), msg.c_str());

    // If everything found add the state to the server
    const string state = what[1];
    return beg + " " + serv + Form(":\033[32m\033[1m%s\033[0m> ", state.c_str());
}

string RemoteShell::GetUpdatePrompt() const
{
    // If we are continously flushing the buffer omit the buffer size
    // If we are buffering show the buffer size
    const string beg = GetLinePrompt();

    // If we have not cd'ed to a server show only the line start
    if (fCurrentServer.empty())
        return beg + "> ";

    // Check if we have cd'ed to a valid server
    const ClientList::const_iterator l = fClientList.find(fCurrentServer);
    if (l==fClientList.end())
        return beg + "> ";//Form("\n[%d] \033[34m\033[1m%s\033[0m> ", GetLine(), fCurrentServer.c_str());

    // The server is valid, check for the state from the last status message
    const string msg = l->second->GetMsg();

    static const boost::regex expr("(.+)\\[[0-9]+\\].*");

    boost::smatch what;
    const int rc = boost::regex_match(msg, what, expr, boost::match_extra);

    // If no match found or something wrong found just output the server
    if (!rc || what.size()!=2)
        return beg + " " + fCurrentServer + "> ";//Form("[%d-%s] ", GetLine(), msg.c_str());

    // If everything found add the state to the server
    const string state = what[1];
    return beg + " " + fCurrentServer + ":" + state + "> ";
}
