// **************************************************************************
/** @class Console

@brief This is an extension to the Readline class provding buffered output

This in an extension to the Readline class. It's purpose is to keep a
buffered output stream and flush the stream either between readline entries
(non continous mode) or continously, keeping the readline prompt as
intact as possible.

 */
// **************************************************************************
#include "Console.h"

#include <iostream>

#include "tools.h"

using namespace std;

// --------------------------------------------------------------------------
//
//! Instantiate a console stream. It will create a WindowLog object
//! and immediatel switch off its output to the console. The default more
//! is non-continous.
//!
//! @param name
//!     The name of the program passed to the Readline constructor
//!
Console::Console(const char *name) : Readline(name), fContinous(false)
{
    fLogO.SetNullOutput();
    fLogI.SetNullOutput();
    fLogO.SetBacklog(true);
    fLogI.SetBacklog(true);
}

// --------------------------------------------------------------------------
//
//! Flush the contents of the buffer before it is destroyed.
//
Console::~Console()
{
    // flush buffer to display before it is destroyed in its destructor
    fLogO.Display();
    fLogI.Display();
}

// --------------------------------------------------------------------------
//
//! Before readline starts flush the buffer to display all stuff which was
//! buffered since the last readline call returned.
//
void Console::Startup()
{
    // Call readline's startup (just in case, it is empty)
    Readline::Startup();

    // Flush the buffer and remove the flushed contents from the buffer
    fLogO.Display(true);
    fLogI.Display(true);
}

// --------------------------------------------------------------------------
//
//! Flush the buffer if we are in continous mode, and call Readline's
//! EventHook to update the prompt.
//
void Console::EventHook()
{
    // If the output is continous and we are going to output something
    // first jump back to the beginning of the line (well, that
    // doesn't work well if the input line is already two lines)
    // and then flush the buffer.
    if (fContinous && fLogO.GetSizeBacklog()>0)
    {
        std::cout << "\r";
        fLogO.Display(true);
    }

    // Call Readline's EventHook to update the prompt
    Readline::EventHook();
}

string Console::GetLinePrompt() const
{
    const string siz = fLogO.GetSizeStr();
    return fContinous ?
        Form("[%d]", GetLine()) : Form("[%d:%s]", GetLine(), siz.c_str());
}

/*
// --------------------------------------------------------------------------
//
//! Flush the buffer before it is filled with contents produced by the
//! command processing. This keeps things as seperated as possible,
//! although there is no gurantee.
//
void Console::Shutdown(const char *)
{
    // Flush the buffer (after readline() returned, before processing commands)
    fLog.Display(true);
    // std::cout << std::endl;
}
*/

// --------------------------------------------------------------------------
//
//! Before Readline::Run() is called the buffer is flushed as well as
//! after the Run() loop has exited.
//! command processing. This keeps things as seperated as possible,
//! although there is no gurantee.
//
void Console::Run(const char *)
{
    // Flush the buffer before we stat out readline loop
    fLogI.Display(true);
    fLogO.Display(true);

    // Now run readlines main loop
    Readline::Run();

    // flush buffer to display
    fLogO.Display(true);
    fLogI.Display(true);
}
