// **************************************************************************
/** @struct Description

@brief A struct which stores a name, a unit and a comment

To have proper descriptions of commands and services in the network
(and later proper informations in the FITS files) this struct provides
a simple storage for a name, a unit and a comment.

Assume you want to write a descriptive string for a command with three arguments.
It could look like this:

   COMMAND=Description|int[addr]:Address range (from - to)|val[byte]:Value to be set

Description is a general description of the command or service itself,
int and val are the names of the arguments (e.g. names of FITS columns),
addr and byte have the meaning of a unit (e.g. unit of FITS column)
and the text after the colon is a description of the arguments
(e.g. comment of a FITS column). The description must not contain a
line-break character \n.

Such a string can then be converted with SplitDescription into a vector
of Description objects, each containing the name, the unit and a
comment indivdually. The first object will contain COMMAND as name and
Description as comment. The unit will remain empty.

You can omit either the name, the unit or the comment or any combination of them.
The descriptions of the individual format strings are separated by a vertical line.

*/
// **************************************************************************
#include "Description.h"

#include <sstream>

using namespace std;

// --------------------------------------------------------------------------
//
//! This function removed whitshapces on both sides of a string.
//!
//! @param s
//!     string to be cleaned
//!
//! @returns
//!     string cleaned from whitespaces
//
string Description::Trim(string s)
{
    // Trim Both leading and trailing spaces
    const size_t start = s.find_first_not_of(' '); // Find the first character position after excluding leading blank spaces
    const size_t end   = s.find_last_not_of(' ');  // Find the first character position from reverse af

    // if all spaces or empty return an empty string
    if (string::npos==start || string::npos==end)
        return string();

    return s.substr(start, end-start+1);
}

// --------------------------------------------------------------------------
//
//! This functio0n breaks down a descriptive string into its components.
//! For details see class reference.
//!
//! @param buffer
//!     string which should be broekn into pieces
//!
//! @returns
//!     A vector<Description> containing all the descriptions found.
//!     The first entry contains the discription name and comment.
//
vector<Description> Description::SplitDescription(const string &buffer)
{
    const size_t p0 = buffer.find_first_of('=');

    const string svc  = buffer.substr(0, p0);
    const string desc = buffer.substr(p0+1);

    const size_t p = desc.find_first_of('|');

    // Extract a general description
    const string d = Description::Trim(desc.substr(0, p));

    vector<Description> vec;
    vec.push_back(Description(svc, "", d));

    if (p==string::npos)
        return vec;

    string buf;
    stringstream stream(desc.substr(p+1));
    while (getline(stream, buf, '|'))
    {
        if (buf.empty())
            continue;

        const size_t p1 = buf.find_first_of(':');

        const string comment = p1==string::npos ? "" : buf.substr(p1+1);
        if (p1!=string::npos)
            buf.erase(p1);

        const size_t p2 = buf.find_last_of('[');
        const size_t p3 = buf.find_last_of(']');

        const bool hasunit = p2<p3 && p2!=string::npos;

        const string unit = hasunit ? buf.substr(p2+1, p3-p2-1) : "";
        const string name = hasunit ? buf.substr(0, p2) : buf;

        vec.push_back(Description(name, unit, comment));
    }

    return vec;
}
