// **************************************************************************
/** @class DimDescriptionService

@brief A DimService which broadcasts descriptions for services and commands

The DimDescriptionService creates a service with the name of the server like
SERVER/SERVICE_DESC. This is meant in addition to the SERVICE_LIST service
of each node to contain a description of the service and its arguments.

Assume you have created a service (or command) with the format I:2;F:1
a valid description string would look like

   Description|int[addr]:Address range (from - to)|val[byte]:Value to be set

Description is a general description of the command or service itself,
int and val are the names of the arguments (e.g. names of FITS columns),
addr and byte have the meaning of a unit (e.g. unit of FITS column)
and the text after the colon is a description of the arguments
(e.g. comment of a FITS column). The description must not contain a
line-break character \n.

You can omit either the name, the unit or the comment or any combination of them.
The descriptions of the individual format strings are separated by a vertical line.

The description should contain as many descriptions as format chunks, e.g.

I:1          should contain one description chunks
I:1;F:1      should contain two description chunks
I:2;F:1      should contain two description chunks
I:2;I:1;F:1  should contain three description chunks

*/
// **************************************************************************
#include "DimDescriptionService.h"

#include <stdexcept>

#include "dis.hxx"

using namespace std;

DimService *DimDescriptionService::fService = 0;
int         DimDescriptionService::fCount   = 0;
std::string DimDescriptionService::fData    = "";

// --------------------------------------------------------------------------
//
//! When the constructor is first called, a service with the name
//! SERVER/SERVICE_DESC is created. The server name SERVER is retrieved
//! from DimServer::itsName. If DimServer::itsName is empty, the
//! server name is extracted from the given name as the part before the
//! first '/'. A string "name=format\n" is added to fData and stored
//! in fDescription.
//!
//! A counter fCount for the number of instantiations is increased.
//!
//! @param name
//!     The name of the service or command to be described, e.g. SERVER/COMMAND
//!
//! @param desc
//!     A description string. For details see class reference
//!
//! @throws
//!     If a server name couldn't be reliably determined a runtime_error
//!     exception is thrown.
//
DimDescriptionService::DimDescriptionService(const std::string &name, const std::string &desc)
{
    string service = DimServer::itsName;
    if (service.empty())
    {
        const size_t p = name.find_first_of('/');
        if (p==string::npos)
            throw runtime_error("Could not determine server name");

        service = name.substr(0, p);
    }

    service += "/SERVICE_DESC";
    if (!fService)
        fService = new DimService(service.c_str(), const_cast<char*>(""));

    fCount++;

    fDescription = name + '=' + desc;

    if (fData.find(fDescription+'\n')!=std::string::npos)
        return;

    fData += fDescription + '\n';

    fService->setData(const_cast<char*>(fData.c_str()));
    fService->updateService();
}


// --------------------------------------------------------------------------
//
//! If fDescription is found in fData it is removed from fData.
//! The counter fCount is decreased and fService deleted if the counter
//! reached 0.
//
DimDescriptionService::~DimDescriptionService()
{
    const size_t pos = fData.find(fDescription+'\n');
    if (pos!=std::string::npos)
        fData.replace(pos, fDescription.size()+1, "");

    if (--fCount>0)
        return;

    delete fService;
    fService=0;
}
