#ifndef FACT_RemoteControl
#define FACT_RemoteControl

// **************************************************************************
/** @class RemoteControlImp

@brief This implements the basic functions of a remote control via dim

Through a ServiceList object this object subscribes to all available
SERVICE_LISTs  in the dim network. This allows to keep an up-to-date
list of all servers and services. Its ProcessCommand member function
allows to emit commands according to the services found in the network.
Its infoHandler() is called as an update notifier from the ClientList
object.

**/
// **************************************************************************
#include "DimNetwork.h"

class RemoteControlImp : public DimNetwork
{
protected:
    std::ostream &lout;          /// Output stream for local synchrounous output

    std::string fCurrentServer;  /// The server to which we currently cd'ed

protected:
    // Redirect asynchronous output to the output window
    RemoteControlImp(std::ostream &out, std::ostream &in) :
        DimNetwork(out), lout(in)
    {
    }
    bool ProcessCommand(const std::string &str);
};



// **************************************************************************
/** @class RemoteControl

@brief Implements a remote control based on a Readline class for the dim network

This template implements all functions which overwrite any function from the
Readline class. Since several derivatives of the Readline class implement
different kind of Readline access, this class can be derived by any of them
due to its template argument. However, the normal case will be
deriving it from either Console or Shell.

@tparam T
   The base class for RemoteControl. Either Readlien or a class
    deriving from it. This is usually either Console or Shell.

**/
// **************************************************************************
#include "WindowLog.h"
#include "ReadlineColor.h"
#include "tools.h"

template <class T>
class RemoteControl : public T, public RemoteControlImp
{
private:
    MessageImp *fImp;

    int32_t fLabel;

    int Write(const Time &time, const std::string &txt, int qos=kMessage)
    {
        return fImp ? fImp->Write(time, txt, qos) : MessageImp::Write(time, txt, qos);
    }

    static void append(std::string &str)
    {
        str.append("/");
    }
    static void chop(std::string &str)
    {
        const size_t p = str.find_first_of('/');
        if (p!=string::npos)
            str = str.substr(p+1);
    }

    // This funtion defines which generator should be called.
    // If it returns 0 the standard reaqdline generator are called.
    // Otherwise set the right generator with rl_completion_matches.
    char **Completion(const char *text, int start, int)
    {
        // Get the whole buffer before the tab-position
        const string b = string(T::GetBuffer());
        const string s = b.substr(0, start);
        const string l = Tools::Trim(s.c_str());
        if (l.empty())
        {
            if (fCurrentServer.empty())
            {
                const size_t p1 = b.find_first_of(' ');
                const size_t p2 = b.find_first_of('/');

                if (p1==string::npos && p2!=string::npos)
                    return T::Complete(GetCommandList(), text);

                std::vector<std::string> v = GetServerList();
                for_each(v.begin(), v.end(), RemoteControl::append);
                return T::Complete(v, text);
            }
            else
            {
                std::vector<std::string> v = GetCommandList(fCurrentServer);
                for_each(v.begin(), v.end(), RemoteControl::chop);
                return T::Complete(v, text);
            }
        }
        return T::Complete(GetCommandList(l), text);
    }

    void infoHandler()
    {
        RemoteControlImp::infoHandler();
        if (!fCurrentServer.empty() && !HasServer(fCurrentServer))
        {
            fCurrentServer = "";
            T::UpdatePrompt();
        }
    }

public:
    // Redirect asynchronous output to the output window
    RemoteControl(const char *name) : T(name),
        RemoteControlImp(T::GetStreamOut(), T::GetStreamIn()), fImp(0), fLabel(-1)
    {
    }

    bool PrintGeneralHelp()
    {
        T::PrintGeneralHelp();
        lout << " " << kUnderline << "Specific commands:\n";
        lout << kBold << "   h,help <arg> " << kReset << "List help text for given server or command.\n";
//        lout << kBold << "   s,servers    " << kReset << "List all servers which are connected." << endl;
        lout << kBold << "   svc,services " << kReset << "List all services in the network.\n";
        lout << kBold << "   st,states    " << kReset << "List all states in the network.\n";
        lout << kBold << "   # <text>     " << kReset << "Echo <text> to the output stream\n";
        lout << kBold << "   .s           " << kReset << "Wait for the state-machine to change to the given state.\n";
        lout <<          "                "              "     .s <server> [<state> [<timeout> [<label>]]]\n";
        lout <<          "                "              "<server>  The server for which state to wait (e.g. FTM_CONTROL)\n";
        lout <<          "                "              "<state>   The state id (see 'states') for which to wait (e.g. 3)\n";
        lout <<          "                "              "<imeout>  A timeout in millisenconds how long to wait (e.g. 500)\n";
        lout <<          "                "              "<label>   A label until which everything is skipped in case of timeout\n";
        lout << endl;
        return true;
    }

    bool PrintCommands()
    {
        lout << endl << kBold << "List of commands:" << endl;
        PrintDescription(lout, true);
        return true;
    }

    // returns whether a command should be put into the history
    bool Process(const std::string &str)
    {
        if (str.substr(0, 2)=="h " || str.substr(0, 5)=="help ")
        {
            const size_t p1 = str.find_first_of(' ');
            const string svc = str.substr(p1+1);

            const size_t p3 = svc.find_first_of('/');
            const string s = svc.substr(0, p3);
            const string c = p3==string::npos?"":svc.substr(p3+1);

            lout << endl;
            if (!fCurrentServer.empty())
            {
                if (PrintDescription(lout, true, fCurrentServer, svc)==0)
                    lout << "   " << svc << ": <not found>" << endl;
            }
            else
            {
                if (PrintDescription(lout, true, s, c)==0)
                    lout << "   <no matches found>" <<endl;
            }

            return true;
        }

        if (str.substr(0, 3)==".s ")
        {
            istringstream in(str.substr(3));

            int state=-100, ms=0;
            string server;

            in >> server >> state >> ms;
            if (state==-100)
            {
                lout << kRed << "Couldn't parse state id in '" << str.substr(3) << "'" << endl;
                return true;
            }

            const ClientList::const_iterator l = fClientList.find(server);
            if (l==fClientList.end())
            {
                lout << kRed << "Server '" << server << "' not found." << endl;
                return true;
            }

            const Time timeout = ms<=0 ? Time(Time::none) : Time()+boost::posix_time::millisec(ms);

            T::GetStreamOut().Display(true);
            T::GetStreamOut().SetBacklog(false);
            T::GetStreamOut().SetNullOutput(false);
            while (l->second->GetState()!=state && timeout>Time())
                usleep(1);
            T::GetStreamOut().SetNullOutput(true);
            T::GetStreamOut().SetBacklog(true);

            if (l->second->GetState()!=state)
            {
                int label = -1;
                in >> label;
                T::SetLabel(label);
            }

            return true;
        }

        if (str[0]=='#')
        {
            //lout << Tools::Trim(str.substr(1)) << endl;
            fImp->Comment(Tools::Trim(str.substr(1)));
            return true;
        }

        if (ReadlineColor::Process(lout, str))
            return true;

        if (T::Process(str))
            return true;

        if (str=="services" || str=="svc")
        {
            PrintDescription(lout, false);
            return true;
        }

        if (str=="states" || str=="st")
        {
            PrintStates(lout);
            return true;
        }

        return ProcessCommand(str);
    }

    void SetReceiver(MessageImp &imp) { fImp = &imp; }
};



// **************************************************************************
/** @class RemoteConsole

@brief Derives the RemoteControl from Control and adds a proper prompt

This is basically a RemoteControl, which derives through the template
argument from the Console class. It enhances the functionality of
the remote control with a proper updated prompt.

 */
// **************************************************************************
#include "Console.h"

class RemoteConsole : public RemoteControl<Console>
{
public:
    RemoteConsole(const char *name, bool continous=false) :
        RemoteControl<Console>(name)
    {
        SetContinous(continous);
    }
    string GetUpdatePrompt() const;
};

// **************************************************************************
/** @class RemoteShell

@brief Derives the RemoteControl from Shell and adds colored prompt

This is basically a RemoteControl, which derives through the template
argument from the Shell class. It enhances the functionality of
the local control with a proper updated prompt.

 */
// **************************************************************************
#include "Shell.h"

class RemoteShell : public RemoteControl<Shell>
{
public:
    RemoteShell(const char *name, bool = false) :
        RemoteControl<Shell>(name)
    {
    }
    string GetUpdatePrompt() const;
};

#endif
