#ifdef HAVE_LIBNOVA
#include <libnova/solar.h>
#include <libnova/lunar.h>
#include <libnova/rise_set.h>
#endif

#include "Dim.h"
#include "Event.h"
#include "Shell.h"
#include "StateMachineDim.h"
#include "Connection.h"
#include "Configuration.h"
#include "Console.h"
#include "Converter.h"
#include "DimServiceInfoList.h"
#include "PixelMap.h"

#include "tools.h"
#include "DimData.h"

#include "LocalControl.h"

#include "HeadersFAD.h"
#include "HeadersBIAS.h"
#include "HeadersFTM.h"

using namespace std;

// ------------------------------------------------------------------------

#include "DimDescriptionService.h"

// ------------------------------------------------------------------------

const static string kHtmlWhite  = "#ffffff";
const static string kHtmlYellow = "#fffff0";
const static string kHtmlRed    = "#fff8f0";
const static string kHtmlGreen  = "#f0fff0";
const static string kHtmlBlue   = "#f0f0ff";

// ========================================================================
// ========================================================================
// ========================================================================

class Astro
{
public:
    Time time;

    Time fSunRiseDayTime;
    Time fSunRiseCivil;
    Time fSunRiseAstronomical;
    Time fSunRiseDarkTime;

    Time fSunSetDayTime;
    Time fSunSetCivil;
    Time fSunSetAstronomical;
    Time fSunSetDarkTime;

    int state;
    string description;
    string color;

    bool isday;

public:
    Astro(double lon, double lat, const Time &t=Time()) :  time(t)
    {
#ifdef HAVE_LIBNOVA
        ln_lnlat_posn observer;
        observer.lng = lon;
        observer.lat = lat;

        // get Julian day from local time
        const double JD = time.JD();

        ln_rst_time sun_day;
        ln_rst_time sun_civil;
        ln_rst_time sun_astronomical;
        ln_rst_time sun_dark;

        // Warning: return code of 1 means circumpolar and is not checked!
        //ln_get_lunar_rst        (JD-0.5, &observer,      &moon);
        ln_get_solar_rst        (JD-0.5, &observer,      &sun_day);
        ln_get_solar_rst_horizon(JD-0.5, &observer, - 6, &sun_civil);
        ln_get_solar_rst_horizon(JD-0.5, &observer, -12, &sun_astronomical);
        ln_get_solar_rst_horizon(JD-0.5, &observer, -18, &sun_dark);

        fSunSetDayTime       = Time(sun_day.set);
        fSunSetCivil         = Time(sun_civil.set);
        fSunSetAstronomical  = Time(sun_astronomical.set);
        fSunSetDarkTime      = Time(sun_dark.set);

        fSunRiseDayTime      = Time(sun_day.rise);
        fSunRiseCivil        = Time(sun_civil.rise);
        fSunRiseAstronomical = Time(sun_astronomical.rise);
        fSunRiseDarkTime     = Time(sun_dark.rise);

        const bool is_day   = JD>sun_day.rise;
        const bool is_night = JD>sun_dark.set;

        //ln_get_lunar_rst        (JD+0.5, &observer,      &moon);
        ln_get_solar_rst        (JD+0.5, &observer,      &sun_day);
        ln_get_solar_rst_horizon(JD+0.5, &observer, - 6, &sun_civil);
        ln_get_solar_rst_horizon(JD+0.5, &observer, -12, &sun_astronomical);
        ln_get_solar_rst_horizon(JD+0.5, &observer, -18, &sun_dark);

        if (is_day)
        {
            fSunRiseDayTime      = Time(sun_day.rise);
            fSunRiseCivil        = Time(sun_civil.rise);
            fSunRiseAstronomical = Time(sun_astronomical.rise);
            fSunRiseDarkTime     = Time(sun_dark.rise);
        }

        if (is_night)
        {
            fSunSetDayTime      = Time(sun_day.set);
            fSunSetCivil        = Time(sun_civil.set);
            fSunSetAstronomical = Time(sun_astronomical.set);
            fSunSetDarkTime     = Time(sun_dark.set);
        }

        // case 0: midnight to sun-rise | !is_day && !is_night | rise/set
        // case 1: sun-rise to sun-set  |  is_day && !is_night | set /rise
        // case 2: sun-set  to midnight |  is_day &&  is_night | rise/set
        /*
        if (is_day^is_night)
        {
            cout << "SunSet:  " << fSunSetDayTime  << endl;
            cout << "SunRise: " << fSunRiseDayTime << endl;
        }
        else
        {
            cout << "SunRise: " << fSunRiseDayTime << endl;
            cout << "SunSet:  " << fSunSetDayTime  << endl;
        }*/

        isday = is_day^is_night;

        state = isday ? 4 : 0;
        if (time>fSunSetDayTime)       state++;
        if (time>fSunSetCivil)         state++;
        if (time>fSunSetAstronomical)  state++;
        if (time>fSunSetDarkTime)      state++;

        if (time>fSunRiseDarkTime)     state++;
        if (time>fSunRiseAstronomical) state++;
        if (time>fSunRiseCivil)        state++;
        if (time>fSunRiseDayTime)      state++;

        string name[] =
        {
            "dark time",
            "astron. twilight",
            "civil twilight",
            "sunrise",
            "day time",
            "sunset",
            "civil twilight",
            "astron. twilight",
            "dark time"
        };

        description = state[name];

        string arr;
        ostringstream out;
        if (isday)
        {
            out << fSunSetDarkTime-time;
            arr = "&darr;";
        }
        else
        {
            out << fSunRiseDayTime-time;
            arr = "&uarr;";
        }

        description += " ["+out.str().substr(0, 5)+arr+"]";

        switch (state)
        {
        case 0: case 1:  color = kHtmlGreen;   break;
        case 2: case 3:  color = kHtmlYellow;  break;
        case 4:          color = kHtmlRed;     break;
        case 5: case 6:  color = kHtmlYellow;  break;
        case 7: case 8:  color = kHtmlGreen;   break;
        }
#endif
    }
};

class Moon
{
public:
    double ra;
    double dec;

    double disk;

    bool visible;

    Time fMoonRise;
    Time fMoonTransit;
    Time fMoonSet;

    string description;
    string color;

    Time time;

    Moon(double lon, double lat, const Time &t=Time()) : time(t)
    {
#ifdef HAVE_LIBNOVA
        const double JD = time.JD();

        ln_lnlat_posn observer;
        observer.lng = lon;
        observer.lat = lat;

        ln_rst_time moon;
        ln_get_lunar_rst(JD-0.5, &observer, &moon);

        fMoonRise    = Time(moon.rise);
        fMoonTransit = Time(moon.transit);
        fMoonSet     = Time(moon.set);

        visible =
            (JD>moon.rise && JD<moon.set  && moon.rise<moon.set) ||
            ((JD<moon.set  || JD>moon.rise) && moon.rise>moon.set);

        const bool is_up      = JD>moon.rise;
        const bool is_sinking = JD>moon.transit;
        const bool is_dn      = JD>moon.set;

        ln_get_lunar_rst(JD+0.5, &observer, &moon);
        if (is_up)
            fMoonRise = Time(moon.rise);
        if (is_sinking)
            fMoonTransit = Time(moon.transit);
        if (is_dn)
            fMoonSet = Time(moon.set);

        ln_equ_posn pos;
        ln_get_lunar_equ_coords(JD, &pos);

        ra  = pos.ra/15;
        dec = pos.dec;

        disk = ln_get_lunar_disk(JD)*100;

        if (!visible || disk<25)
            color = kHtmlGreen;
        else
            color = disk>75 ? kHtmlRed : kHtmlYellow;

        string arr;
        ostringstream dt;
        if (fMoonSet<fMoonRise)
        {
            dt << fMoonSet-time;
            arr = "&darr;";
        }
        else
        {
            dt << fMoonRise-time;
            arr = "&uarr;";
        }

        ostringstream out;
        out << setprecision(2);
        out << (visible?"visible ":"") << (disk<0.1?0:disk) << "% [" << dt.str().substr(0,5) << arr << "]";

        description = out.str();
#endif
    }

    double Angle(double r, double d)
    {
        const double theta0 = M_PI/2-d*M_PI/180;
        const double phi0   = r*M_PI/12;

        const double theta1 = M_PI/2-dec*M_PI/180;
        const double phi1   = ra*M_PI/12;

        const double x0 = sin(theta0) * cos(phi0);
        const double y0 = sin(theta0) * sin(phi0);
        const double z0 = cos(theta0);

        const double x1 = sin(theta1) * cos(phi1);
        const double y1 = sin(theta1) * sin(phi1);
        const double z1 = cos(theta1);

        double arg = x0*x1 + y0*y1 + z0*z1;
        if(arg >  1.0) arg =  1.0;
        if(arg < -1.0) arg = -1.0;

        return acos(arg) * 180/M_PI;
    }
};

// ========================================================================
// ========================================================================
// ========================================================================

class DimState : public DimInfoHandler
{
protected:
    typedef function<void(const DimData &)> callback;

public:
    DimState(const string &n, const string s="STATE") : server(n),
        info(make_pair(Time(), -4)),
        dim((n+"/"+s).c_str(), (void*)NULL, 0, this) { }

    string server;
    pair<Time, int> info;
    string msg;

    DimStampedInfo dim;

    callback fCallback;

    void SetCallback(const callback &cb)
    {
        fCallback = cb;
    }

    void infoHandler()
    {
        DimInfo *curr = getInfo(); // get current DimInfo address
        if (!curr || curr != &dim)
            return;

        const bool disconnected = dim.getSize()==0;

        // Make sure getTimestamp is called _before_ getTimestampMillisecs
        const int tsec = dim.getTimestamp();
        const int tms  = dim.getTimestampMillisecs();

        info = make_pair(Time(tsec, tms*1000),
                         disconnected ? -4 : dim.getQuality());

        msg = disconnected ? "" : dim.getString();

        if (fCallback)
            fCallback(DimData(curr));
    }

    const Time &time() const { return info.first; }
    const int  &state() const { return info.second; }

    bool online() const { return info.second>-4; }

    const string &name() const { return server; }
};

class DimVersion : public DimState
{
public:
    DimVersion() : DimState("DIS_DNS", "VERSION_NUMBER") { }

    void infoHandler()
    {
        DimInfo *curr = getInfo(); // get current DimInfo address
        if (!curr || curr != &dim)
            return;

        DimState::infoHandler();

        info.second = dim.getSize()==4 ? dim.getInt() : 0;
    }

    string version() const
    {
        if (info.second==0)
            return "Offline";

        ostringstream out;
        out << "V" << info.second/100 << 'r' << info.second%100;
        return out.str();
    }
};

class DimControl : public DimState
{
    map<string, callback> fCallbacks;
public:
    DimControl() : DimState("DIM_CONTROL") { }

    void AddCallback(const string &script, const callback &cb)
    {
        fCallbacks[script] = cb;
    }

    void infoHandler()
    {
        DimInfo *curr = getInfo(); // get current DimInfo address
        if (!curr || curr != &dim)
            return;

        DimState::infoHandler();

        // Evaluate msg
        const size_t p0 = msg.find_first_of(':');
        if (p0==string::npos)
            return;

        const size_t p1 = msg.find_last_of('[');
        if (p1==string::npos)
            return;

        const size_t p2 = msg.find_first_of(':', p0+1);

        const size_t p3 = p2==string::npos || p2>p1 ? p1-1 : p2;

        const string file = msg.substr(p0+2, p3-p0-2);

        const auto func = fCallbacks.find(file);
        if (func==fCallbacks.end())
            return;

        // Call callback
        func->second(DimData(curr));
    }
};

struct DimSubscriptions
{
    DimServiceInfoList fNetwork;

    DimVersion fDNS;
    DimControl fControl;
    DimState   fMcp;
    DimState   fDataLogger;
    DimState   fDriveControl;
    DimState   fMagicWeather;
    DimState   fFeedback;
    DimState   fBiasControl;
    DimState   fFtmControl;
    DimState   fFadControl;
    DimState   fFscControl;
    DimState   fRateControl;
    DimState   fRateScan;
    DimState   fChatServer;

    DimStampedInfo fControlMessage;

    DimStampedInfo fMcpConfiguration;

    DimStampedInfo fDriveControlPointing;
    DimStampedInfo fDriveControlTracking;
    DimStampedInfo fDriveControlSource;

    DimStampedInfo fFscControlTemperature;
    DimStampedInfo fFscControlHumidity;

    DimStampedInfo fMagicWeatherData;

    DimStampedInfo fFeedbackDeviation;
    DimStampedInfo fFeedbackCalibration;

    DimStampedInfo fBiasControlVoltage;
    DimStampedInfo fBiasControlCurrent;

    DimStampedInfo fFadConnections;
    DimStampedInfo fFadEvents;

    DimStampedInfo fFtmControlTriggerRates;
    DimStampedInfo fFtmControlStaticData;
    DimStampedInfo fFtmControlFtuList;

    DimStampedInfo fRateScanData;

    DimStampedInfo fFadControlEventData;

    DimSubscriptions(DimInfoHandler *h) :
        fMcp                   ("MCP"),
        fDataLogger            ("DATA_LOGGER"),
        fDriveControl          ("DRIVE_CONTROL"),
        fMagicWeather          ("MAGIC_WEATHER"),
        fFeedback              ("FEEDBACK"),
        fBiasControl           ("BIAS_CONTROL"),
        fFtmControl            ("FTM_CONTROL"),
        fFadControl            ("FAD_CONTROL"),
        fFscControl            ("FSC_CONTROL"),
        fRateControl           ("RATE_CONTROL"),
        fRateScan              ("RATE_SCAN"),
        fChatServer            ("CHAT_SERVER"),
        //---
        fControlMessage        ("DIM_CONTROL/MESSAGE",             (void*)NULL, 0, h),
        //---
        fMcpConfiguration      ("MCP/CONFIGURATION",               (void*)NULL, 0, h),
        //---
        fDriveControlPointing  ("DRIVE_CONTROL/POINTING_POSITION", (void*)NULL, 0, h),
        fDriveControlTracking  ("DRIVE_CONTROL/TRACKING_POSITION", (void*)NULL, 0, h),
        fDriveControlSource    ("DRIVE_CONTROL/SOURCE_POSITION",   (void*)NULL, 0, h),
        //---
        fFscControlTemperature ("FSC_CONTROL/TEMPERATURE",         (void*)NULL, 0, h),
        fFscControlHumidity    ("FSC_CONTROL/HUMIDITY",            (void*)NULL, 0, h),
        //---
        fMagicWeatherData      ("MAGIC_WEATHER/DATA",              (void*)NULL, 0, h),
        //---
        fFeedbackDeviation     ("FEEDBACK/DEVIATION",              (void*)NULL, 0, h),
        fFeedbackCalibration   ("FEEDBACK/CALIBRATION",            (void*)NULL, 0, h),
        //---
        fBiasControlVoltage    ("BIAS_CONTROL/VOLTAGE",            (void*)NULL, 0, h),
        fBiasControlCurrent    ("BIAS_CONTROL/CURRENT",            (void*)NULL, 0, h),
        //---
        fFadConnections        ("FAD_CONTROL/CONNECTIONS",         (void*)NULL, 0, h),
        fFadEvents             ("FAD_CONTROL/EVENTS",              (void*)NULL, 0, h),
        //---
        fFtmControlTriggerRates("FTM_CONTROL/TRIGGER_RATES",       (void*)NULL, 0, h),
        fFtmControlStaticData  ("FTM_CONTROL/STATIC_DATA",         (void*)NULL, 0, h),
        fFtmControlFtuList     ("FTM_CONTROL/FTU_LIST",            (void*)NULL, 0, h),
        //---
        fRateScanData          ("RATE_SCAN/DATA",                  (void*)NULL, 0, h),
        //---
        fFadControlEventData   ("FAD_CONTROL/EVENT_DATA",          (void*)NULL, 0, h)

    {
    }

    const State GetState(const DimState &s) const
    {
        return fNetwork.GetState(s.name(), s.state());
    }
};

// ========================================================================
// ========================================================================
// ========================================================================


class StateMachineSmartFACT : public StateMachineDim, public DimInfoHandler
{
private:
    enum states_t
    {
        kStateDimNetworkNA = 1,
        kStateRunning,
    };

    // ------------------------- Internal variables -----------------------

    PixelMap fPixelMap;

    Time fLastUpdate;

    string fPath;

    // ----------------------------- Data storage -------------------------

    deque<string> fControlMessageHist;

    uint32_t fMcpConfigurationState;
     int64_t fMcpConfigurationMaxTime;
     int64_t fMcpConfigurationMaxEvents;
    string   fMcpConfigurationName;
    Time     fMcpConfigurationRunStart;

    enum weather_t { kWeatherBegin=0, kTemp = kWeatherBegin, kDew, kHum, kPress, kWind, kGusts, kDir, kWeatherEnd = kDir+1 };
    deque<float> fMagicWeatherHist[kWeatherEnd];

    vector<float> fFeedbackCalibration;

    float fFeedbackTempOffset;
    float fFeedbackUserOffset;

    vector<float> fBiasControlVoltageVec;

    float  fBiasControlVoltageMed;
    float  fBiasControlCurrentMed;
    float  fBiasControlCurrentMax;

    deque<float> fBiasControlCurrentHist;
    deque<float> fFscControlTemperatureHist;

    float fFscControlHumidityAvg;

    float  fDriveControlPointingZd;
    string fDriveControlPointingAz;
    string fDriveControlSourceName;

    deque<float> fDriveControlTrackingDevHist;

    int64_t fFadControlNumEvents;

    float  fFtmControlTriggerRateCam;
    deque<float> fFtmControlTriggerRateHist;

    vector<float> fFtmPatchThresholds;

    bool fFtmControlFtuOk;

    uint64_t  fRateScanDataId;
    uint8_t   fRateScanBoard;
    deque<float> fRateScanDataHist[41];

    // ------------- Initialize variables before the Dim stuff ------------

    DimSubscriptions *fDim;

    // -------------------------------------------------------------------


    // -------------------------------------------------------------------

    bool HandleService(DimInfo *curr, const DimInfo &service, void (StateMachineSmartFACT::*handle)(const DimData &))
    {
        if (curr!=&service)
            return false;

        (this->*handle)(DimData(curr));
        return true;
    }


    bool CheckDataSize(const DimData &d, const char *name, size_t size, bool min=false)
    {
        if ((!min && d.data.size()==size) || (min && d.data.size()>size))
            return true;

        ostringstream msg;
        msg << name << " - Received service has " << d.data.size() << " bytes, but expected ";
        if (min)
            msg << "more than ";
        msg << size << ".";
        Warn(msg);
        return false;
    }


    // -------------------------------------------------------------------

    template<class T>
        void WriteBinary(const DimData &d, const string &fname, const T &t, double scale, double offset=0)
    {
        vector<uint8_t> val(t.size(), 0);
        for (uint64_t i=0; i<t.size(); i++)
        {
            float range = nearbyint(128*(t[i]-offset)/scale); // [-2V; 2V]
            if (range>127)
                range=127;
            if (range<0)
                range=0;
            val[i] = (uint8_t)range;
        }

        const char *ptr = reinterpret_cast<char*>(val.data());

        ostringstream out;
        out << d.time.JavaDate() << '\n';
        out << offset << '\n';
        out << offset+scale << '\n';
        out.write(ptr, val.size()*sizeof(uint8_t));

        ofstream(fPath+"/"+fname+".bin") << out.str();
    }

    // -------------------------------------------------------------------

    struct Statistics
    {
        float min;
        float max;
        float med;
        float avg;
        //float rms;

        template<class T>
            Statistics(const T &t, size_t offset_min=0, size_t offset_max=0)
            : min(0), max(0), med(0), avg(0)
        {
            if (t.size()==0)
                return;

            T copy(t);
            sort(copy.begin(), copy.end());

            if (offset_min>t.size())
                offset_min = 0;
            if (offset_max>t.size())
                offset_max = 0;

            min = copy[offset_min];
            max = copy[copy.size()-1-offset_max];
            avg = accumulate (t.begin(), t.end(), 0.)/t.size();

            const size_t p = t.size()/2;

            med = copy[p];
        }
    };

    void HandleControlMessageImp(const DimData &d)
    {
        if (d.size()==0)
            return;

        const string str  = fControlMessageHist.size()>0 ? fControlMessageHist.back() : "<pre>  :  :  </pre> ";
        const string time = "<pre>"+d.time.GetAsStr("%H:%M:%S")+"</pre> ";

        ostringstream tst;
        tst << d.qos;

        string msg;
        msg += str.substr(0, time.length())==time ? "<pre>        </pre> " : time;
        msg += d.ptr<char>();

        fControlMessageHist.push_back(msg);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << "#ffffff\t";

        for (auto it=fControlMessageHist.begin(); it!=fControlMessageHist.end(); it++)
            out << *it << "<br/>";

        out << '\n';

        ofstream(fPath+"/test.txt") << out.str();
    }

    void HandleControlMessage(const DimData &d)
    {
        if (d.qos==90)
            HandleControlMessageImp(d);
    }

    void HandleControlStateChange(const DimData &d)
    {
        if (d.qos==-2)
            fControlMessageHist.clear();

        if (d.qos<0)
            HandleControlMessageImp(d);
    }

    void HandleMcpConfiguration(const DimData &d)
    {
        if (!CheckDataSize(d, "Mcp:Configuration", 16, true))
            return;

        fMcpConfigurationState     = d.qos;
        fMcpConfigurationMaxTime   = d.get<uint64_t>();
        fMcpConfigurationMaxEvents = d.get<uint64_t>(8);
        fMcpConfigurationName      = d.ptr<char>(16);

        if (d.qos==12)
            fMcpConfigurationRunStart = Time();
    }

    void WriteWeather(const DimData &d, const string &name, int i, float min, float max)
    {
        const Statistics stat(fMagicWeatherHist[i]);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';

        out << "#ffffff\t" << fMagicWeatherHist[i].back() << '\n';
        out << "#ffffff\t" << stat.min << '\n';
        out << "#ffffff\t" << stat.avg << '\n';
        out << "#ffffff\t" << stat.max << '\n';

        ofstream(fPath+"/"+name+".txt") << out.str();

        WriteBinary(d, "magicweather-"+name+"-hist", fMagicWeatherHist[i], max-min, min);
    }

    void HandleMagicWeatherData(const DimData &d)
    {
        if (!CheckDataSize(d, "MagicWeather:Data", 7*4+2))
            return;

        // Store a history of the last 300 entries
        for (int i=kWeatherBegin; i<kWeatherEnd; i++)
        {
            fMagicWeatherHist[i].push_back(d.ptr<float>(2)[i]);
            if (fMagicWeatherHist[i].size()>300)
                fMagicWeatherHist[i].pop_front();
        }

        static const char *dir[] =
        {
            "N", "NNE", "NE", "ENE",
            "E", "ESE", "SE", "SSE",
            "S", "SSW", "SW", "WSW",
            "W", "WNW", "NW", "NNW"
        };


        const uint16_t idx = uint16_t(floor(fMagicWeatherHist[kDir].back()/22.5+16.5))%16;
        //const uint16_t idx = uint16_t(floor(fmod(fMagicWeatherHist[kDir].back()/22.5+360+11.25, 360)/22.5))%16;

        Astro astro(-(17.+53./60+26.525/3600), 28.+45./60+42.462/3600);
        Moon  moon (-(17.+53./60+26.525/3600), 28.+45./60+42.462/3600);

        ostringstream out;
        out << d.time.JavaDate() << '\n';
        out << astro.color << '\t' << astro.description << '\n';
        out << setprecision(2);
        out << (astro.isday?kHtmlWhite:moon.color) << '\t' << moon.description << '\n';
        out << setprecision(3);
        for (int i=0; i<6; i++)
            out << "#ffffff\t" << fMagicWeatherHist[i].back() << '\n';
        out << "#ffffff\t" << dir[idx] << '\n';

        ofstream(fPath+"/magicweather.txt") << out.str();

        out.str("");
        out << astro.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << astro.fSunRiseDarkTime.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunRiseAstronomical.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunRiseCivil.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunRiseDayTime.GetAsStr("%H:%M") << '\n';

        out << kHtmlWhite << '\t' << astro.fSunSetDayTime.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunSetCivil.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunSetAstronomical.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << astro.fSunSetDarkTime.GetAsStr("%H:%M") << '\n';

        out << kHtmlWhite << '\t' << moon.fMoonRise.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << moon.fMoonTransit.GetAsStr("%H:%M") << '\n';
        out << kHtmlWhite << '\t' << moon.fMoonSet.GetAsStr("%H:%M") << '\n';

        ofstream(fPath+"/astro.txt") << out.str();

        WriteWeather(d, "temp",  kTemp,   -5,   35);
        WriteWeather(d, "dew",   kDew,    -5,   35);
        WriteWeather(d, "hum",   kHum,     0,  100);
        WriteWeather(d, "wind",  kWind,    0,  100);
        WriteWeather(d, "gusts", kGusts,   0,  100);
        WriteWeather(d, "press", kPress, 700, 1000);
    }

    void HandleDriveControlPointing(const DimData &d)
    {
        if (!CheckDataSize(d, "DriveControl:Pointing", 16))
            return;

        fDriveControlPointingZd = d.get<double>();

        const double az = d.get<double>(8);

        static const char *dir[] =
        {
            "N", "NNE", "NE", "ENE",
            "E", "ESE", "SE", "SSE",
            "S", "SSW", "SW", "WSW",
            "W", "WNW", "NW", "NNW"
        };

        const uint16_t idx = uint16_t(floor(az/22.5+16.5))%16;
        fDriveControlPointingAz = dir[idx];

        ostringstream out;
        out << d.time.JavaDate() << '\n';

        out << setprecision(5);
        out << fDriveControlPointingZd << '\n';
        out << az << '\n';

        ofstream(fPath+"/drive-pointing.txt") << out.str();
    }

    void HandleDriveControlTracking(const DimData &d)
    {
        if (!CheckDataSize(d, "DriveControl:Tracking", 56))
            return;

        const double Ra  = d.get<double>(0*8);
        const double Dec = d.get<double>(1*8);
        const double Zd  = d.get<double>(3*8);
        const double Az  = d.get<double>(4*8);

        const double zd  = Zd                 * M_PI / 180;
        const double dzd = d.get<double>(5*8) * M_PI / 180;
        const double daz = d.get<double>(6*8) * M_PI / 180;

        // Correct:
        // const double d  = cos(del) - sin(zd+dzd)*sin(zd)*(1.-cos(daz));

        // Simplified:
        double dev = cos(dzd) - sin(zd+dzd)*sin(zd)*(1.-cos(daz));
        dev = acos(dev) * 180 / M_PI * 3600;

        fDriveControlTrackingDevHist.push_back(dev);
        if (fDriveControlTrackingDevHist.size()>300)
            fDriveControlTrackingDevHist.pop_front();

        WriteBinary(d, "control-deviation-hist", fDriveControlTrackingDevHist, 120);

        ostringstream out;
        out << d.time.JavaDate() << '\n';

        out << kHtmlWhite << '\t' << fDriveControlSourceName << '\n';
        out << setprecision(5);
        out << kHtmlWhite << '\t' << Ra  << '\n';
        out << kHtmlWhite << '\t' << Dec << '\n';
        out << setprecision(3);
        out << kHtmlWhite << '\t' << Zd  << '\n';
        out << kHtmlWhite << '\t' << Az  << '\n';
        out << kHtmlWhite << '\t' << dev << '\n';

        Moon moon(-(17.+53./60+26.525/3600), 28.+45./60+42.462/3600);
        if (moon.visible)
        {
            const double angle = moon.Angle(Ra, Dec);

            string col = kHtmlGreen;
            if (angle<35 || angle>145)
                col = kHtmlYellow;
            if (angle<25 || angle>155)
                col = kHtmlRed;
            out << col << '\t' << setprecision(3) << angle << '\n';
        }
        else
            out << kHtmlWhite << "\t&mdash; \n";

        ofstream(fPath+"/tracking.txt") << out.str();
    }

    void HandleDriveControlSource(const DimData &d)
    {
        if (!CheckDataSize(d, "DriveControl:Source", 7*4+2, true))
            return;

        const double *ptr = d.ptr<double>();

        const double ra   = ptr[0];  // Ra[h]
        const double dec  = ptr[1];  // Dec[deg]
        const double woff = ptr[4];  // Wobble offset [deg]
        const double wang = ptr[5];  // Wobble angle  [deg]

        fDriveControlSourceName = d.ptr<char>(6*8);

        ostringstream out;
        out << d.time.JavaDate() << '\n';

        out << "#ffffff\t" << fDriveControlSourceName << '\n';
        out << setprecision(5);
        out << "#ffffff\t" << ra  << '\n';
        out << "#ffffff\t" << dec << '\n';
        out << setprecision(3);
        out << "#ffffff\t" << woff << '\n';
        out << "#ffffff\t" << wang << '\n';

        ofstream(fPath+"/source.txt") << out.str();
    }

    void HandleFeedbackCalibration(const DimData &d)
    {
        if (!CheckDataSize(d, "Feedback:Calibration", 3*4*416))
        {
            fFeedbackCalibration.clear();
            return;
        }

        const float *ptr = d.ptr<float>();
        fFeedbackCalibration.assign(ptr+2*416, ptr+3*416);
    }

    void HandleFeedbackDeviation(const DimData &d)
    {
        if (!CheckDataSize(d, "Feedback:Deviation", 2*4*416+8))
            return;

        const float *ptr = d.ptr<float>();
        vector<float> dev(ptr+416, ptr+416+320);

        fFeedbackTempOffset = ptr[2*416];
        fFeedbackUserOffset = ptr[2*416+1];

        for (int i=0; i<320; i++)
            dev[i] -= fFeedbackTempOffset+fFeedbackUserOffset;

        // Write the 160 patch values to a file
        WriteBinary(d, "feedback-deviation", dev, 1);

        const Statistics stat(dev, 3);

        ostringstream out;
        out << d.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << fFeedbackUserOffset << '\n';
        out << setprecision(3);
        out << kHtmlWhite << '\t' << fFeedbackTempOffset << '\n';
        out << kHtmlWhite << '\t' << stat.min << '\n';
        out << kHtmlWhite << '\t' << stat.med << '\n';
        out << kHtmlWhite << '\t' << stat.avg << '\n';
        out << kHtmlWhite << '\t' << stat.max << '\n';
        ofstream(fPath+"/feedback.txt") << out.str();
    }

    void HandleBiasControlVoltage(const DimData &d)
    {
        if (!CheckDataSize(d, "BiasControl:Voltage", 1664))
        {
            fBiasControlVoltageVec.clear();
            return;
        }

        fBiasControlVoltageVec.assign(d.ptr<float>(), d.ptr<float>()+320);

        const Statistics stat(fBiasControlVoltageVec);

        fBiasControlVoltageMed = stat.med;

        vector<float> val(320, 0);
        for (int i=0; i<320; i++)
        {
            const int idx = (fPixelMap.hv(i).hw()/9)*2+fPixelMap.hv(i).group();
            val[idx] = fBiasControlVoltageVec[i];
        }

        if (fDim->fBiasControl.state()==BIAS::kVoltageOn)
            WriteBinary(d, "biascontrol-voltage", val, 10, 65);
        else
            WriteBinary(d, "biascontrol-voltage", val, 75);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << stat.min << '\n';
        out << kHtmlWhite << '\t' << stat.med << '\n';
        out << kHtmlWhite << '\t' << stat.avg << '\n';
        out << kHtmlWhite << '\t' << stat.max << '\n';
        ofstream(fPath+"/voltage.txt") << out.str();

    }

    void HandleBiasControlCurrent(const DimData &d)
    {
        if (!CheckDataSize(d, "BiasControl:Current", 832))
            return;

        // Convert dac counts to uA
        vector<float> v(320);
        for (int i=0; i<320; i++)
            v[i] = d.ptr<uint16_t>()[i] * 5000./4096;

        const bool cal = fFeedbackCalibration.size()>0 && fBiasControlVoltageVec.size()>0;

        double power_tot = 0;
        double power_apd = 0;

        // 3900 Ohm/n + 1000 Ohm + 1100 Ohm  (with n=4 or n=5)
        const double R[2] = { 3075, 2870 };

        // Calibrate the data (subtract offset)
        if (cal)
            for (int i=0; i<320; i++)
            {
                // Measued current minus leakage current (bias crate calibration)
                v[i] -= fBiasControlVoltageVec[i]/fFeedbackCalibration[i]*1e6;

                // Total power participated in the camera at the G-APD
                // and the serial resistors (total voltage minus voltage
                // drop at resistors in bias crate)
                power_tot += v[i]*(fBiasControlVoltageVec[i] - 1100e-6*v[i])*1e-6;

                // Group index (0 or 1) of the of the pixel (4 or 5 pixel patch)
                const int g = fPixelMap.hv(i).group();

                // Current per G-APD
                v[i] /= g ? 5 : 4;

                // Power consumption per G-APD
                if (i!=66 && i!=191 && i!=193)
                    power_apd += v[i]*(fBiasControlVoltageVec[i]-R[g]*v[i]*1e-6)*1e-6;
            }

        // Divide by number of summed channels, concert to mW
        power_apd /= 317e-3; // [mW]

        if (power_tot<1e-3)
            power_tot = 0;
        if (power_apd<1e-3)
            power_apd = 0;

        // Get the maximum of each patch
        vector<float> val(320, 0);
        for (int i=0; i<320; i++)
        {
            const int idx = (fPixelMap.hv(i).hw()/9)*2+fPixelMap.hv(i).group();
            val[idx] = v[i];
        }

        // Write the 160 patch values to a file
        WriteBinary(d, "biascontrol-current", val, 100);

        const Statistics stat(v, 0, 3);

        // Exclude the three crazy channels
        fBiasControlCurrentMed = stat.med;
        fBiasControlCurrentMax = stat.max;

        // Store a history of the last 60 entries
        fBiasControlCurrentHist.push_back(fBiasControlCurrentMed);
        if (fBiasControlCurrentHist.size()>360)
            fBiasControlCurrentHist.pop_front();

        // write the history to a file
        WriteBinary(d, "biascontrol-current-hist", fBiasControlCurrentHist, 100);

        const string col0 = cal ? kHtmlGreen : kHtmlWhite;

        string col1 = col0;
        string col2 = col0;
        string col3 = col0;
        string col4 = col0;

        if (cal && stat.min>65)
            col1 = kYellow;
        if (cal && stat.min>80)
            col1 = kRed;

        if (cal && stat.med>65)
            col2 = kYellow;
        if (cal && stat.med>80)
            col2 = kRed;

        if (cal && stat.avg>65)
            col3 = kYellow;
        if (cal && stat.avg>80)
            col3 = kRed;

        if (cal && stat.max>65)
            col4 = kYellow;
        if (cal && stat.max>80)
            col4 = kRed;

        ostringstream out;
        out << setprecision(2);
        out << d.time.JavaDate() << '\n';
        out << col0 << '\t' << (cal?"yes":"no") << '\n';
        out << col1 << '\t' << stat.min << '\n';
        out << col2 << '\t' << stat.med << '\n';
        out << col3 << '\t' << stat.avg << '\n';
        out << col4 << '\t' << stat.max << '\n';
        out << kHtmlWhite << '\t' << power_tot << "W [" << power_apd << "mW]\n";
        ofstream(fPath+"/current.txt") << out.str();
    }

    void HandleFadEvents(const DimData &d)
    {
        if (!CheckDataSize(d, "FadControl:Events", 4*4))
        {
            fFadControlNumEvents = -1;
            return;
        }

        fFadControlNumEvents = d.get<uint32_t>();
    }

    void HandleFadConnections(const DimData &d)
    {
        if (!CheckDataSize(d, "FadControl:Connections", 41))
        {
            //fStatusEventBuilderLabel->setText("Offline");
            return;
        }

        string rc(40, '-'); // orange/red [45]

        const uint8_t *ptr = d.ptr<uint8_t>();

        int c[4] = { '.', '.', '.', '.' };

        for (int i=0; i<40; i++)
        {
            const uint8_t stat1 = ptr[i]&3;
            const uint8_t stat2 = ptr[i]>>3;

            if (stat1==0 && stat2==0)
                rc[i] = '.'; // gray [46]
            else
                if (stat1>=2 && stat2==8)
                    rc[i] = stat1==2?'+':'*';  // green [43] : check [42]

            if (rc[i]<c[i/10])
                c[i/10] = rc[i];
        }

        string col[4];
        for (int i=0; i<4; i++)
            switch (c[i])
            {
            case '.': col[i]=kHtmlWhite;  break;
            case '-': col[i]=kHtmlRed;    break;
            case '+': col[i]=kHtmlYellow; break;
            case '*': col[i]=kHtmlGreen;  break;
            }

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << col[0] << '\t' << rc.substr( 0, 10) << '\n';
        out << col[1] << '\t' << rc.substr(10, 10) << '\n';
        out << col[2] << '\t' << rc.substr(20, 10) << '\n';
        out << col[3] << '\t' << rc.substr(30, 10) << '\n';
        ofstream(fPath+"/fad.txt") << out.str();
    }

    void HandleFtmControlTriggerRates(const DimData &d)
    {
        if (!CheckDataSize(d, "FtmControl:TriggerRates", 24+160+640+8))
            return;

        // New run started
        if (d.get<float>(20)<0)
            return;

        fFtmControlTriggerRateCam = d.get<float>(20);

        const float *brates = d.ptr<float>(24);     // Board rate
        const float *prates = d.ptr<float>(24+160); // Patch rate

        // Store a history of the last 60 entries
        fFtmControlTriggerRateHist.push_back(fFtmControlTriggerRateCam);
        if (fFtmControlTriggerRateHist.size()>60)
            fFtmControlTriggerRateHist.pop_front();

        // FIXME: Add statistics for all kind of rates

        WriteBinary(d, "ftmcontrol-triggerrate-hist",
                    fFtmControlTriggerRateHist, 100);
        WriteBinary(d, "ftmcontrol-boardrates",
                    vector<float>(brates, brates+40), 10);
        WriteBinary(d, "ftmcontrol-patchrates",
                    vector<float>(prates, prates+160), 10);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << "#ffffff\t" << fFtmControlTriggerRateCam << '\n';

        ofstream(fPath+"/trigger.txt") << out.str();

        const Statistics bstat(vector<float>(brates, brates+40));
        const Statistics pstat(vector<float>(prates, prates+160));

        out.str("");
        out << d.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << bstat.min << '\n';
        out << kHtmlWhite << '\t' << bstat.med << '\n';
        out << kHtmlWhite << '\t' << bstat.avg << '\n';
        out << kHtmlWhite << '\t' << bstat.max << '\n';
        ofstream(fPath+"/boardrates.txt") << out.str();

        out.str("");
        out << d.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << pstat.min << '\n';
        out << kHtmlWhite << '\t' << pstat.med << '\n';
        out << kHtmlWhite << '\t' << pstat.avg << '\n';
        out << kHtmlWhite << '\t' << pstat.max << '\n';
        ofstream(fPath+"/patchrates.txt") << out.str();
    }

    void HandleFtmControlStaticData(const DimData &d)
    {
        if (!CheckDataSize(d, "FtmControl:StaticData", 740))
            return;

        const uint16_t *ptr = d.ptr<uint16_t>(260);
        vector<uint16_t> vec(ptr, ptr+160);

        WriteBinary(d, "ftmcontrol-thresholds", vec, 1000);

        const Statistics stat(vec);

        ostringstream out;
        out << d.time.JavaDate() << '\n';
        out << kHtmlWhite << '\t' << stat.min << '\n';
        out << kHtmlWhite << '\t' << stat.med << '\n';
        out << kHtmlWhite << '\t' << stat.avg << '\n';
        out << kHtmlWhite << '\t' << stat.max << '\n';
        ofstream(fPath+"/thresholds.txt") << out.str();
    }

    void HandleFtmControlFtuList(const DimData &d)
    {
        if (!CheckDataSize(d, "FtmControl:FtuList", sizeof(FTM::DimFtuList)))
            return;

        const FTM::DimFtuList &sdata = d.ref<FTM::DimFtuList>();

        ostringstream out;
        out << d.time.JavaDate() << '\n';

        int cnt = 0;
        for (int i=0; i<4; i++)
        {
            out << kHtmlWhite << '\t';
            for (int j=0; j<10; j++)
                if (sdata.IsActive(i*10+j))
                {
                    if (sdata.fPing[i*10+j]==1)
                    {
                        out << '*';
                        cnt++;
                    }
                    else
                        out << sdata.fPing[i*10+j];
                }
                else
                    out << '-';
            out << '\n';
        }

        fFtmControlFtuOk = cnt==40;

        ofstream(fPath+"/ftu.txt") << out.str();
    }

    void HandleFadControlEventData(const DimData &d)
    {
        if (!CheckDataSize(d, "FadControl:EventData", 23040))
            return;

        //const float *avg = d.ptr<float>();
        //const float *rms = d.ptr<float>(1440*sizeof(float));
        const float *dat = d.ptr<float>(1440*sizeof(float)*2);
        //const float *pos = d.ptr<float>(1440*sizeof(float)*3);

        vector<float> max(320, -2);
        for (int i=0; i<1440; i++)
        {
            if (i%9==8)
                continue;

            const int idx = (fPixelMap.hw(i).hw()/9)*2+fPixelMap.hw(i).group();
            const double v = dat[i]/1000;
            if (v>max[idx])
                max[idx]=v;
        }

        switch (d.qos)
        {
        case 0:  WriteBinary(d, "fadcontrol-eventdata", max, 2,   -1); break;
        case 1:  WriteBinary(d, "fadcontrol-eventdata", max, 2,    0); break;
        default: WriteBinary(d, "fadcontrol-eventdata", max, 0.25, 0); break;
        }
    }

    void HandleFscControlTemperature(const DimData &d)
    {
        if (!CheckDataSize(d, "FscControl:Temperature", 240))
            return;

        const float *ptr = d.ptr<float>(4);

        double avg =   0;
        double rms =   0;
        double min =  99;
        double max = -99;

        int num = 0;
        for (const float *t=ptr; t<ptr+31; t++)
        {
            if (*t==0)
                continue;

            if (*t>max)
                max = *t;

            if (*t<min)
                min = *t;

            avg += *t;
            rms += *t * *t;

            num++;
        }

        avg /= num;
        rms = sqrt(rms/num-avg*avg);

        if (fMagicWeatherHist[kTemp].size()>0)
        {
            fFscControlTemperatureHist.push_back(avg-fMagicWeatherHist[kTemp].back());
            if (fFscControlTemperatureHist.size()>300)
                fFscControlTemperatureHist.pop_front();
        }

        const Statistics stat(fFscControlTemperatureHist);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << "#ffffff\t" << fFscControlHumidityAvg << '\n';
        out << "#ffffff\t" << min      << '\n';
        out << "#ffffff\t" << avg      << '\n';
        out << "#ffffff\t" << max      << '\n';
        out << "#ffffff\t" << stat.min << '\n';
        out << "#ffffff\t" << stat.avg << '\n';
        out << "#ffffff\t" << stat.max << '\n';

        ofstream(fPath+"/fsc.txt") << out.str();

        WriteBinary(d, "fsccontrol-temperature-hist",
                    fFscControlTemperatureHist, 10);
    }

    void HandleFscControlHumidity(const DimData &d)
    {
        if (!CheckDataSize(d, "FscControl:Humidity", 5*4))
            return;

        const float *ptr = d.ptr<float>(4);

        double avg =   0;
        int num = 0;

        for (const float *t=ptr; t<ptr+4; t++)
            if (*t>0)
            {
                avg += *t;
                num++;
            }

        fFscControlHumidityAvg = avg/num;
    }

    void HandleRateScanData(const DimData &d)
    {
        if (!CheckDataSize(d, "RateScan:Data", 824))
            return;

        const uint64_t id   = d.get<uint64_t>();
        const float   *rate = d.ptr<float>(20);

        if (fRateScanDataId!=id)
        {
            for (int i=0; i<41; i++)
                fRateScanDataHist[i].clear();
            fRateScanDataId = id;
        }
        fRateScanDataHist[0].push_back(log10(rate[0]));

        double max = 0;
        for (int i=1; i<41; i++)
        {
            fRateScanDataHist[i].push_back(log10(rate[i]));
            if (rate[i]>max)
                max = rate[i];
        }

        fRateScanBoard ++;
        fRateScanBoard %= 40;

        WriteBinary(d, "ratescan-hist",  fRateScanDataHist[0],                10, -2);
        WriteBinary(d, "ratescan-board", fRateScanDataHist[fRateScanBoard+1], 10, -2);

        ostringstream out;
        out << setprecision(3);
        out << d.time.JavaDate() << '\n';
        out << "#ffffff\t" << pow(10, fRateScanDataHist[0].back()) << '\n';
        out << "#ffffff\t" << max << '\n';

        ofstream(fPath+"/ratecan.txt") << out.str();

        out.str("");
        out << d.time.JavaDate() << '\n';
        out << "#ffffff\t" << int(fRateScanBoard) << '\n';
        out << "#ffffff\t" << pow(10, fRateScanDataHist[fRateScanBoard+1].back()) << '\n';

        ofstream(fPath+"/ratecan_board.txt") << out.str();
    }

    // -------------------------------------------------------------------

    void HandleDoTest(const DimData &d)
    {
        ostringstream out;
        out << d.time.JavaDate() << '\t' << fDim->fControl.online() << '\n';
        switch (d.qos)
        {
        case -3: out << kHtmlWhite << "\tNot running\n";               break;
        case -2: out << kHtmlBlue  << "\tLoading\n";                   break;
        case -1: out << kHtmlBlue  << "\tStarted\n";                   break;
        default: out << kHtmlGreen << "\tRunning [" << d.qos << "]\n"; break;
        }

        ofstream(fPath+"/dotest.txt") << out.str();
    }

    // -------------------------------------------------------------------

    void infoHandler()
    {
        DimInfo *curr = getInfo(); // get current DimInfo address
        if (!curr)
            return;

        if (HandleService(curr, fDim->fMcpConfiguration,       &StateMachineSmartFACT::HandleMcpConfiguration))
            return;
        if (HandleService(curr, fDim->fMagicWeatherData,       &StateMachineSmartFACT::HandleMagicWeatherData))
            return;
        if (HandleService(curr, fDim->fDriveControlPointing,   &StateMachineSmartFACT::HandleDriveControlPointing))
            return;
        if (HandleService(curr, fDim->fDriveControlTracking,   &StateMachineSmartFACT::HandleDriveControlTracking))
            return;
        if (HandleService(curr, fDim->fDriveControlSource,     &StateMachineSmartFACT::HandleDriveControlSource))
            return;
        if (HandleService(curr, fDim->fFeedbackDeviation,      &StateMachineSmartFACT::HandleFeedbackDeviation))
            return;
        if (HandleService(curr, fDim->fFeedbackCalibration,    &StateMachineSmartFACT::HandleFeedbackCalibration))
            return;
        if (HandleService(curr, fDim->fBiasControlVoltage,     &StateMachineSmartFACT::HandleBiasControlVoltage))
            return;
        if (HandleService(curr, fDim->fBiasControlCurrent,     &StateMachineSmartFACT::HandleBiasControlCurrent))
            return;
        if (HandleService(curr, fDim->fFadConnections,         &StateMachineSmartFACT::HandleFadConnections))
            return;
        if (HandleService(curr, fDim->fFadEvents,              &StateMachineSmartFACT::HandleFadEvents))
            return;
        if (HandleService(curr, fDim->fFtmControlTriggerRates, &StateMachineSmartFACT::HandleFtmControlTriggerRates))
            return;
        if (HandleService(curr, fDim->fFtmControlStaticData,   &StateMachineSmartFACT::HandleFtmControlStaticData))
            return;
        if (HandleService(curr, fDim->fFtmControlFtuList,      &StateMachineSmartFACT::HandleFtmControlFtuList))
            return;
        if (HandleService(curr, fDim->fFadControlEventData,    &StateMachineSmartFACT::HandleFadControlEventData))
            return;
        if (HandleService(curr, fDim->fFscControlTemperature,  &StateMachineSmartFACT::HandleFscControlTemperature))
            return;
        if (HandleService(curr, fDim->fFscControlHumidity,     &StateMachineSmartFACT::HandleFscControlHumidity))
            return;
        if (HandleService(curr, fDim->fControlMessage,         &StateMachineSmartFACT::HandleControlMessage))
            return;
        if (HandleService(curr, fDim->fRateScanData,           &StateMachineSmartFACT::HandleRateScanData))
            return;
    }

    bool CheckEventSize(size_t has, const char *name, size_t size)
    {
        if (has==size)
            return true;

        ostringstream msg;
        msg << name << " - Received event has " << has << " bytes, but expected " << size << ".";
        Fatal(msg);
        return false;
    }

    void PrintState(const DimState &state) const
    {
        const State rc = fDim->GetState(state);

        Out() << state.time().GetAsStr("%H:%M:%S.%f").substr(0, 12) << " - ";
        Out() << kBold << state.name() << ": ";
        if (rc.index==-3)
        {
            Out() << kReset << "Offline" << endl;
            return;
        }
        if (rc.index==-2)
            Out() << state.state();
        else
            Out() << rc.name << "[" << rc.index << "]";
        Out() << kReset << " - " << kBlue << rc.comment << endl;
    }

    int Print() const
    {
        Out() << fDim->fDNS.time().GetAsStr("%H:%M:%S.%f").substr(0, 12) << " - ";
        Out() << kBold << "DIM_DNS: " << fDim->fDNS.version() << endl;

        PrintState(fDim->fMcp);
        PrintState(fDim->fControl);
        PrintState(fDim->fDataLogger);
        PrintState(fDim->fDriveControl);
        PrintState(fDim->fFadControl);
        PrintState(fDim->fFtmControl);
        PrintState(fDim->fBiasControl);
        PrintState(fDim->fFeedback);
        PrintState(fDim->fRateControl);
        PrintState(fDim->fFscControl);
        PrintState(fDim->fMagicWeather);
        PrintState(fDim->fRateScan);
        PrintState(fDim->fChatServer);

        return GetCurrentState();
    }

    string GetStateHtml(const DimState &state, int green) const
    {
        if (!state.online())
            return kHtmlWhite+"\t&mdash;\n";

        if (&state==&fDim->fControl)
        {
            ostringstream out;
            out << kHtmlGreen << '\t';
            return kHtmlGreen+'\t'+(state.state()==-3?"Idle":state.msg)+'\n';
        }

        const State rc = fDim->GetState(state);

        // Sate not found in list, server online (-3: offline; -2: not found)
        if (rc.index==-2)
        {
            ostringstream out;
            out << kHtmlWhite << '\t' << state.state() << '\n';
            return out.str();
        }

        //ostringstream msg;
        //msg << kHtmlWhite << '\t' << rc.name << " [" << rc.index << "]\n";
        //return msg.str();

        if (rc.index<1)
            return kHtmlWhite + "\t&mdash;\n";


        return (rc.index<green?kHtmlYellow:kHtmlGreen) + '\t' + rc.name + '\n';
    }

    int Execute()
    {
        // Dispatch (execute) at most one handler from the queue. In contrary
        // to run_one(), it doesn't wait until a handler is available
        // which can be dispatched, so poll_one() might return with 0
        // handlers dispatched. The handlers are always dispatched/executed
        // synchronously, i.e. within the call to poll_one()
        //poll_one();

        if (fDim->fDNS.state()==0)
            return kStateDimNetworkNA;

        Time now;
        if (now-fLastUpdate<boost::posix_time::seconds(1))
            return kStateRunning;

        fLastUpdate=now;

        ostringstream out;
        out << now.JavaDate() << '\t' << fDim->fControl.online() << '\n';
        out << setprecision(3);

        // -------------- System status --------------
        if (fDim->fMcp.state()>=5) // Idle
        {
            string col = kHtmlBlue;
            /*
            if (fMcpConfigurationState!= 5 &&  // Idle
                fMcpConfigurationState!=11 &&  // Trigger On
                fMcpConfigurationState!=12)    // Taking Data*/
            if (fDim->fMcp.state()!= 5 &&  // Idle
                fDim->fMcp.state()!=11 &&  // Trigger On
                fDim->fMcp.state()!=12)    // Taking Data
                col = kHtmlYellow;
            else
                if (fDim->fFadControl.state()==FAD::kWritingData)
                    col = kHtmlGreen;

            out << col << '\t';

            if (fDim->fRateControl.state()!=5 && fDim->fRateScan.state()!=5)
            {
                switch (fDim->fMcp.state()/*fMcpConfigurationState*/)
                {
                // kStateIdle
                case  5: out << "Idle [" << fMcpConfigurationName << "]"; break;
                // Configuring1 - Configuring3
                case  7:
                case  8:
                case  9: out << "Configuring [" << fMcpConfigurationName << "]"; break;
                // Configured
                case 10: out << "Configured [" << fMcpConfigurationName << "]"; break;
                // TriggerOn / Taking Data
                case 11: 
                case 12: out << fMcpConfigurationName; break;
                }
            }
            else
                if (fDim->fRateControl.state()==5/*kStateSettingGlobalThreshold*/)
                    out << "Calibrating threshold";
                else

                    if (fDim->fRateScan.state()==5/*kStateSettingGlobalThreshold*/)
                        out << "Rate scan in progress";

            if (fDim->fMcp.state()>10 && fDim->fRateControl.state()!=5)
            {
                if (fMcpConfigurationMaxEvents>0 || fMcpConfigurationMaxTime>0 || fMcpConfigurationState==12)
                    out << " [";
                if (fMcpConfigurationMaxEvents>0)
                {
                    if (fFadControlNumEvents>0 && fMcpConfigurationState==12)
                        out << fMcpConfigurationMaxEvents-fFadControlNumEvents;
                    else
                        out << fMcpConfigurationMaxEvents;
                }
                if (fMcpConfigurationMaxEvents>0 && (fMcpConfigurationMaxTime>0 || fMcpConfigurationState==12))
                    out << '/';
                if (fMcpConfigurationMaxTime>0)
                {
                    if (fMcpConfigurationState==12)
                    {

                        const uint32_t dt = (Time()-fMcpConfigurationRunStart).total_seconds();
                        if (dt>fMcpConfigurationMaxTime)
                            out << "---";
                        else
                            out << fMcpConfigurationMaxTime-dt << 's';
                    }
                    else
                        out << fMcpConfigurationMaxTime << 's';
                }
                else
                {
                    if (fMcpConfigurationState==12)
                    {
                        ostringstream d;
                        d << Time()-fMcpConfigurationRunStart;
                        out << d.str().substr(3, 5);
                    }
                }

                if (fMcpConfigurationMaxEvents>0 || fMcpConfigurationMaxTime>0 || fMcpConfigurationState==12)
                    out << ']';
            }
        }
        else
            out << kHtmlWhite;
        out << '\n';

        // ------------------ Drive -----------------
        if (fDim->fDriveControl.state()>=5)   // Armed, Moving, Tracking
        {
            const double dev = fDriveControlTrackingDevHist.size()>0 ? fDriveControlTrackingDevHist.back() : 0;
            const State rc = fDim->GetState(fDim->fDriveControl);
            string col = kHtmlGreen;
            if (rc.index==6) // Moving
                col = kHtmlBlue;
            if (rc.index==5) // Armed
                col = kHtmlWhite;
            if (rc.index==7) // Tracking
            {
                if (dev>60)   // ~1.5mm
                    col = kHtmlYellow;
                if (dev>100)  // ~1/4 of a pixel ~ 2.5mm
                    col = kHtmlRed;
            }
            out << col << '\t';

            //out << rc.name << '\t';
            out << fDriveControlPointingZd  << '\t';
            out << fDriveControlPointingAz  << '\t';
            if (fDim->fDriveControl.state()==7)
            {
                out << fDriveControlSourceName  << '\t';
                out << setprecision(2);
                out << dev << '\n';
                out << setprecision(3);
            }
            else
                out << "\t\n";
        }
        else
            out << kHtmlWhite << '\n';

        // ------------------- FSC ------------------
        if (fDim->fFscControl.state()>1 && fFscControlTemperatureHist.size()>0)
        {
            out << kHtmlGreen << '\t' << fFscControlTemperatureHist.back() << '\n';
        }
        else
            out << kHtmlWhite << '\n';

        // --------------- MagicWeather -------------
        if (fDim->fMagicWeather.state()==3 && fMagicWeatherHist[kWeatherBegin].size()>0)
        {
            /*
            const float diff = fMagicWeatherHist[kTemp].back()-fMagicWeatherHist[kDew].back();
            string col1 = kHtmlRed;
            if (diff>0.3)
                col1 = kHtmlYellow;
            if (diff>0.7)
                col1 = kHtmlGreen;
                */

            const float wind = fMagicWeatherHist[kGusts].back();
            const float hum  = fMagicWeatherHist[kHum].back();
            string col = kHtmlGreen;
            if (wind>35 || hum>95)
                col = kHtmlYellow;
            if (wind>50 || hum>98)
                col = kHtmlRed;

            out << col << '\t';
            out << fMagicWeatherHist[kHum].back()   << '\t';
            out << setprecision(2);
            out << fMagicWeatherHist[kGusts].back() << '\n';
            out << setprecision(3);
        }
        else
            out << kHtmlWhite << "\n";

        // --------------- FtmControl -------------
        if (fDim->fFtmControl.state()==FTM::kTriggerOn)
        {
            string col = kHtmlGreen;
            if (fFtmControlTriggerRateCam<15)
                col = kHtmlYellow;
            if (fFtmControlTriggerRateCam>100)
                col = kHtmlRed;

            out << col << '\t' << fFtmControlTriggerRateCam << '\n';
        }
        else
            out << kHtmlWhite << '\n';

        // --------------- BiasControl -------------
        if (fDim->fBiasControl.state()==BIAS::kRamping     ||
            fDim->fBiasControl.state()==BIAS::kOverCurrent ||
            fDim->fBiasControl.state()==BIAS::kVoltageOn   ||
            fDim->fBiasControl.state()==BIAS::kVoltageOff)
        {
            const bool off = fDim->fBiasControl.state()==BIAS::kVoltageOff;
            const bool oc  = fDim->fBiasControl.state()==BIAS::kOverCurrent;

            string col = fBiasControlVoltageMed>3?kHtmlGreen:kHtmlWhite;
            if (fBiasControlCurrentMax>65)
                col = kHtmlYellow;
            if (fBiasControlCurrentMax>80)
                col = kHtmlRed;

            // Bias in overcurrent => Red
            if (fDim->fBiasControl.state()==BIAS::kOverCurrent)
                col = kHtmlRed;

            // MCP in ReadyForDatataking/Configuring/Configured/TriggerOn/TakingData
            // and Bias not in "data-taking state' => Red
            if (fDim->fMcp.state()>5 &&
                fDim->fBiasControl.state()!=BIAS::kVoltageOn &&
                fDim->fBiasControl.state()!=BIAS::kVoltageOff)
                col = kHtmlRed;

            const bool cal = fFeedbackCalibration.size();

            // Feedback is currently calibrating => Blue
            if (fDim->fFeedback.state()==13)
            {
                out << kHtmlBlue << '\t';
                out << "***\t";
                out << "***\t";
            }
            else
            {
                out << setprecision(2);
                out << col << '\t';
                out << (off ? 0 : fBiasControlCurrentMed) << '\t';
                if (oc)
                    out << "(OC) ";
                else
                {
                    if (cal)
                        out << (off ? 0 : fBiasControlCurrentMax);
                    else
                        out << "&mdash; ";
                }
                out << '\t';
                out << setprecision(3);
            }
            out << (off ? 0 : fBiasControlVoltageMed) << '\n';
        }
        else
            out << kHtmlWhite << '\n';


        // ------------------------------------------

        ofstream(fPath+"/fact.txt") << out.str();

        // ==========================================

        out.str("");
        out << now.JavaDate() << '\t' << fDim->fControl.online() << '\n';

        if (fDim->fDNS.state()==0)
            out << kHtmlWhite << "\tOffline\n\n\n\n\n\n\n\n\n\n\n\n";
        else
        {
            out << kHtmlGreen << '\t' << fDim->fDNS.version() << '\n';

            out << GetStateHtml(fDim->fMcp,          4);
            out << GetStateHtml(fDim->fControl,      0);
            out << GetStateHtml(fDim->fDataLogger,   1);
            out << GetStateHtml(fDim->fDriveControl, 2);
            out << GetStateHtml(fDim->fFadControl,   FAD::kConnected);
            out << GetStateHtml(fDim->fFtmControl,   FTM::kConnected);
            out << GetStateHtml(fDim->fBiasControl,  BIAS::kConnected);
            out << GetStateHtml(fDim->fFeedback,     4);
            out << GetStateHtml(fDim->fRateControl,  4);
            out << GetStateHtml(fDim->fFscControl,   2);
            out << GetStateHtml(fDim->fMagicWeather, 2);
            out << GetStateHtml(fDim->fRateScan,     4);
            out << GetStateHtml(fDim->fChatServer,   1);
        }

        ofstream(fPath+"/status.txt") << out.str();

        return kStateRunning;
    }

public:
    StateMachineSmartFACT(ostream &out=cout) : StateMachineDim(out, "SMART_FACT"),
        fPath("www/smartfact/data"),
        fMcpConfigurationMaxTime(0),
        fMcpConfigurationMaxEvents(0),
        fFscControlHumidityAvg(0),
        fRateScanDataId(0),
        fRateScanBoard(0),
        fDim(0)
    {
        // State names
        AddStateName(kStateDimNetworkNA, "DimNetworkNotAvailable",
                     "The Dim DNS is not reachable.");

        AddStateName(kStateRunning, "Running", "");

        AddEvent("PRINT")
            (bind(&StateMachineSmartFACT::Print, this))
            ("");

    }
    ~StateMachineSmartFACT()
    {
        delete fDim;
    }
    int EvalOptions(Configuration &conf)
    {
        if (!fPixelMap.Read(conf.Get<string>("pixel-map-file")))
        {
            Error("Reading mapping table from "+conf.Get<string>("pixel-map-file")+" failed.");
            return 1;
        }

        fPath = conf.Get<string>("path");

        fDim = new DimSubscriptions(this);
        fDim->fControl.SetCallback(bind(&StateMachineSmartFACT::HandleControlStateChange, this, placeholders::_1));
        fDim->fControl.AddCallback("dotest.dim", bind(&StateMachineSmartFACT::HandleDoTest, this, placeholders::_1));

        return -1;
    }
};

// ------------------------------------------------------------------------

#include "Main.h"

template<class T>
int RunShell(Configuration &conf)
{
    return Main::execute<T, StateMachineSmartFACT>(conf);
}

void SetupConfiguration(Configuration &conf)
{
    po::options_description control("Smart FACT");
    control.add_options()
        ("pixel-map-file", var<string>("FACTmapV5a.txt"),     "Pixel mapping file. Used here to get the default reference voltage")
        ("path",           var<string>("www/smartfact/data"), "Output path for the data-files")
        ;

    conf.AddOptions(control);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "SmartFACT is a tool writing the files needed for the SmartFACT web interface.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: smartfact [-c type] [OPTIONS]\n"
        "  or:  smartfact [OPTIONS]\n";
    cout << endl;
}

void PrintHelp()
{
    Main::PrintHelp<StateMachineSmartFACT>();

    /* Additional help text which is printed after the configuration
     options goes here */

    /*
     cout << "bla bla bla" << endl << endl;
     cout << endl;
     cout << "Environment:" << endl;
     cout << "environment" << endl;
     cout << endl;
     cout << "Examples:" << endl;
     cout << "test exam" << endl;
     cout << endl;
     cout << "Files:" << endl;
     cout << "files" << endl;
     cout << endl;
     */
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    Main::SetupConfiguration(conf);
    SetupConfiguration(conf);

    if (!conf.DoParse(argc, argv, PrintHelp))
        return -1;

    //try
    {
        // No console access at all
        if (!conf.Has("console"))
        {
//            if (conf.Get<bool>("no-dim"))
//                return RunShell<LocalStream, StateMachine, ConnectionFSC>(conf);
//            else
                return RunShell<LocalStream>(conf);
        }
        // Cosole access w/ and w/o Dim
/*        if (conf.Get<bool>("no-dim"))
        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell, StateMachine, ConnectionFSC>(conf);
            else
                return RunShell<LocalConsole, StateMachine, ConnectionFSC>(conf);
        }
        else
*/        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell>(conf);
            else
                return RunShell<LocalConsole>(conf);
        }
    }
    /*catch (std::exception& e)
    {
        cerr << "Exception: " << e.what() << endl;
        return -1;
    }*/

    return 0;
}
