/**
 * @fileOverview
 *    Documentation of dim namespace.
 */

/**
 * @namespace
 *
 * Namespace for extension functions dealing with the DIM network.
 *
 * @author <a href="mailto:thomas.bretz@epfl.ch">Thomas Bretz</a>
 */
var dim = { };

/**
 * Sleep for a while. This can be used to just wait or give time
 * back to the operating system to produce less CPU load if the
 * main purpose of a loop is, e.g., to wait for something to happen.
 *
 * @param {Integer} [milliseconds=0]
 *     Number of millliseconds to sleep. Note that even 0 will always
 *     sleep at least one millisecond.
 *
 */
dim.sleep = function() { /* [native code] */ }

/**
 *
 * Post a message into the dim log stream.
 *
 * It will be logged by the datalogger, displayed on the console
 * and in the smartfact web-gui.
 *
 * @param argument
 *     Any kind of argument can be given. If it is not a String, it
 *     is converted using the toString() member function.
 *
 * @param [. . .]
 *     Any number of additional arguments. Each argument will appear in
 *     a new line.
 *
 * @example
 *     dim.print("Five="+5, "--- new line ---");
 *
 */
dim.print = function() { /* [native code] */ }

/**
 *
 * Posts a message to the dim network with alarm severity.
 *
 * Similar to dim.print, but the message is posted to the network
 * with alarm severity. This means that it is displayed in red
 * and the smartfact web-gui will play an alarm sound.
 * The alarm state will stay valid (displayed in the web-gui) until it
 * is reset.
 *
 * @param argument
 *     Any kind of argument can be given. If it is not a String, it
 *     is converted using the toString() member function.
 *
 * @param [. . .]
 *     Any number of additional arguments. Each argument will appear as
 *     individual alarm.
 *
 * @example
 *     dim.alarm("Alarm for 30 seconds!");
 *     dim.sleep(30000);
 *     dim.alarm();
 */
dim.alarm = function() { /* [native code] */ }

/**
 *
 * Displays a message on the local console (only).
 *
 * @param argument
 *     Any kind of argument can be given. If it is not a String, it
 *     is converted using the toString() member function.
 *
 * @param [. . .]
 *     Any number of additional arguments. Each argument will appear in
 *     a new line.
 *
 * @example
 *     dim.out("Five="+5, "--- new line ---");
 *
 */
dim.out = function() { /* [native code] */ }

/**
 *
 * Send a dim command to a dim client.
 *
 * @param {String} commandId
 *     The command id is a string and usually compiles like
 *     'SERVER/COMMAND'
 *
 * @param argument
 *     Any kind of argument can be given. Arguments are internally
 *     converted into strings using toString() and processed as
 *     if they were typed on th console.
 *
 * @param [. . .]
 *     Any number of additional arguments.
 *
 * @example
 *     dim.send('DRIVE_CONTROL/TRACK_SOURCE 0.5 180 "Mrk 421"');
 *     dim.send('DRIVE_CONTROL/TRACK_SOURCE', 0.5, 180, 'Mrk 421');
 *
 * @returns
 *     A boolean value is returned whether the command was succesfully
 *     posted into the network or not. Note that true does by no means
 *     mean that the command was sucessfully received or even processed.
 */
dim.send = function() { /* [native code] */ }

/**
 * Returns the state of the given server.
 *
 * @param {String} name
 *     The name of the server of which you want to get the state.
 *
 * @throws
 *    If number or type of arguments is wrong
 *
 * @returns {Object}
 *     An object with the properties 'index' {Integer} and 'name' {String}
 *     is returned if a connection to the server is established and
 *     state information have been received, 'undefined' otherwise. If
 *     the time of the last state change is available it is stored
 *     in the 'property'. If a server disconnected, a valid object will
 *     be returned, but without any properties.
 */
dim.state = function(server, timeout) { /* [native code] */ }

/**
 * Wait for the given state of a server.
 *
 * @param {String} name
 *     The name of the server of which you want to wait for a state.
 *     The name must not contain quotation marks.
 *
 * @param {String,Integer} state
 *     The state you want to wait for. It can either be given as an Integer
 *     or as the corresponding short name. If given as String it must
 *     not contain quotation marks.
 *
 * @param {Integer} [timeout]
 *     An optional timeout. If no timeout is given, waiting will not stop
 *     until the condition is fulfilled. A timeout of 0 is allowed and
 *     will essentially just check if the server is in this state or not.
 *
 * @throws
 *    <li> If number or type of arguments is wrong
 *    <li> If no connection to the server is established or no state
 *         has been received yet. This is identical to dim.state()
 *         returning 'undefined'.
 *
 * @returns {Boolean}
 *     true if the state was achived within the timeout, false otherwise.
 */
dim.wait = function() { /* [native code] */ }

/**
 *
 * Callback in case of state changes.
 *
 * To install a callback in case the state of a server changes. set
 * the corresponding property of this array to a function. The argument
 * provided to the function is identical with the object returned
 * by dim.state(). In addition the name of the server is added
 * as the 'name' property and the comment sent with the state change
 * as 'comment' property. For the code executed, the same rules apply
 * than for a thread created with Thread.<P>
 *
 * If a state change is defined for a server for which no callback
 * has been set, the special entry '*' is checked.
 *
 *
 * @type Array[Function]
 *
 * @example
 *     dim.onchange['*'] = function(state) { dim.out("State change from "+state.name+" received"); }
 *     dim.onchange['DIM_CONTROL'] = function(state) { dim.out(JSON.stringify(state); }
 *     ...
 *     delete dim.onchange['DIM_CONTROL']; // to remove the callback
 *
 */
dim.onchange = [];
