// amcreportsocket.cpp: Implementierung der Klasse ReportController.
//
//////////////////////////////////////////////////////////////////////

#include "amcreportsocket.h"
#include "activemirrorcontrol.h"
#include "amcstate.h"
#include "heatstate.h"
#include "cc_defs.h"
#include <qsocket.h>
#include <qtimer.h>
#include <qdatetime.h>

extern AMCState*	g_pAMCState;
extern HEATState* g_pHEATState;

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

AMCReportSocket::AMCReportSocket( QObject* parent )
{
	m_pParent = (ActiveMirrorControl*) parent;

  m_qtComTime = QDateTime::currentDateTime();
  m_qtComTime.setTime_t(0);
	m_iComStatus = 0;

	// create the command socket and connect to its signals
	m_pReportSocket = new QSocket( this );
	connect( m_pReportSocket, SIGNAL( hostFound() ), SLOT( hostFound() ) );
	connect( m_pReportSocket, SIGNAL( connected() ), SLOT( connected() ) );
	connect( m_pReportSocket, SIGNAL( connectionClosed() ), SLOT( connectionClosed() ) );
  connect( m_pReportSocket, SIGNAL( delayedCloseFinished() ), SLOT( delayedCloseFinished() ) );
  connect( m_pReportSocket, SIGNAL( readyRead() ), SLOT( readyRead() ) );
  connect( m_pReportSocket, SIGNAL( bytesWritten( int ) ), SLOT( bytesWritten( int ) ) );
  connect( m_pReportSocket, SIGNAL( error( int ) ), SLOT( error( int ) ) );
	m_zConnected = false;

	// create the timer and connect to its signals
	// we ill start the timer when connected every second to send a report
	m_pSecTimer = new QTimer( this );
  connect( m_pSecTimer, SIGNAL(timeout()), SLOT(sendReport()) );
	
  m_pReportSocket->connectToHost( CC_HOST_ADDRESS, AMC_REP_PORT);
}

AMCReportSocket::~AMCReportSocket()
{
	delete m_pSecTimer;
	delete m_pReportSocket;
}

void AMCReportSocket::connectToHost( const QString& m_qsHost, Q_UINT16 m_iPort )
{
  m_pReportSocket->connectToHost( m_qsHost, m_iPort);
}

void AMCReportSocket::hostFound()
{
//	qDebug("Found host");
//	sendedMsgs->append("Found host");
}

void AMCReportSocket::connected()
{
	m_pSecTimer->start( 1000 );
	m_zConnected = true;
	qDebug("Connected to host");	
	emit ccConnected();
}

void AMCReportSocket::connectionClosed()
{
	m_pSecTimer->stop();
	m_zConnected = true;
	qDebug("Other side closed connection");
}

void AMCReportSocket::delayedCloseFinished()
{
	qDebug("Remaining bytes written to socket");
}

void AMCReportSocket::readyRead()
{
	qDebug("Messages available on socket");
}

void AMCReportSocket::bytesWritten( int p_iNum )
{
	QString text;
	text.sprintf("%d bytes written to socket", p_iNum );
//	qDebug( text );
}

void AMCReportSocket::error( int p_iErrno )
{
	QString errText;
	switch( p_iErrno )
	{
	case QSocket::ErrConnectionRefused:
		errText.sprintf( "TCPIP error: ConnectionRefused" );
		emit ccConnectionRefused();
		break;
	case QSocket::ErrHostNotFound :
		errText.sprintf( "TCPIP error: HostNotFound " );
		break;
	case QSocket::ErrSocketRead :
		errText.sprintf( "TCPIP error: SocketRead " );
		break;
	default:
		errText.sprintf( "TCPIP error: Unknown" );
		break;
	}
	qDebug( errText );
	if( m_zConnected )
		m_zConnected = false;
}

void AMCReportSocket::sendReport()
{
	QString cmd;
	QDateTime amcTime = QDateTime::currentDateTime();
	cmd.sprintf("AMC-REPORT %02d %04d %02d %02d %02d %02d %02d %03d %02d %04d %02d %02d %02d %02d %02d %03d %02d PANELS %03d %03d OVER\n",
				 g_pAMCState->getState(),
				 amcTime.date().year(), amcTime.date().month(), amcTime.date().day(),
				 amcTime.time().hour(), amcTime.time().minute(), amcTime.time().second(), amcTime.time().msec(),
				 m_iComStatus,
				 m_qtComTime.date().year(), m_qtComTime.date().month(), m_qtComTime.date().day(),
				 m_qtComTime.time().hour(), m_qtComTime.time().minute(), m_qtComTime.time().second(),
         m_qtComTime.time().msec(),
				 g_pHEATState->getState(),
				 g_pAMCState->installedPanels(), g_pAMCState->errorPanels() );
	emit reportDelivered( amcTime, cmd );
	if( m_zConnected )
	{
		m_pReportSocket->writeBlock( cmd.latin1(), cmd.length() );
	}
}
