/*-----------------------------------------------------------------------------
cal.h -- CAN Application Layer CAL

Copyright (c) 1994 JANZ Computer AG
All Rights Reserved

Created 94/10/11 by Soenke Hansen
Version 1.9 of 98/07/31

CALO: CAN Application Layer Object

CALO's receive CAN messages.  Using only the identifier and the RTR bit
of a CAN message CAL forwards a message to the appropriate CALO.
CALO's are either waiting on an event (e.g., CAN message, timer, user
request) or are ready to do work.  The latter objects are maintained on a
queue until they have finished their work.

Prototypes of functions defined in cal.c.

-----------------------------------------------------------------------------*/


#ifndef cal_DEFINED
#define cal_DEFINED

#ifdef __cplusplus
extern "C" {
#endif

#include "defs.h"
#include "msg.h"


#ifdef FIRMWARE
/* CAN Application Layer Object */
struct calo {
	WORD_t		id;		/* identifier (handle) of object */
	WORD_t		head;		/* head of CAN messages received */
	OBJECT		obj;		/* object specific data */
	void		(*mhdl)();	/* CAL handler for received message */
	struct calo	*nxtrx;		/* next on rx id hash queue */
	struct calo	*nxtcal;	/* next on CAL id hash queue */
};

#define NIL_CALO (struct calo *)0

/* Identifiers (handles) of CAL objects -- not to be confused with
   CAN message Ids -- are 11 bit numbers.  Ids of requests issued to
   ICANOS are 16 bit words with the lower bits equal to the
   Id of the requesting CAL Object.  CALID_MASK is used in retrieving
   the CAL Object Id from the request Id. */
#define CALID_MASK	0x07ff

/* Nil CAN message head (descriptor) */
#define NIL_CAN_HEAD	0xffff	/* invalid descriptor */

#endif /* FIRMWARE */

/* CAN Identifiers used by CAL (cf. CiA/DS204-1, Annex I) */
#define ID_START_STOP		   0	/* Node Start, Stop, Disconnect */
#define ID_CMS_NIL		   0	/* invalid CMS Id */
#define ID_CMS_MIN		   1	/* range of CMS   */
#define ID_CMS_MAX		1760	/*    identifiers */
#define ID_GUARD_NIL		   0	/* invalid guard Id */
#define ID_GUARD_MIN		1761	/* range of guarding */
#define ID_GUARD_MAX		2015	/*    identifiers    */
#define ID_LMT_S		2020	/* from LMT Slave */
#define ID_LMT_M		2021	/* from LMT Master */
#define ID_NMT_IDENTIFY		2022	/* Identify Node Protocol */
#define ID_DBT_S		2023	/* from DBT Slave */
#define ID_DBT_M		2024	/* from DBT Master */
#define ID_NMT_S		2025	/* from NMT Slave */
#define ID_NMT_M		2026	/* from NMT Master */
#define ID_SELFTEST		2027	/* for module selftest */
#define ID_MIN			   0	/* range of identifiers */
#define ID_MAX			2031	/*    controlled by CiA */


/*--------- Prototypes of CAL User Functions -------------------------------*/

#ifdef FIRMWARE

/* Initialize CAL */
extern void InitCal(
	WORD_t,		/* req. clock period */
	WORD_t,		/* bus timing */
	WORD_t,		/* timeout interval */
	void (*)(),	/* user handler for NMT Slave ind/con */
	void (*)(),	/* user handler for DBT Slave ind/con */
	void (*)()	/* user handler for LMT Slave ind/con */
);

/* Free the resources held by CAL */
extern void DeleteCal(void);

/* Execute CAL */
extern void RunCal(void);

/*--------- Enable/Disable Remote Services ---------------------------------*/

#define EnableCAN()	buson_bcan()
#define DisableCAN()	busoff_bcan()

/*--------- Prototypes of CAL internal functions ---------------------------*/

/* Create a CAL object. */
extern struct calo *NewCalO(void);

/* Delete a CAL object. */
extern void DeleteCalO(struct calo *);

/* Get a CAL object by its Id (handle). */
extern struct calo *GetCalO(WORD_t);

/* Assign CAN message head to a CAL Object */
extern int AssignCalO(struct calo *, WORD_t);

/* Unassign CAN message head from a CAL Object */
extern void UnassignCalO(struct calo *);

#endif /* FIRMWARE */

#ifdef __cplusplus
}
#endif

#endif /* !cal_DEFINED */
