//
// This File contains the definition of the MGCoordinate-class
//
//   Author: Thomas Bretz
//   Version: V1.0 (1-8-2000)

#include "MGCoordinate.h"

#include <stdlib.h>    // atoi
#include <iostream.h>  // cout

#include <TSystem.h>
#include <TGLabel.h>
#include <TGTextEntry.h>

#include "slalib.h"
#include "slamac.h"

enum {
    IDM_kDeg,
    IDM_kMin,
    IDM_kSec
};

MGCoordinate::MGCoordinate(const TGWindow* p,
                           const Bool_t flag, const char *txt,
                           const Int_t deg, const UInt_t min, const UInt_t sec)
: TGFrame(p, 114, flag?76:46, kSunkenFrame|kFixedSize), fDeg(deg), fMin(min), fSec(sec)
{
    // p = pointer to MainFrame (not owner)
    if (flag)
    {
        fTextEntryDeg = new TGTextEntry(this, "****", IDM_kDeg);
        fTextEntryMin = new TGTextEntry(this, "***",  IDM_kMin);
        fTextEntrySec = new TGTextEntry(this, "***",  IDM_kSec);
        // fTextEntryDeg->SetAlignment(kTextCenterX);
        // fTextEntryMin->SetAlignment(kTextCenterX);
        // fTextEntrySec->SetAlignment(kTextCenterX);
        fTextEntryDeg->Move( 4, 26);
        fTextEntryMin->Move(44, 26);
        fTextEntrySec->Move(78, 26);
        fTextEntryDeg->MapWindow();
        fTextEntryMin->MapWindow();
        fTextEntrySec->MapWindow();
        fList.Add(fTextEntrySec);
        fList.Add(fTextEntryMin);
        fList.Add(fTextEntryDeg);

        Set(fTextEntryDeg, fDeg);
        Set(fTextEntryMin, fMin);
        Set(fTextEntrySec, fSec);
    }

    const int ypos = flag?56:26;

    fLabelDeg = new TGLabel(this, "****");
    fLabelMin = new TGLabel(this, "***");
    fLabelSec = new TGLabel(this, "***");
    fLabelDeg->SetTextJustify(kTextRight);
    fLabelMin->SetTextJustify(kTextRight);
    fLabelSec->SetTextJustify(kTextRight);
    fLabelDeg->Move( 4+6, ypos);
    fLabelMin->Move(44+6, ypos);
    fLabelSec->Move(78+6, ypos);
    fLabelDeg->MapWindow();
    fLabelMin->MapWindow();
    fLabelSec->MapWindow();
    fList.Add(fLabelSec);
    fList.Add(fLabelDeg);
    fList.Add(fLabelMin);

    Set(fLabelDeg, fDeg);
    Set(fLabelMin, fMin);
    Set(fLabelSec, fSec);

    fLabel = new TGLabel(this, txt);
    fLabel->SetTextJustify(kTextLeft);
    fLabel->Move(4, 4);
    fLabel->MapWindow();
    fList.Add(fLabel);

    TGLabel *label;

    if (flag)
    {
        label = new TGLabel(this, "\xb0");
        label->SetTextJustify(kTextLeft);
        label->Move(36, 26);
        label->MapWindow();
        fList.Add(label);

        label = new TGLabel(this, "'");
        label->SetTextJustify(kTextLeft);
        label->Move(70, 26);
        label->MapWindow();
        fList.Add(label);

        label = new TGLabel(this, "\"");
        label->SetTextJustify(kTextLeft);
        label->Move(104, 26);
        label->MapWindow();
        fList.Add(label);
    }

    label = new TGLabel(this, "\xb0");
    label->SetTextJustify(kTextLeft);
    label->Move(36, ypos);
    label->MapWindow();
    fList.Add(label);

    label = new TGLabel(this, "'");
    label->SetTextJustify(kTextLeft);
    label->Move(70, ypos);
    label->MapWindow();
    fList.Add(label);
 
    label = new TGLabel(this, "\"");
    label->SetTextJustify(kTextLeft);
    label->Move(104, ypos);
    label->MapWindow();
    fList.Add(label);

    MapWindow();
}

MGCoordinate::~MGCoordinate()
{
    cout << "MGCoordinate destroyed." << endl;
}

Double_t MGCoordinate::GetVal() const
{
    const Int_t deg = fDeg<0 ? -fDeg : fDeg;
    const Int_t sgn = fDeg<0 ? -1 : 1;

    return (Double_t)sgn*(60*(60*deg+fMin)+fSec)/3600;
}

void MGCoordinate::Print()
{
    cout << fLabel->GetText()->GetString() << " " << fDeg << "\xb0 " << fMin << "' " << fSec << "\"" << endl;
}

void MGCoordinate::Set(TGLabel *label, Int_t val)
{
    char txt[20];

    sprintf(txt, "%d", val);

    label->SetText(new TGString(txt));
}

void MGCoordinate::Set(TGTextEntry *entry, Int_t val)
{
    char txt[20];

    sprintf(txt, "%d", val);

    entry->SetText(txt);
}

void MGCoordinate::SetVal(const Double_t d)
{
    int dms[4];
    char sign;

    //** ndp    int     number of decimal places of arcseconds
    //** angle  double  angle in radians
    //** sign   char*   '+' or '-'
    //** idmsf  int[4]  degrees, arcminutes, arcseconds, fraction

    slaDr2af(0, d*D2PI/360.0, &sign, dms);


    fDeg = sign=='-'?-dms[0]:dms[0];
    fMin = dms[1];
    fSec = dms[2];

    Set(fLabelDeg, fDeg);
    Set(fLabelMin, fMin);
    Set(fLabelSec, fSec);

    // Set(fTextEntryDeg, fDeg);
    // Set(fTextEntryMin, fMin);
    // Set(fTextEntrySec, fSec);
}


Bool_t MGCoordinate::Set(TGLabel *label, Int_t &val, TGTextEntry *entry)
{
    Int_t newval = atoi(entry->GetText());

    Bool_t ok = (entry == fTextEntryDeg || (newval>=0 && newval<60));

    if (ok)
    {
        val = newval;
        Set(label, val);
    }

    Set(entry, val);

    return ok;
}

Bool_t MGCoordinate::ProcessMessage(Long_t msg, Long_t mp1, Long_t mp2)
{
    if (GET_MSG(msg)!=kC_TEXTENTRY || GET_SUBMSG(msg)!=kTE_ENTER)
        return kTRUE;

    // kC_TEXTENTRY        = 4,
    // kTE_TEXTCHANGED     = 1,
    // kTE_ENTER           = 2,

    switch (mp1)
    {
    case IDM_kDeg:
        if (!Set(fLabelDeg, fDeg, fTextEntryDeg))
            fTextEntryDeg->SelectAll();
        else
        {
            fTextEntryMin->SetFocus();
            fTextEntryMin->SelectAll();
        }
        return kTRUE;

    case IDM_kMin:
        if(!Set(fLabelMin, fMin, fTextEntryMin))
            fTextEntryMin->SelectAll();
        else
        {
            fTextEntrySec->SetFocus();
            fTextEntrySec->SelectAll();
        }
        return kTRUE;

    case IDM_kSec:
        if (!Set(fLabelSec, fSec, fTextEntrySec))
            fTextEntrySec->SelectAll();
        else
        {
            fTextEntryDeg->SetFocus();
            fTextEntryDeg->SelectAll();
        }
        return kTRUE;
    }

    return kTRUE;
}

