#ifndef CANOPEN_H
#define CANOPEN_H

#include "sdolist.h"
#include "vmodican.h"

#include "MTimeout.h"

//#define FALSE       0
//#define TRUE        1

// CAL Function Codes
// COB-ID = function code | node-id (5bits)
#define kNMT           0x0 // only with COB-ID 0
#define kSYNC          0x1 // only with COB-ID 0
#define kTIMESTAMP     0x2 // only with COB-ID 0

#define kEMERGENCY     0x1
#define kPDO1_TX       0x3
#define kPDO2_TX       0x5
#define kPDO3_TX       0x7
#define kPDO4_TX       0x9
#define kSDO_RX        0xb  // this is used to set data of the shaft encoder
#define kSDO_TX        0xc  // this is used to request data from the shaft encoder
#define kNodeguard     0xe

// NMT: no answer to NMT command
// cob-id=0, command (byte), id (byte)
#define kNMT_START     0x01 // change to operational state (start)
#define kNMT_STOP      0x02 // change to prepared    state (stop)
#define kNMT_PREOP     0x80 // enter pre operational state
#define kNMT_RESET     0x81 // reset node (set parameter to power on values)
#define kNMT_REINIT    0x82 // reset communication of node (set communication parameters to power on values)

// command for SDOs
#define kSDO_LEN4      0x3
#define kSDO_LEN2      0xb
#define kSDO_LEN1      0xf

#define kSDO_RXm4      0x22  // this is used with SDO_TX to send a maximum of 4 bytes
#define kSDO_RX4       0x20|kSDO_LEN4  // this is used with SDO_TX to send 4 bytes
#define kSDO_RX2       0x20|kSDO_LEN2  // this is used with SDO_TX to send 2 bytes
#define kSDO_RX1       0x20|kSDO_LEN1  // this is used with SDO_TX to send 1 byte
#define kSDO_RX_DATA   0x40            // this is used to request parameters from the encoder
#define kSDO_TX4       0x40|kSDO_LEN4  // answer to 0x40 with 4 bytes of data
#define kSDO_TX3       0x40|kSDO_LEN2  // answer to 0x40 with 2 bytes of data
#define kSDO_TX1       0x40|kSDO_LEN1  // answer to 0x40 with 1 byte  of data
#define kSDO_TX_OK     0x60            // answer to a SDO_TX message
#define kSDO_TX_ERROR  0x80            // error message (instead of 0x60)

class CanOpen : public VmodIcan
{
private:
    PendingSDOList fSdoList;

    pthread_cond_t  fPdoCond[32][4]; // one for every PDO of every node
    pthread_mutex_t fPdoMux[32][4];

    void HandleCanMessage(WORD_t cobid, BYTE_t *data, struct timeval *tv);

    virtual void HandleSDO(BYTE_t node, BYTE_t cmd, WORD_t idx, BYTE_t subidx, LWORD_t data, struct timeval *tv)=0;
    virtual void HandlePDO1(BYTE_t node, BYTE_t *data, struct timeval *tv)=0;
    virtual void HandlePDO2(BYTE_t node, BYTE_t *data, struct timeval *tv)=0;
    virtual void HandlePDO3(BYTE_t node, BYTE_t *data, struct timeval *tv)=0;
    virtual void HandlePDO4(BYTE_t node, BYTE_t *data, struct timeval *tv)=0;

public:
    CanOpen(const char *dev, const int baud, MLog &out=gLog);
    virtual ~CanOpen();

    void SendPDO1(BYTE_t node, BYTE_t data[8]);
    void SendPDO2(BYTE_t node, BYTE_t data[8]);
    void SendPDO3(BYTE_t node, BYTE_t data[8]);
    void SendPDO4(BYTE_t node, BYTE_t data[8]);
    void SendPDO1(BYTE_t node,
                  BYTE_t m0=0, BYTE_t m1=0, BYTE_t m2=0, BYTE_t m3=0,
                  BYTE_t m4=0, BYTE_t m5=0, BYTE_t m6=0, BYTE_t m7=0);
    void SendPDO2(BYTE_t node,
                  BYTE_t m0=0, BYTE_t m1=0, BYTE_t m2=0, BYTE_t m3=0,
                  BYTE_t m4=0, BYTE_t m5=0, BYTE_t m6=0, BYTE_t m7=0);
    void SendPDO3(BYTE_t node,
                  BYTE_t m0=0, BYTE_t m1=0, BYTE_t m2=0, BYTE_t m3=0,
                  BYTE_t m4=0, BYTE_t m5=0, BYTE_t m6=0, BYTE_t m7=0);
    void SendPDO4(BYTE_t node,
                  BYTE_t m0=0, BYTE_t m1=0, BYTE_t m2=0, BYTE_t m3=0,
                  BYTE_t m4=0, BYTE_t m5=0, BYTE_t m6=0, BYTE_t m7=0);

    void SendSDO(BYTE_t node, WORD_t idx, BYTE_t subidx, BYTE_t val);
    void SendSDO(BYTE_t node, WORD_t idx, BYTE_t subidx, WORD_t val);
    void SendSDO(BYTE_t node, WORD_t idx, BYTE_t subidx, LWORD_t val);

    void SendSDO(BYTE_t node, WORD_t idx, BYTE_t val)   { SendSDO(node, idx, 0, val); }
    void SendSDO(BYTE_t node, WORD_t idx, WORD_t val)   { SendSDO(node, idx, 0, val); }
    void SendSDO(BYTE_t node, WORD_t idx, LWORD_t val)  { SendSDO(node, idx, 0, val); }

    void SendNMT(BYTE_t node, BYTE_t cmd);

    void RequestSDO(BYTE_t node, WORD_t idx, BYTE_t subidx=0);

    void EnableCanMsg(BYTE_t node, BYTE_t fcode, int flag=TRUE);

    void EnableSdoRx(BYTE_t node);
    void EnablePdo1Rx(BYTE_t node);
    void EnablePdo2Rx(BYTE_t node);
    void EnablePdo3Rx(BYTE_t node);
    void EnablePdo4Rx(BYTE_t node);
    void EnableEmcy(BYTE_t node);

    void WaitForNextPdo1(BYTE_t node) { node -= 1; pthread_cond_wait(&fPdoCond[node][0], &fPdoMux[node][0]); }
    void WaitForNextPdo2(BYTE_t node) { node -= 1; pthread_cond_wait(&fPdoCond[node][1], &fPdoMux[node][1]); }
    void WaitForNextPdo3(BYTE_t node) { node -= 1; pthread_cond_wait(&fPdoCond[node][2], &fPdoMux[node][2]); }
    void WaitForNextPdo4(BYTE_t node) { node -= 1; pthread_cond_wait(&fPdoCond[node][3], &fPdoMux[node][3]); }

    //
    // This function must
    //
    virtual int StopWaitingForSDO() const { return FALSE; }

    void WaitForSdos(WORD_t ms=500)
    {
        MTimeout t(ms);
        while (fSdoList.IsPending() &&
               !StopWaitingForSDO() &&
               !t.HasTimedOut())
            usleep(1);

        if (ms && t.HasTimedOut())
            cout << "WaitForSdos timed out." << endl;

        if ((ms && t.HasTimedOut()) || StopWaitingForSDO())
            fSdoList.DelAll();
    }

    void WaitForSdo(BYTE_t node, WORD_t idx, BYTE_t subidx, WORD_t ms=500)
    {
        MTimeout t(ms);
        while (fSdoList.IsPending(node, idx, subidx) &&
               !StopWaitingForSDO() &&
               !t.HasTimedOut())
            usleep(1);

        if (ms && t.HasTimedOut())
            cout << "WaitForSdo #" << (int)node << " " << idx << "/" << (int)subidx << " timed out." << endl;

        if ((ms && t.HasTimedOut()) || StopWaitingForSDO())
            fSdoList.Del(node, idx, subidx);
    }

public:
    WORD_t CobId(BYTE_t node, BYTE_t fcode) const;
};

#endif
