/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!   Author(s): Rudolf Bock     10/2001 <mailto:Rudolf.Bock@cern.ch>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHillasSrc
//
// Storage Container for image parameters
//
//    source-dependent image parameters
// fAlpha    angle between major axis and line source-to-center
// fDist     distance from source to center of ellipse
//
/////////////////////////////////////////////////////////////////////////////
#include "MHillasSrc.h"

#include <fstream.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MSrcPosCam.h"

ClassImp(MHillasSrc);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MHillasSrc::MHillasSrc(const char *name, const char *title)
{
    fName  = name  ? name  : "MHillasSrc";
    fTitle = title ? title : "parameters depending in source position";
}

// --------------------------------------------------------------------------
//
//  calculation of source-dependent parameters
//
void MHillasSrc::Calc(const MHillas *hillas)
{
    const Double_t mx = hillas->GetMeanX();        // [mm]
    const Double_t my = hillas->GetMeanY();        // [mm]

    const Double_t sx = mx - fSrcPos->GetX();      // [mm]
    const Double_t sy = my - fSrcPos->GetY();      // [mm]

    const Double_t tand = tan(hillas->GetDelta()); // [1]

    fDist   = sqrt(sx*sx + sy*sy);                 // [mm]
    fAlpha  = atan2(sy - tand*sx, sx + tand*sy);   // [rad]
    fAlpha *= kRad2Deg;                            // [deg]

    fHillas = hillas;

    SetReadyToSave();
} 

void MHillasSrc::Print(Option_t *) const
{
    *fLog << all;
    *fLog << "Source dependant Image Parameters (" << GetName() << ")" << endl;
    *fLog << " - Dist  = " << fDist  << " mm"  << endl;
    *fLog << " - Alpha = " << fAlpha << " deg" << endl;
}

// -----------------------------------------------------------------------
//
// overloaded MParContainer to read MHillasSrc from an ascii file
//
void MHillasSrc::AsciiRead(ifstream &fin)
{
    fin >> fAlpha;
    fin >> fDist;
}

// -----------------------------------------------------------------------
//
// overloaded MParContainer to write MHillasSrc to an ascii file
/*
void MHillasSrc::AsciiWrite(ofstream &fout) const
{
    fout << fAlpha << " " << fDist;
}
*/
