/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//   MCerPhotCalc                                                           //
//                                                                          //
//   This is a task which calculates the number of photons from the FADC    //
//   time slices. At the moment it integrates simply the FADC values.       //
//                                                                          //
//  Input Containers:                                                       //
//   MRawEvtData, MPedesdtalCam                                             //
//                                                                          //
//  Output Containers:                                                      //
//   MCerPhotEvt                                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MCerPhotCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcRunHeader.hxx"

#include "MRawRunHeader.h"
#include "MRawEvtData.h"       // MRawEvtData::GetNumPixels
#include "MCerPhotEvt.h"
#include "MPedestalPix.h"
#include "MPedestalCam.h"
#include "MRawEvtPixelIter.h"

ClassImp(MCerPhotCalc);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MCerPhotCalc::MCerPhotCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MCerPhotCalc";
    fTitle = title ? title : "Task to calculate Cerenkov photons from raw data";

    AddToBranchList("MRawEvtData.fHiGainPixId");
    AddToBranchList("MRawEvtData.fLoGainPixId");
    AddToBranchList("MRawEvtData.fHiGainFadcSamples");
    AddToBranchList("MRawEvtData.fLoGainFadcSamples");

}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawRunHeader
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Bool_t MCerPhotCalc::PreProcess(MParList *pList)
{
    fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!fRunHeader)
    {
        *fLog << dbginf << "MRawRunHeader not found... aborting." << endl;
        return kFALSE;
    }

    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindObject("MPedestalCam");
    if (!fPedestals)
    {
        *fLog << dbginf << "MPedestalCam not found... aborting." << endl;
        return kFALSE;
    }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the run type and camera version.
// If the file is a MC file and the used camera version is <= 40
// we enable a fix for truncated pedestal means in this version.
//
Bool_t MCerPhotCalc::ReInit(MParList *pList)
{
    const MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    if (runheader->GetRunType() != kRTMonteCarlo)
        return kTRUE;

    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check for camera version, MC run header not found." << endl;
        return kTRUE;
    }

    fEnableFix = kFALSE;
    if (mcrunheader->GetCamVersion() <= 40)
        fEnableFix = kTRUE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MCerPhotEvt container.
//
Bool_t MCerPhotCalc::Process()
{
    fCerPhotEvt->InitSize(fRawEvt->GetNumPixels());

    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
	const UInt_t pixid = pixel.GetPixelId();

        const MPedestalPix &ped = (*fPedestals)[pixid];

	//
	// sanity check (old MC files sometimes have pixids>577)
	//
        if (!fPedestals->CheckBounds(pixid))
        {
	    *fLog << inf << "Pixel ID larger than camera... skipping event." << endl;
	    return kCONTINUE;
	}

        Float_t nphot = (Float_t)pixel.GetSumHiGainSamples();

        //
	// We check that the pixel is not empty, if it is empty
	// we won't substract the pedestal. Empty means that it has
        // 0 signal in all the slices.
        //
        const Double_t mean = fEnableFix ? ped.GetMean()-0.5 : ped.GetMean();
        if (nphot!=0)
            nphot -= fRunHeader->GetNumSamplesHiGain()*mean;

        fCerPhotEvt->AddPixel(pixid, nphot, sqrt(fRunHeader->GetNumSamplesHiGain())*ped.GetSigma());

        // FIXME! Handling of Lo Gains is missing!
    }

    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}
