/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo 7/2002  (moralejo@pd.infn.it)
!
!   Copyright: MAGIC Software Development, 2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//   MCerPhotCalc2                                                          //
//                                                                          //
//   This is a task which calculates the number of photons from the FADC    //
//   time slices. It weights the each slice according to the numbers in     //
//   the array fWeight. The default values (see end of this file) are       //
//   the fraction of signal which, in average, falls within each slice.     //
//   This averages have been determined over a run (about 1000 triggers,    //
//   0 deg z.a. gammas).                                                    // 
//                                                                          //
//  Input Containers:                                                       //
//   MRawRunHeader, MRawEvtData, MPedestalCam                               //
//                                                                          //
//  Output Containers:                                                      //
//   MCerPhotEvt                                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MCerPhotCalc2.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcRunHeader.hxx"

#include "MRawRunHeader.h"
#include "MRawEvtData.h"       // MRawEvtData::GetNumPixels
#include "MCerPhotEvt.h"
#include "MPedestalPix.h"
#include "MPedestalCam.h"
#include "MRawEvtPixelIter.h"

ClassImp(MCerPhotCalc2);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MCerPhotCalc2::MCerPhotCalc2(const char *name, const char *title)
{
    fName  = name  ? name  : "MCerPhotCalc2";
    fTitle = title ? title : "Task to calculate pixel signal from raw data";

    AddToBranchList("MRawEvtData.fHiGainPixId");
    AddToBranchList("MRawEvtData.fLoGainPixId");
    AddToBranchList("MRawEvtData.fHiGainFadcSamples");
    AddToBranchList("MRawEvtData.fLoGainFadcSamples");

    SetDefaultWeights();
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawRunHeader
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Bool_t MCerPhotCalc2::PreProcess(MParList *pList)
{
    fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!fRunHeader)
    {
        *fLog << dbginf << "MRawRunHeader not found... aborting." << endl;
        return kFALSE;
    }

    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedestals)
    {
        *fLog << dbginf << "MPedestalCam not found... aborting." << endl;
        return kFALSE;
    }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
        return kFALSE;

    // Calculate quadratic sum of weights:
    fSumQuadWeights = 0.;
    for (Int_t i = 0; i < fWeight.GetSize(); i++)
      fSumQuadWeights += fWeight[i]*fWeight[i];

    fSumQuadWeights = sqrt(fSumQuadWeights);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the run type and camera version.
// If the file is a MC file and the used camera version is <= 40
// we enable a fix for truncated pedestal means in this version.
//
Bool_t MCerPhotCalc2::ReInit(MParList *pList)
{
    const MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    if (fRunHeader->GetNumSamplesHiGain() != fWeight.GetSize())
    {
        *fLog << dbginf << "Number of FADC slices (" << fRunHeader->GetNumSamplesHiGain() <<") is different from assumed one (" << fWeight.GetSize() << ")... aborting." << endl;
        return kFALSE;
    }

    if (runheader->GetRunType() != kRTMonteCarlo)
        return kTRUE;

    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check for camera version, MC run header not found." << endl;
        return kTRUE;
    }

    fEnableFix = kFALSE;
    if (mcrunheader->GetCamVersion() <= 40)
        fEnableFix = kTRUE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MCerPhotEvt container.
//
Bool_t MCerPhotCalc2::Process()
{
    fCerPhotEvt->InitSize(fRawEvt->GetNumPixels());

    MRawEvtPixelIter pixel(fRawEvt);

    TArrayF BinSignal(fWeight.GetSize());

    while (pixel.Next())
      {
	const UInt_t pixid = pixel.GetPixelId();
        const MPedestalPix &ped = (*fPedestals)[pixid];

	//
	// sanity check (old MC files sometimes have pixids>577)
	//
        if (!fPedestals->CheckBounds(pixid))
        {
	    *fLog << inf << "Pixel ID larger than camera... skipping event." << endl;
	    return kCONTINUE;
	}

	// Mean pedestal:
        const Double_t mean = fEnableFix ? ped.GetMean()-0.5 : ped.GetMean();

	Byte_t *ptr = pixel.GetHiGainSamples();

	Float_t nphot = 0.;

	for(Int_t i = 0; i<fWeight.GetSize(); i++)
	  {
	    BinSignal[i] =  (Float_t) ptr[i] - mean;
	    nphot       +=  BinSignal[i] * fWeight[i];
	  }

	Float_t nphoterr = ped.GetSigma()* fSumQuadWeights;

        fCerPhotEvt->AddPixel(pixid, nphot, nphoterr);

        // FIXME! Handling of Lo Gains is missing!
      }

    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}

//
// Set default values for the number of slices and weights:
//

void MCerPhotCalc2::SetDefaultWeights()
{
  const Float_t dummy[15] = {0, 0.0809835, 0.289593, 0.366926, 0.211665, 0.0508328, 0., 0., 0., 0., 0., 0., 0., 0., 0.};

  fWeight.Set(15,dummy);
  return;
}




