/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MCerPhotEvt.h"

#include <math.h>
#include <fstream.h>

#include <TCanvas.h>

#include "MLog.h"

#include "MGeomCam.h"

ClassImp(MCerPhotEvt);

// --------------------------------------------------------------------------
//
// Creates a MCerPhotPix object for each pixel in the event
//
MCerPhotEvt::MCerPhotEvt(const char *name, const char *title) : fNumPixels(0)
{
    fName  = name  ? name  : "MCerPhotEvt";
    fTitle = title ? title : "(Number of Photon)-Event Information";

    fPixels = new TClonesArray("MCerPhotPix", 0);
}

// --------------------------------------------------------------------------
//
// This is not yet implemented like it should.
//
void MCerPhotEvt::Draw(Option_t* option) 
{
    //
    //   FIXME!!! Here the Draw function of the CamDisplay
    //   should be called to add the CamDisplay to the Pad.
    //   The drawing should be done in MCamDisplay::Paint
    //

    //    MGeomCam *geom = fType ? new MGeomCamMagic : new MGeomCamCT1;
    //    MCamDisplay *disp = new MCamDisplay(geom);
    //    delete geom;
    //    disp->DrawPhotNum(this);
}

// --------------------------------------------------------------------------
//
// reset counter and delete netries in list.
//
void MCerPhotEvt::Reset()
{
    fNumPixels = 0;
    fPixels->Delete();
}

// --------------------------------------------------------------------------
//
//  Dump the cerenkov photon event to *fLog
//
void MCerPhotEvt::Print(Option_t *) const
{
    const Int_t entries = fPixels->GetEntries();

    *fLog << GetDescriptor() << dec << endl;
    *fLog << " Number of Pixels: " << fNumPixels << "(" << entries << ")" << endl;

    for (Int_t i=0; i<entries; i++ )
        (*this)[i].Print();
}

// --------------------------------------------------------------------------
//
// Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelExisting(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t i=0; i<entries; i++)
    {
        if (id == (*this)[i].GetPixId())
            return kTRUE;
    }

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelUsed(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        if (id == pix.GetPixId() && pix.IsPixelUsed())
            return kTRUE;
    }

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelCore(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        if (id == pix.GetPixId() && pix.IsPixelCore())
            return kTRUE;
    } 

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
// get the minimum number of photons  of all valid pixels in the list
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetNumPhotonsMin(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t minval = (*this)[0].GetNumPhotons();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetNumPhotons();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval < minval)
            minval = testval;
    }

    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum number of photons of all valid pixels in the list
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetNumPhotonsMax(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetNumPhotons();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetNumPhotons();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

// --------------------------------------------------------------------------
//
// get the minimum ratio of photons/error
//
Float_t MCerPhotEvt::GetRatioMin() const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t minval = (*this)[0].GetNumPhotons()/(*this)[0].GetErrorPhot();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetNumPhotons()/pix.GetErrorPhot();
        if (testval < minval)
            minval = testval;
    }

    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum ratio of photons/error
//
Float_t MCerPhotEvt::GetRatioMax() const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t maxval = (*this)[0].GetNumPhotons()/(*this)[0].GetErrorPhot();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetNumPhotons()/pix.GetErrorPhot();
        if (testval > maxval)
            maxval = testval;
    }

    return maxval;
}

// --------------------------------------------------------------------------
//
// get the minimum of error
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetErrorPhotMin(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetErrorPhot();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetErrorPhot();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum ratio of photons/error
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetErrorPhotMax(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetErrorPhot();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];

        Float_t testval = pix.GetErrorPhot();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

// --------------------------------------------------------------------------
//
// Return a pointer to the pixel with the requested id. NULL if it doesn't
// exist.
//
MCerPhotPix *MCerPhotEvt::GetPixById(int id) const
{
    TIter Next(fPixels);
    MCerPhotPix *pix = NULL;

    while ((pix=(MCerPhotPix*)Next()))
        if (pix->GetPixId()==id)
            return pix;

    return NULL;
}
