/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  02/2003 <wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MSelStandard                                                           //
//                                                                         //
//  This is a task to evaluate the Standard Cuts                           //
//                                                                         //
//  to be called after the calculation of the image parameters             //
//               before the g/h separation                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MSelStandard.h"

#include "MParList.h"

#include "MHillas.h"
#include "MHillasSrc.h"
#include "MCerPhotEvt.h"
#include "MMcEvt.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MSelStandard);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MSelStandard::MSelStandard(const MHillas *parhil, const MHillasSrc *parhilsrc,
                           const char *name, const char *title)
{
    fName  = name  ? name  : "MSelStandard";
    fTitle = title ? title : "Task to evaluate the Standard Cuts";

    fHil    = parhil;
    fHilsrc = parhilsrc;
}

// --------------------------------------------------------------------------
//
// 
// 
// 
//
Bool_t MSelStandard::PreProcess(MParList *pList)
{
    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    fEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }


    fCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fCam)
    {
        *fLog << dbginf << "MGeomCam (Camera Geometry) missing in Parameter List... aborting." << endl;
        return kFALSE;
    }
    fMm2Deg = fCam->GetConvMm2Deg();

    *fLog << "fMm2Deg = " << fMm2Deg << endl;

    memset(fErrors, 0, sizeof(fErrors));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluate standard cuts
// 
// if cuts are fulfilled      : return 0
// if they are not fullfilled : skip the remaining tasks for this event
// 
//
Bool_t MSelStandard::Process()
{
    Int_t rc = 0;

    //Double_t fLength       = fHil->GetLength() * fMm2Deg;
    //Double_t fWidth        = fHil->GetWidth()  * fMm2Deg;
    Double_t fDist         = fHilsrc->GetDist()* fMm2Deg;
    //Double_t fDelta        = fHil->GetDelta()  * kRad2Deg;
    Double_t fSize         = fHil->GetSize();
    Int_t fNumUsedPixels   = fHil->GetNumUsedPixels();
    Int_t fNumCorePixels   = fHil->GetNumCorePixels();

    if (     fSize <= 60.0         ||  fDist< 0.4           ||  fDist > 1.1  
         ||  fNumUsedPixels >= 92  ||  fNumCorePixels < 4)
    {
      //*fLog << "MSelStandard::Process; fSize, fDist, fNumUsedPixels, fNumCorePixels = "
      //      << fSize << ",  " << fDist << ",  " << fNumUsedPixels << ",  "
      //      << fNumCorePixels << endl;
      rc = 1;
    }    


    fErrors[rc]++;

    return rc==0 ? kTRUE : kCONTINUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Standard selections.
//
Bool_t MSelStandard::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fErrors[1] << " (" << setw(3) << (int)(fErrors[1]*100/GetNumExecutions()) << "%) Evts skipped due to: Standard selections are not fullfilled" << endl;

    *fLog << " " << fErrors[0] << " (" << (int)(fErrors[0]*100/GetNumExecutions()) << "%) Evts survived Standard selections!" << endl;
    *fLog << endl;

    return kTRUE;
}
