/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo  12/2003 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalRead                                                        //
//                                                                         //
//  This task looks for the nformation about FADC pedestals in            //
//  MMcFadcHeader and translates it to the pedestal values in              //
//  MPedestalCam. It is intended for camera >= 0.7 files                   //
//                                                                         //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalRead.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

#include "MRawRunHeader.h"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalRead);

using namespace std;

MMcPedestalRead::MMcPedestalRead(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalRead";
    fTitle = title ? title : "Copy MC pedestals into MPedestal Container";

    //
    // This is not needed here using MReadMarsFile because for the
    // RunHeader tree the auto scheme is disabled by default
    //

    AddToBranchList("MMcFadcHeader.fPedesMean");
    AddToBranchList("MMcFadcHeader.fPedesSigmaHigh");

    // FIXME: may be we'll have to take into account the low gain 
    // pedestal sigma in the future.
}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcPedestalRead::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return run->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// Check runtype and search for MPedestalCam and MMcFadcHeader.
// If the runtype check fails the task is removed from the task list.
//
Int_t MMcPedestalRead::PreProcess(MParList *pList)
{
    if (!CheckRunType(pList))
    {
        *fLog << warn << dbginf << " MMcPedestalRead is for Monte Carlo files only... ";
        *fLog << "removing task from list." << endl;
        return kSKIP;
    }

    fPedCam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedCam)
        return kFALSE;

    fMcPed = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!fMcPed)
    {
        *fLog << warn << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Initialize the size of MPedestalCam to the number of pixels from
// MMcFadcHeader.
//
Bool_t MMcPedestalRead::ReInit(MParList *pList)
{
    if (!CheckRunType(pList))
        return kFALSE;

    const int num = fPedCam->GetSize();

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*fPedCam)[i];

        const Float_t pedest = fMcPed->GetPedestal(i);
        const Float_t sigma  = fMcPed->GetPedestalRmsHigh(i);

        pix.Set(pedest, sigma);
    }

    return kTRUE;
}

