/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug  12/2003 <mailto:markus@ifae.es>
!   Author(s): Thomas Bretz 12/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MExtractedSignalCam                                                     //
//                                                                         //
// Hold the Extracted Signal information for all pixels in the camera      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

ClassImp(MExtractedSignalCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. Creates a MExtractedSignalPix object for each pixel
//
MExtractedSignalCam::MExtractedSignalCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MExtractedSignalCam";
    fTitle = title ? title : "Storage container for all Extracted Signal Information in the camera";

    fArray = new TClonesArray("MExtractedSignalPix", 1);

}

// --------------------------------------------------------------------------
//
// Delete the array conatining the pixel pedest information
//
MExtractedSignalCam::~MExtractedSignalCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MExtractedSignalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Get the size of the MExtractedSignalCam
//
Int_t MExtractedSignalCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MExtractedSignalPix &MExtractedSignalCam::operator[](Int_t i)
{
    return *static_cast<MExtractedSignalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MExtractedSignalPix &MExtractedSignalCam::operator[](Int_t i) const
{
    return *static_cast<MExtractedSignalPix*>(fArray->UncheckedAt(i));
}


void MExtractedSignalCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

void MExtractedSignalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MExtractedSignalPix *pix;
    while ((pix=(MExtractedSignalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": " << endl;
	pix->PrintOut();
    }
}

Bool_t MExtractedSignalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
    switch (type)
    {
    case 0:
        val = (*this)[idx].GetExtractedSignalHiGain();
        break;
    case 1:
        val = (*this)[idx].GetExtractedSignalHiGainError();
        break;
    case 2:
        val = (*this)[idx].GetExtractedSignalLoGain();
        break;
    case 3:
        val = (*this)[idx].GetExtractedSignalLoGainError();
        break;
    default:
	return kFALSE;
    }
    return val>=0;
}

void MExtractedSignalCam::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MExtractedSignaCam::DrawPixelContent - not available." << endl;
}
