/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo, 12/2003 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMcCalibrationCalc
//
//  This task looks for the nformation about FADC pedestals in
//  MMcFadcHeader and translates it to the pedestal mean and rms,
//  and in the conversion factor between photons and ADC counts in
//  MCalibrationCam.
//
//  Input Containers:
//   MMcFadcHeader
//   [MRawRunHeader]
//
//  Output Containers:
//   MCalibrationCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcCalibrationCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationPix.h"
#include "MCalibrationCam.h"

#include "MRawRunHeader.h"
#include "MMcFadcHeader.hxx"

ClassImp(MMcCalibrationCalc);

using namespace std;

MMcCalibrationCalc::MMcCalibrationCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcCalibrationCalc";
    fTitle = title ? title : "Write MC pedestals and conversion factors into MCalibration Container";

}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcCalibrationCalc::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return  run->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// Make sure, that there is an MCalibrationCam Object in the Parameter List.
//
Int_t MMcCalibrationCalc::PreProcess(MParList *pList)
{
  //
  // If it is no MC file skip this function...
  //
  if (!CheckRunType(pList))
    return kTRUE;

  return pList->FindCreateObj(AddSerialNumber("MCalibrationCam")) ? kTRUE : kFALSE;
}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Search for MCalibrationCam and MMcFadcHeader.
//
Bool_t MMcCalibrationCalc::ReInit(MParList *pList)
{
    //
    // If it is no MC file skip this function...
    //
    if (!CheckRunType(pList))
        return kTRUE;

    //
    // Now check the existence of all necessary containers. This has
    // to be done only if this is a MC file.
    //

    fHeaderFadc = (MMcFadcHeader*)pList->FindObject(AddSerialNumber("MMcFadcHeader"));
    if (!fHeaderFadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    fCalCam = (MCalibrationCam*)pList->FindCreateObj(AddSerialNumber("MCalibrationCam"));

    if (!fCalCam)
    {
        *fLog << err << dbginf << "Could not create MCalibrationCam... aborting. " << endl;
        return kFALSE;
    }

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the MCalibrationCam object
//
Int_t MMcCalibrationCalc::Process()
{
    const int num = fCalCam->GetSize();

    for (int i=0; i<num; i++)
    {
        MCalibrationPix &pix = (*fCalCam)[i];

        const Float_t pedestmean = fHeaderFadc->GetPedestal(i);
        const Float_t pedestrms  = fHeaderFadc->GetPedestalRmsHigh(i); 
	// per slice

        pix.SetPedestal(pedestmean, pedestrms);
    }

    return kTRUE;
}
