/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo, 12/2003 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMcCalibrationCalc
//
//  This task looks for the nformation about FADC pedestals in
//  MMcFadcHeader and translates it to the pedestal mean and rms (in adc counts),
//  and in the conversion factor between photons and ADC counts in
//  MCalibrationCam.
//  Then we create and fill also the MPedPhotCam object containing the pedestal
//  rms in units of photons.
//
//  Input Containers:
//   MMcFadcHeader
//   [MRawRunHeader]
//
//  Output Containers:
//   MCalibrationCam
//   MPedPhotCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcCalibrationCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationPix.h"
#include "MCalibrationCam.h"
#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"
#include "MGeomCam.h"
#include "MPedPhotCam.h"
#include "MPedPhotPix.h"

#include "MRawRunHeader.h"
#include "MMcFadcHeader.hxx"

ClassImp(MMcCalibrationCalc);

using namespace std;

MMcCalibrationCalc::MMcCalibrationCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcCalibrationCalc";
    fTitle = title ? title : "Write MC pedestals and conversion factors into MCalibration Container";

    fADC2PhInner = 1.;
    fADC2PhOuter = 1.;

}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcCalibrationCalc::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return  run->GetRunType() == kRTMonteCarlo;
}


//---------------------------------------------------------------------------
//
// Set ADC to photon conversion factors.
//
void MMcCalibrationCalc::SetADC2PhInner(Float_t x)
{
  fADC2PhInner = x;

  return;
}

// --------------------------------------------------------------------------
//
// Make sure, that there is an MCalibrationCam Object in the Parameter List.
//
Int_t MMcCalibrationCalc::PreProcess(MParList *pList)
{
  //
  // If it is no MC file skip this function...
  //
  if (!CheckRunType(pList))
    return kTRUE;

  if ( ! pList->FindCreateObj(AddSerialNumber("MCalibrationCam")))
    {
      *fLog << err << dbginf << "Cannot create MCalibrationCam... aborting." << endl;
      return kFALSE;
    }

  fPedPhotCam = (MPedPhotCam*) pList->FindCreateObj(AddSerialNumber("MPedPhotCam"));
  if ( ! fPedPhotCam)
    {
      *fLog << err << dbginf << "Cannot create MPedPhotCam... aborting." << endl;
      return kFALSE;
    }

  fSignalCam = (MExtractedSignalCam*) pList->FindObject(AddSerialNumber("MExtractedSignalCam"));
  if ( ! fSignalCam)
    {
      *fLog << err << dbginf << "Cannot find MExtractedSignalCam... aborting." << endl;
      return kFALSE;
    }

    fCalCam = (MCalibrationCam*)pList->FindObject(AddSerialNumber("MCalibrationCam"));

    if (!fCalCam)
    {
        *fLog << err << dbginf << "Could not create MCalibrationCam... aborting. " << endl;
        return kFALSE;
    }

  return kTRUE;

}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Search for MCalibrationCam, MPedPhotCam and MMcFadcHeader.
//
Bool_t MMcCalibrationCalc::ReInit(MParList *pList)
{
    //
    // If it is no MC file skip this function...
    //
    if (!CheckRunType(pList))
        return kTRUE;

    //
    // Now check the existence of all necessary containers.
    //

    fGeom = (MGeomCam*) pList->FindObject(AddSerialNumber("MGeomCam"));
    if ( ! fGeom )
      {
	*fLog << err << dbginf << "Cannot find MGeomCam... aborting." << endl;
	return kFALSE;
      }

    fHeaderFadc = (MMcFadcHeader*)pList->FindObject(AddSerialNumber("MMcFadcHeader"));
    if (!fHeaderFadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    //
    // Set the ADC to photons conversion factor for outer pixels:
    //
    fADC2PhOuter = fADC2PhInner *
      (fHeaderFadc->GetAmplitud() / fHeaderFadc->GetAmplitudOuter());

    //
    // Set the conversion factor between high and low gain:
    //
    fConversionHiLo = fHeaderFadc->GetLow2HighGain();

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the MCalibrationCam and MCerPhotPed objects
//
Int_t MMcCalibrationCalc::Process()
{
    const int num = fCalCam->GetSize();

    for (int i=0; i<num; i++)
    {
        MCalibrationPix &calpix = (*fCalCam)[i];
	MExtractedSignalPix &sigpix = (*fSignalCam)[i];

	//
	// ped mean and rms per pixel, in ADC counts, according to signal 
	// calculation (hi or low gain and number of integrated slices):
	//
        const Float_t pedestmean = sigpix.IsLoGainUsed()? 
	  fSignalCam->GetNumUsedLoGainFADCSlices()*fHeaderFadc->GetPedestal(i) : 
	  fSignalCam->GetNumUsedHiGainFADCSlices()*fHeaderFadc->GetPedestal(i);

	//
	// In some cases, depending on the camera simulation parameters, one can have
	// very little or no noise in the FADC. In the case the rms of pedestal is zero, 
	// the pixel will be cleaned out later in the image cleaning. To avoid this problem,
	// we set a default value of 0.01 ADC counts for the RMS per slice:
	//


        const Float_t pedestrms  = sigpix.IsLoGainUsed()? 
	  sqrt((Double_t)(fSignalCam->GetNumUsedLoGainFADCSlices())) * 
	  (fHeaderFadc->GetPedestalRmsLow(i)>0.? fHeaderFadc->GetPedestalRmsLow(i): 0.01)
	  : 
	  sqrt((Double_t)(fSignalCam->GetNumUsedHiGainFADCSlices())) * 
	  (fHeaderFadc->GetPedestalRmsHigh(i)>0.? fHeaderFadc->GetPedestalRmsHigh(i) : 0.01);

        calpix.SetPedestal(pedestmean, pedestrms);
	calpix.SetBlindPixelMethodValid();
	calpix.SetFitValid();

	calpix.SetConversionHiLo(fConversionHiLo);
	calpix.SetConversionHiLoError(0.);         // FIXME ?

	//
	// Write conversion factor ADC to photons (different for inner
	// and outer pixels).
	//

	Float_t adc2phot = (fGeom->GetPixRatio(i) < fGeom->GetPixRatio(0))?
	  fADC2PhOuter : fADC2PhInner;

	calpix.SetConversionBlindPixelMethod(adc2phot, 0., 0.);

	//
	// Write mean pedestal and pedestal rms per pixel
	// in number of photons:
	//
	MPedPhotPix &pedpix = (*fPedPhotCam)[i];

	if (sigpix.IsLoGainUsed())
	  pedpix.Set(adc2phot*fConversionHiLo*pedestmean, 
		     adc2phot*fConversionHiLo*pedestrms);
	else
	  pedpix.Set(adc2phot*pedestmean, adc2phot*pedestrms);

    }

    return kTRUE;
}
