/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!              Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MHExtractedSignalPix.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include "MGeomCam.h"

ClassImp(MPedestalCam);

using namespace std;

const UInt_t MPedestalCam::gkBlindPixelId = 559;
// --------------------------------------------------------------------------
//
// Default constructor. Creates a MPedestalPix object for each pixel
//
MPedestalCam::MPedestalCam(const char *name, const char *title) 
    : fTotalEntries(0), fExtractSlices(0.)
{
    fName  = name  ? name  : "MPedestalCam";
    fTitle = title ? title : "Storage container for all Pedestal Information in the camera";

    fArray = new TClonesArray("MPedestalPix", 1);
    
    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fHArray = new TObjArray;
    fHArray->SetOwner();

}

// --------------------------------------------------------------------------
//
// Delete the array conatining the pixel pedest information
//
MPedestalCam::~MPedestalCam()
{
  delete fArray;
  delete fHArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// This function returns the current size of the TClonesArray 
// independently if the MPedestalPix is filled with values or not.
//
// Get the size of the MPedestalCam
//
Int_t MPedestalCam::GetSize() const
{
  return fArray->GetEntriesFast();
}

Int_t MPedestalCam::GetHistSize() const 
{
  return fHArray->GetEntriesFast();
}


// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i)
{
  return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
  return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHPedestalPixel &MPedestalCam::operator()(UInt_t i)
{
  return *static_cast<MHPedestalPixel*>(fHArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHPedestalPixel &MPedestalCam::operator()(UInt_t i) const
{
  return *static_cast<MHPedestalPixel*>(fHArray->UncheckedAt(i));
}


// -------------------------------------------------------------------------
//
void MPedestalCam::Clear(Option_t *o)
{

    fArray->ForEach(TObject, Clear)();

    fTotalEntries     = 0;
    fExtractSlices    = 0.;
}


// --------------------------------------------------------------------------
//
// Our own clone function is necessary since root 3.01/06 or Mars 0.4
// I don't know the reason
//
TObject *MPedestalCam::Clone(const char *) const
{

  const Int_t n1 = fArray->GetSize();
  const Int_t n2 = fHArray->GetSize();
  
  //
  // FIXME, this might be done faster and more elegant, by direct copy.
  //
  MPedestalCam *cam = new MPedestalCam;
  
  cam->fArray->Expand(n1);
  cam->fHArray->Expand(n2);
  
  for (int i=0; i<n1; i++)
    {
      delete (*cam->fArray)[i];
      (*cam->fArray)[i] = (*fArray)[i]->Clone();
    }

  for (int i=0; i<n2; i++)
    {
      delete (*cam->fHArray)[i];
      (*cam->fHArray)[i] = (*fHArray)[i]->Clone();
    }
  return cam;
}



// --------------------------------------------------------------------------
//
// To setup the object we get the number of pixels from a MGeomCam object
// in the Parameter list. 
// MPedestalPix sets its parameters to 0. (other than default which is -1.)
//
Bool_t MPedestalCam::SetupFill(const MParList *pList)
{

  fHArray->Delete();

  return kTRUE;

}

// --------------------------------------------------------------------------
//
Bool_t MPedestalCam::Fill(const MParContainer *par, const Stat_t w)
{

  MExtractedSignalCam *signal = (MExtractedSignalCam*)par;
  if (!signal)
    {
      gLog << err << "No argument in MExtractedSignalCam::Fill... abort." << endl;
      return kFALSE;
    }
  

  Float_t slices = (Float_t)signal->GetNumUsedFADCSlices();

  if (slices == 0.)
    {
      gLog << err << "Number of used signal slices in MExtractedSignalCam is zero  ... abort." 
           << endl;
      return kFALSE;
    }

  if (fExtractSlices != 0. && slices != fExtractSlices )
    {
      gLog << err << "Number of used signal slices changed in MExtractedSignalCam  ... abort." 
           << endl;
      return kFALSE;
    }

  fExtractSlices = slices;

  const Int_t n = signal->GetSize();
  
  if (fHArray->GetEntries()==0)
    {
      fHArray->Expand(n);
      
      for (Int_t i=0; i<n; i++)
        {
          (*fHArray)[i] = new MHPedestalPixel;
          MHPedestalPixel &hist = (*this)(i);
          hist.ChangeHistId(i);
          MPedestalPix &pix = (*this)[i];
          pix.InitUseHists();
        }
    }
  
  if (fHArray->GetEntries() != n)
    {
      gLog << err << "ERROR - Size mismatch... abort." << endl;
      return kFALSE;
    }
  
  for (Int_t i=0; i<n; i++)
    {

      const MExtractedSignalPix &pix = (*signal)[i];
      
      const Float_t sig = pix.GetExtractedSignalHiGain();

      MHPedestalPixel &hist = (*this)(i);
      //
      // Don't fill signal per slice, we get completely screwed up 
      // with the sigma. Better fill like it is and renorm later
      //
      //      const Float_t sigPerSlice = sig/fExtractSlices;
      //      hist.FillCharge(sigPerSlice);
      //      hist.FillChargevsN(sigPerSlice);
      const Float_t signal = sig;
      hist.FillCharge(signal);
      hist.FillChargevsN(signal);
    }
  
  return kTRUE;
}

Bool_t MPedestalCam::Finalize()
{
    for (Int_t i=0; i<fHArray->GetSize(); i++)
    {

        MHPedestalPixel &hist = (*this)(i);

        //
        // 1) Return if the charge distribution is already succesfully fitted
        //    or if the histogram is empty
        //
        if (hist.IsFitOK() || hist.IsEmpty())
            continue;

        hist.CutAllEdges();
      
        //
        // 2) Fit the Hi Gain histograms with a Gaussian
        //
        hist.FitCharge();
        hist.Renorm(fExtractSlices);

    }
    return kTRUE;
}


void MPedestalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetPedestal() << " " << pix->GetPedestalRms() << endl;
    }
}

Float_t MPedestalCam::GetPedestalMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetPedestalMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

Bool_t MPedestalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (GetSize() <= idx)
    return kFALSE;

  if (!(*this)[idx].IsValid())
    return kFALSE;

  if ((UInt_t)idx == gkBlindPixelId)
    return kFALSE;

  const Float_t ped      = (*this)[idx].GetPedestal();
  const Float_t rms      = (*this)[idx].GetPedestalRms();

  const Float_t pederr   = rms/TMath::Sqrt((Float_t)fTotalEntries);
  const Float_t rmserr   = rms/TMath::Sqrt((Float_t)fTotalEntries*2.);

  Float_t mean     = 0.;
  Float_t meanerr  = 0.;
  Float_t sigma    = 0.;
  Float_t sigmaerr = 0.;
  Float_t prob     = 0.;
  
  if (type > 3) 
    if (GetHistSize() != 0)
      {
        if (!(*this)(idx).IsFitOK())
          return kFALSE;

        const MHPedestalPixel &hist = (*this)(idx);
        mean     = hist.GetChargeMean();
        meanerr  = hist.GetChargeMeanErr();
        sigma    = hist.GetChargeSigma() ;
        sigmaerr = hist.GetChargeSigmaErr();
        prob     = hist.GetChargeProb();
      }
    else
      return kFALSE;

  switch (type)
    {
    case 0:
      val = ped;
      break;
    case 1:
      val = pederr;
      break;
    case 2:
      val = rms;
      break;
    case 3:
      val = rmserr;
      break;
    case 4:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = mean;
      break;
    case 5:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = meanerr;
      break;
    case 6:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = sigma;
      break;
    case 7:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = sigmaerr;
      break;
    case 8:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = prob;
      break;
    case 9:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = 2.*(ped-mean)/(ped+mean);
      break;
    case 10:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = TMath::Sqrt((pederr*pederr + meanerr*meanerr) * (ped*ped + mean*mean))
            *2./(ped+mean)/(ped+mean);
      break;
    case 11:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = 2.*(pederr - meanerr)/(pederr + meanerr);
      break;
    case 12:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = 2.*(sigma-rms)/(sigma+rms);
      break;
    case 13:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = TMath::Sqrt((rmserr*rmserr + sigmaerr*sigmaerr) * (rms*rms + sigma*sigma))
            *2./(rms+sigma)/(rms+sigma);
      break;
    case 14:
      if (!(*this)(idx).IsFitOK())
        return kFALSE;
      val = 2.*(sigmaerr - rmserr)/(sigmaerr + rmserr);
      break;
    default:
      return kFALSE;
    }
  return kTRUE;
}

void MPedestalCam::DrawPixelContent(Int_t idx) const
{
  (*this)(idx).Draw();
}
