/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo 7/2002  <mailto:moralejo@pd.infn.it>
!   Author(s): Thomas Bretz 2002  <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2002-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MPedestalSum
//
//   This is a task which calculates the number of photons from the FADC
//   time slices. It weights the each slice according to the numbers in
//   the array fWeight (default: all slices added up with weight 1).
//
//   The weights are rescaled, such that sum(weigths)=num slices
//
//  Input Containers:
//   MRawRunHeader, MRawEvtData, MPedestalCam
//
//  Output Containers:
//   MCerPhotEvt
//
//////////////////////////////////////////////////////////////////////////////
#include "MPedestalSum.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcRunHeader.hxx"

#include "MRawRunHeader.h"
#include "MRawEvtData.h"       // MRawEvtData::GetNumPixels
#include "MCerPhotEvt.h"
#include "MPedestalPix.h"
#include "MPedestalCam.h"
#include "MRawEvtPixelIter.h"

ClassImp(MPedestalSum);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MPedestalSum::MPedestalSum(const char *name, const char *title)
{
    fName  = name  ? name  : "MPedestalSum";
    fTitle = title ? title : "Calculate pixel signal from FADC data";

    AddToBranchList("MRawEvtData.fHiGainPixId");
    AddToBranchList("MRawEvtData.fLoGainPixId");
    AddToBranchList("MRawEvtData.fHiGainFadcSamples");
    AddToBranchList("MRawEvtData.fLoGainFadcSamples");

    SetDefaultWeights();
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawRunHeader
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Int_t MPedestalSum::PreProcess(MParList *pList)
{
    fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!fRunHeader)
    {
        *fLog << dbginf << "MRawRunHeader not found... aborting." << endl;
        return kFALSE;
    }
    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the run type and camera version.
// If the file is a MC file and the used camera version is <= 40
// we enable a fix for truncated pedestal means in this version.
//
Bool_t MPedestalSum::ReInit(MParList *pList)
{
    /*
    const MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    if (fRunHeader->GetNumSamplesHiGain() != fWeight.GetSize())
    {
        *fLog << dbginf << "Number of FADC slices (" << fRunHeader->GetNumSamplesHiGain() <<") is different from assumed one (" << fWeight.GetSize() << ")... aborting." << endl;
        return kFALSE;
    }

    Bool_t fIsMcFile = runheader->GetRunType() == kRTMonteCarlo;
    if (!fIsMcFile)
        return kTRUE;

    ScalePedestals();

    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check for camera version, MC run header not found." << endl;
        return kTRUE;
    }

    fEnableFix = kFALSE;
    if (mcrunheader->GetCamVersion() <= 40)
        fEnableFix = kTRUE;
     */
    return kTRUE;
}

void MPedestalSum::ScalePedestals()
{
    Int_t n = 577; //FIXME: fPedestals->GetNumPixel();

    for (int idx=0; idx<n; idx++)
    {
        MPedestalPix &ped = (*fPedestals)[idx];

        const Double_t offset = fEnableFix ? ped.GetPedestal()-0.5 : ped.GetPedestal();
        ped.Set(offset*fSumWeights, ped.GetPedestalRms()*sqrt(fSumQuadWeights));
    }

    fPedestals->SetReadyToSave();
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MCerPhotEvt container.
//
Int_t MPedestalSum::Process()
{
    //fCerPhotEvt->InitSize(fRawEvt->GetNumPixels());

//    if (fIsMcFile)
//        ScalePedestals();

    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
        const UInt_t idx = pixel.GetPixelId();

        const int n = fRunHeader->GetNumSamplesHiGain();

        //
        // Calculate pixel signal unless it has all FADC slices empty:
        //
        Byte_t *ptr = pixel.GetHiGainSamples();

        Float_t nphot = 0;
        Float_t nerr  = 0;
        for(Int_t i=0; i<n; i++)
        {
            nphot += ptr[i];
            nerr  += ptr[i]*ptr[i];
        }

        nphot /= n;

        fCerPhotEvt->AddPixel(idx, nphot, sqrt(nerr/n-nphot*nphot));

        // FIXME! Handling of Lo Gains is missing!
    }

    fCerPhotEvt->FixSize();
    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Set default values for the number of slices and weights:
//
void MPedestalSum::SetDefaultWeights()
{
    const Float_t dummy[15] = { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };
    fWeight.Set(15, dummy);
}
