/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug,  3/2004 <mailto:markus@ifae.es>
! 
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//
// MBadPixelsPix
//
// The bits of an integer array fInfo are used to declare and inform about 
// possible defects in a pixel. Default and absence of defects create an array
// of zeros. 
//
// The first index (fInfo[0]) holds general information which is coded as follows:
// * BIT(1): Unsuitable Run: The pixel is not suited for analysis for the entire run
// * BIT(2): Unsuitable Evt: The pixel is not suited for analysis for the current event
// * BIT(3): Unreliable Run: The pixel can in principle be used for analysis, although 
//                           previous analysis steps have yielded certain inconsistencies
//
// These bits can be called with the enum MBadPixelsPix::UnsuitableTupe_t in combination 
// with the function IsUnsuitable(MBadPixelsPix::UnsuitableTupe_t), e.g. 
// MBadPixelsPix::IsUnsuitalbe(MBadPixelsPix::kUnsuitableRun) asks if the first bit is set. 
//
// The second index (fInfo[1]) hold information acquired during the calibration. The bits 
// are coded in the following form:
// BIT(1 ): kHiGainNotCalibrated :  Any High Gain signal is not calibrated and cannot be used
// BIT(2 ): kLoGainNotCalibrated :  Any Low  Gain signal is not calibrated and cannot be used
// BIT(3 ): kHiGainNotFitted     :  Any High Gain signal is calibrated without a Gauss Fit to the signal distribution
// BIT(4 ): kLoGainNotFitted     :  Any Low  Gain signal is calibrated without a Gauss Fit to the signal distribution
// BIT(5 ): kHiGainOscillating   :  The High Gain signals fourier transform showed abnormal behavior  
// BIT(6 ): kLoGainOscillating   :  The Low  Gain signals fourier transform showed abnormal behavior  
// BIT(7 ): kLoGainSaturation    :  The Low  Gain signals were saturated during calibration
// BIT(8 ): kChargeIsPedestal    :  The calibration signal contained only pedestals - presumably dead pixel
// BIT(10): kChargeRelErrNotValid:  The relative error of the derived charge was too large or too small
// BIT(11): kChargeSigmaNotValid :  The sigma of the pedestal distribution smaller than the pedestal RMS - presumably a pixel with a star in its FOV only during the pedestal taking 
// BIT(12): kMeanTimeInFirstBin  :  The signal has its mean maximum in the first used FADC slice - signal extractor bad
// BIT(13): kMeanTimeInLast2Bins :  The signal has its mean maximum in the last two used FADC slice - signal extractor bad
// BIT(14): kDeviatingNumPhes    :  The calculated number of photo-electrons deviates too much from the mean - inconsistency
// BIT(15): kDeviatingFFactor    :  The calculated overall F-Factor deviates too much from the mean - inconsistency
// BIT(16): kConversionHiLoNotValid: The calibrated Conversion between Hi-Gain and Low Gain gives absurd results
//
// These bits can be called with the enum MBadPixelsPix::UncalibratedType_t in combination 
// with the function IsUncalibrated(MBadPixelsPix::UncalibratedTupe_t), e.g. 
// MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kHiGainNotCalibrated) asks if the Hi Gain signal
// could be calibrated.
//
// Two additional functions yield specific calibration information:
// * IsCalibrationSignalOK() asks if the extracted calibration signal showed any inconsistency
// * IsCalibrationResultOK() asks if the applied calibration can be used at all.
//
/////////////////////////////////////////////////////////////////////////////
#include "MBadPixelsPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MBadPixelsPix);

using namespace std;

const Int_t MBadPixelsPix::fgRunMask =
    MBadPixelsPix::kUnsuitableRun |
    MBadPixelsPix::kUnreliableRun;

// ------------------------------------------------------------------------
//
// Initialize Pixel to be Ok.
//
MBadPixelsPix::MBadPixelsPix(const char* name, const char* title)
    : fInfo(5)
{
    fName  = name  ? name  : "MBadPixelsPix";
    fTitle = title ? title : "Container storing bad pixel information for a single pixel";

    fInfo[1] = 0;
}

// ------------------------------------------------------------------------
//
// Invalidate all bits which are not run-wise. This will be called for
// all entries in the parameter list, just before each time the task-list
// is executed.
//
void MBadPixelsPix::Reset()
{
    fInfo[0] &= fgRunMask;
}

// ------------------------------------------------------------------------
//
// Invalidate values (set=0 which mean Pixel OK)
//
void MBadPixelsPix::Clear(Option_t *o)
{
    fInfo.Reset(0);
}

// ------------------------------------------------------------------------
//
// Merge (bitwise or) the information in pix into this pixel.
//
void MBadPixelsPix::Merge(const MBadPixelsPix &pix)
{
    const Int_t n = pix.fInfo.GetSize();
    if (n>fInfo.GetSize())
        fInfo.Set(n);

    for (int i=0; i<n; i++)
        fInfo[i] |= pix.fInfo[i];
}


/****************************************************************************
           This is a collection of possible defects for later use
 ****************************************************************************/

/*
 1  PMT defective.
 2  Preamplifier defective.
 3  Optical link defective.
 4  HV cannot be set.
 7  HV unstable.
 5  HV readout defective.
 8  DC unstable.
 6  DC readout defective.
 9  Discriminator threshold cannot be set.
 10  Trigger delay cannot be set.
 11  Discriminator gives no output.
 <-? 12  Pixel out of L1T.
 13  FADC defective.
 14  FADC board digital information defective.
 */

/*
 1  Pixel shows no signal
 */

/*
 MCalibrationCalc - valid for the result of a calibration run:

      3  Hi-Gain saturated, no LoGain available

      4  Conversion Factor HiGain - LoGain not valid

      5  Cannot be calibrated at all
      6  Cannot be fitted - calibrated using Histogram Mean and RMS

     */

/*

Hardware defects which cannot be detected automatically by software. This might be stored at least in the data-base. I think we should wait until we implement these things...
Preamplifier defective.
Optical link defective.
HV cannot be set.
HV readout defective.
DC readout defective.
Discriminator threshold cannot be set.
Trigger delay cannot be set.
Discriminator gives no output.
FADC defective.
FADC board digital information defective.
Pixel out of L1T. (this is an important information, but not necessarily a defect, is it?)

In addition here are some cases which I think can be detected by software:
- no signal
- wrong signal
- hv problem
- dc problem
- Conversion Factor HiGain - LoGain not valid (what does this mean?)
- No calibration possible
- No fit possible - calibrated using Histogram Mean and RMS
- Mean Charge smaller than PedRMS
- Sigma Charge smaller than PedRMS
- Calib.methods inconsistency (there are in pricipal 6 combinations... do we need 6 bits?)
- Gains oscillate (what does it mean?)
- Sigma  Arrival Time  bigger than  FADC window (from calib)
- Mean   Arrival Time  at edge of   FADC window (from calib)
*/
