/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationBlindPix                                                    //
//                                                                         //
// This is the storage container to hold informations about the calibration//
// blind pixel                                                             //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationBlindPix.h"
#include "MHCalibrationBlindPixel.h"

#include <TH1.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationBlindPix);

using namespace std;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MCalibrationBlindPix::MCalibrationBlindPix(const char *name, const char *title)
{

  fName  = name  ? name  : "MCalibrationBlindPix";
  fTitle = title ? title : "Container of the MHCalibrationBlindPixel and the fit results";

  fHist = new MHCalibrationBlindPixel();
  
  if (!fHist)
    *fLog << warn << dbginf << " Could not create MHCalibrationBlindPixel " << endl;

  Clear();
}

MCalibrationBlindPix::~MCalibrationBlindPix() 
{
  delete fHist;
}

// ------------------------------------------------------------------------
//
// Invalidate values
//
void MCalibrationBlindPix::Clear(Option_t *o)
{
  fHist->Reset();

  fLambda = -1.;
  fMu0    = -1.;
  fMu1    = -1.;
  fSigma0 = -1.;
  fSigma1 = -1.;
  fErrLambda = -1.;
  fErrMu0    = -1.;
  fErrMu1    = -1.;
  fErrSigma0 = -1.;
  fErrSigma1 = -1.;
  fTime      = -1.;
  fErrTime   = -1;

}

Bool_t MCalibrationBlindPix::IsFitOK() const 
{
  return fHist->IsFitOK();
}

Bool_t MCalibrationBlindPix::FillCharge(const Int_t q)
{
  return fHist->FillBlindPixelCharge(q);
}

Bool_t MCalibrationBlindPix::FillTime(const Float_t t)
{
  return fHist->FillBlindPixelTime(t);
}

Bool_t MCalibrationBlindPix::FitCharge() 
{

  if (!fHist->FitSinglePhe())
    return kFALSE;
  
  fLambda = TMath::IsNaN(fHist->GetLambda()) ? -1. : fHist->GetLambda();
  fMu0    = TMath::IsNaN(fHist->GetMu0())    ? -1. : fHist->GetMu0();
  fMu1    = TMath::IsNaN(fHist->GetMu1())    ? -1. : fHist->GetMu1();
  fSigma0 = TMath::IsNaN(fHist->GetSigma0()) ? -1. : fHist->GetSigma0();
  fSigma1 = TMath::IsNaN(fHist->GetSigma1()) ? -1. : fHist->GetSigma1();

  fErrLambda = TMath::IsNaN(fHist->GetLambdaErr()) ? -1. : fHist->GetLambdaErr();
  fErrMu0    = TMath::IsNaN(fHist->GetMu0Err())    ? -1. : fHist->GetMu0Err();
  fErrMu1    = TMath::IsNaN(fHist->GetMu1Err())    ? -1. : fHist->GetMu1Err();
  fErrSigma0 = TMath::IsNaN(fHist->GetSigma0Err()) ? -1. : fHist->GetSigma0Err();
  fErrSigma1 = TMath::IsNaN(fHist->GetSigma1Err()) ? -1. : fHist->GetSigma1Err();

  return kTRUE;
}



Bool_t MCalibrationBlindPix::FitTime() 
{

  if(!fHist->FitTime())
    return kFALSE;

  fTime    = fHist->GetMeanTime();
  fErrTime = fHist->GetMeanTimeErr();
  
  return kTRUE;

}

Bool_t MCalibrationBlindPix::CheckOscillations()
{
  return fHist->CheckOscillations();
}
